A `'static` requirement in a return type involving a trait is not fulfilled.

Erroneous code examples:

```compile_fail,E0759
use std::fmt::Debug;

fn foo(x: &i32) -> impl Debug {
    x
}
```

```compile_fail,E0759
# use std::fmt::Debug;
fn bar(x: &i32) -> Box<dyn Debug> {
    Box::new(x)
}
```

These examples have the same semantics as the following:

```compile_fail,E0759
# use std::fmt::Debug;
fn foo(x: &i32) -> impl Debug + 'static {
    x
}
```

```compile_fail,E0759
# use std::fmt::Debug;
fn bar(x: &i32) -> Box<dyn Debug + 'static> {
    Box::new(x)
}
```

Both [`dyn Trait`] and [`impl Trait`] in return types have a an implicit
`'static` requirement, meaning that the value implementing them that is being
returned has to be either a `'static` borrow or an owned value.

In order to change the requirement from `'static` to be a lifetime derived from
its arguments, you can add an explicit bound, either to an anonymous lifetime
`'_` or some appropriate named lifetime.

```
# use std::fmt::Debug;
fn foo(x: &i32) -> impl Debug + '_ {
    x
}
fn bar(x: &i32) -> Box<dyn Debug + '_> {
    Box::new(x)
}
```

These are equivalent to the following explicit lifetime annotations:

```
# use std::fmt::Debug;
fn foo<'a>(x: &'a i32) -> impl Debug + 'a {
    x
}
fn bar<'a>(x: &'a i32) -> Box<dyn Debug + 'a> {
    Box::new(x)
}
```

[`dyn Trait`]: https://doc.rust-lang.org/book/ch17-02-trait-objects.html#using-trait-objects-that-allow-for-values-of-different-types
[`impl Trait`]: https://doc.rust-lang.org/book/ch10-02-traits.html#returning-types-that-implement-traits
