/* This file is part of the KDE project

   Copyright (C) 2001 Lukas Tinkl <lukas@kde.org>
                      Andreas Schlapbach <schlpbch@iam.unibe.ch>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qlabel.h>
#include <qvbox.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qcombobox.h>
#include <qwhatsthis.h>

#include <klocale.h>
#include <kurl.h>
#include <kfontdialog.h>
#include <kiconloader.h>
#include <kdebug.h>
#include <klineedit.h>
#include <knuminput.h>

#include "imgallerydialog.h"
#include "imgallerydialog.moc"

KIGPDialog::KIGPDialog(QWidget *parent, const QString& path, const char *name )
  : KDialogBase( IconList, i18n("Configure"), Ok|Cancel,
		 Ok, parent, name, true, true ),
    m_dialogOk( false )
{
  setCaption(i18n("Create Image Gallery"));
  setupLookPage(path);
  setupDirectoryPage(path);
}

void KIGPDialog::setupLookPage(const QString& path) {
  QFrame *page = addPage( i18n("Look"), i18n("Page Look"),
			  BarIcon("colorize", KIcon::SizeMedium ) );

  QVBoxLayout *vlay = new QVBoxLayout( page, 0, spacingHint() );

  QLabel *label;

  label = new QLabel( i18n("&Page title:"), page);
  vlay->addWidget(label);

  m_title = new QLineEdit(i18n("Image Gallery for %1").arg(path), page);
  vlay->addWidget( m_title );
  label->setBuddy(m_title); 

  m_imagesPerRow = new KIntNumInput(4, page);
  m_imagesPerRow->setRange(1, 8, 1, true );
  m_imagesPerRow->setLabel( i18n("I&mages per row:") );
  vlay->addWidget( m_imagesPerRow );

  QGridLayout *grid = new QGridLayout( 2, 2 );
  vlay->addLayout( grid );

  m_imageName = new QCheckBox( i18n("Show image file &name"), page);
  m_imageName->setChecked( true );
  grid->addWidget( m_imageName, 0, 0 );

  m_imageSize = new QCheckBox( i18n("Show image file &size"), page);
  m_imageSize->setChecked( false );
  grid->addWidget( m_imageSize, 0, 1 );

  m_imageProperty = new QCheckBox( i18n("Show image &dimensions"), page);
  m_imageProperty->setChecked( false );
  grid->addWidget( m_imageProperty, 1, 0 );

  m_generateXHTML = new QCheckBox(i18n("Generate &XHTML"), page);
  m_generateXHTML->setChecked(false);
  grid->addWidget( m_generateXHTML, 1, 1 );

  QHBoxLayout *hlay11  = new QHBoxLayout( );
  vlay->addLayout( hlay11 );

  m_fontName = new QComboBox( false,page );
  QStringList standardFonts;
  KFontChooser::getFontList(standardFonts, 0);
  m_fontName->insertStringList( standardFonts );

  label = new QLabel( i18n("Fon&t name:"), page );
  label->setBuddy( m_fontName );
  hlay11->addWidget( label );
  hlay11->addStretch( 1 );
  hlay11->addWidget( m_fontName );

  QHBoxLayout *hlay12  = new QHBoxLayout( );
  vlay->addLayout( hlay12 );

  m_fontSize = new QSpinBox( 6, 15, 1, page );
  m_fontSize->setValue( 14 );

  label = new QLabel( i18n("Font si&ze:"), page );
  label->setBuddy( m_fontSize );
  hlay12->addWidget( label );
  hlay12->addStretch( 1 );
  hlay12->addWidget( m_fontSize );

  QHBoxLayout *hlay1  = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay1 );

  m_foregroundColor = new KColorButton(page);
  m_foregroundColor->setColor(QColor("#d0ffd0"));

  label = new QLabel( i18n("&Foreground color:"), page);
  label->setBuddy( m_foregroundColor );
  hlay1->addWidget( label );
  hlay1->addStretch( 1 );
  hlay1->addWidget(m_foregroundColor);

  QHBoxLayout *hlay2 = new QHBoxLayout( spacingHint() );
  vlay->addLayout( hlay2 );

  m_backgroundColor = new KColorButton(page);
  m_backgroundColor->setColor(QColor("#333333"));

  label = new QLabel( i18n("&Background color:"), page);
  hlay2->addWidget( label );
  label->setBuddy( m_backgroundColor );
  hlay2->addStretch( 1 );
  hlay2->addWidget(m_backgroundColor);

  vlay->addStretch(1);
}

void KIGPDialog::setupDirectoryPage(const QString& path) {
  QFrame *page = addPage( i18n("Directories"), i18n("Directories"),
			  BarIcon("folder", KIcon::SizeMedium ) );

  QVBoxLayout *dvlay = new QVBoxLayout( page, spacingHint() );

  QLabel *label;
  label = new QLabel(i18n("&Save to:"), page);
  dvlay->addWidget( label );

  m_imageNameReq = new KURLRequester(path + "images.html", page);
  label->setBuddy( m_imageNameReq );
  dvlay->addWidget(m_imageNameReq);
  connect( m_imageNameReq, SIGNAL(textChanged(const QString&)),
      this, SLOT(imageUrlChanged(const QString&)) );

  m_recurseSubDir = new QCheckBox(i18n("&Recurse subdirectories"), page);
  m_recurseSubDir->setChecked(false);

  m_recursionLevel = new KIntNumInput( page );
  m_recursionLevel->setRange( 0, 99, 1, true );
  m_recursionLevel->setLabel( i18n("Rec&ursion depth:") );
  m_recursionLevel->setSpecialValueText( i18n("Endless"));
  m_recursionLevel->setEnabled(false);

  connect(m_recurseSubDir, SIGNAL( toggled(bool) ), 
          m_recursionLevel, SLOT( setEnabled(bool) ) );

  dvlay->addWidget(m_recurseSubDir);
  dvlay->addWidget(m_recursionLevel);

  m_copyOriginalFiles = new QCheckBox(i18n("Copy or&iginal files"), page);
  m_copyOriginalFiles->setChecked(false);
  dvlay->addWidget(m_copyOriginalFiles);

  m_useCommentFile = new QCheckBox(i18n("Use &comment file"), page);
  m_useCommentFile->setChecked(false);
  dvlay->addWidget(m_useCommentFile);

  label = new QLabel(i18n("Comments &file:"), page);
  label->setEnabled( false );
  dvlay->addWidget( label );

  m_commentFileReq = new KURLRequester(path + "comments", page);
  m_commentFileReq->setEnabled(false);
  label->setBuddy( m_commentFileReq );
  dvlay->addWidget(m_commentFileReq);

  connect(m_useCommentFile, SIGNAL(toggled(bool)), 
      label, SLOT(setEnabled(bool)));
  connect(m_useCommentFile, SIGNAL(toggled(bool)), 
      m_commentFileReq, SLOT(setEnabled(bool)));

  dvlay->addStretch(1);
}

KIGPDialog::~KIGPDialog()
{
}

void KIGPDialog::imageUrlChanged(const QString &url )
{
    enableButtonOK( !url.isEmpty());
}

bool  KIGPDialog::printImageName()  const
{
  return m_imageName->isChecked();
}

bool  KIGPDialog::printImageSize() const
{
  return m_imageSize->isChecked();
}

bool  KIGPDialog::printImageProperty() const
{
  return m_imageProperty->isChecked();
}

bool KIGPDialog::recurseSubDirectories() const
{
  return m_recurseSubDir->isChecked();
}

int KIGPDialog::recursionLevel() const
{
  return m_recursionLevel->value();
}

bool KIGPDialog::copyOriginalFiles() const
{
  return m_copyOriginalFiles->isChecked();
}

bool KIGPDialog::useCommentFile() const
{
  return m_useCommentFile->isChecked();
}

bool KIGPDialog::generateXHTML() const
{
  return m_generateXHTML->isChecked();
}

int KIGPDialog::getImagesPerRow() const
{
  return m_imagesPerRow->value();
}

const QString KIGPDialog::getTitle() const
{
  return m_title->text();
}

const QString KIGPDialog::getImageName() const
{
  return m_imageNameReq->url();
}

const QString KIGPDialog::getCommentFile() const
{
  return m_commentFileReq->url();
}

const QString KIGPDialog::getFontName() const
{
  return m_fontName->currentText();
}

const QString KIGPDialog::getFontSize() const
{
  return m_fontSize->text();
}

const QColor KIGPDialog::getBackgroundColor() const
{
  return m_backgroundColor->color();
}

const QColor KIGPDialog::getForegroundColor() const
{
  return m_foregroundColor->color();
}

