/*
 *   kghostview: KDE PostScript viewer
 *   This file: Copyright (C) 2000 Espen Sand, espen@kde.org
 *   Uses ideas from ealier dialog code by Mark Donohoe and Tim Theisen
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


// Add header files alphabetically

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qradiobutton.h>

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <kinstance.h>

#include "interpreterdialog.h"

#include "kgv_view.h"

InterpreterDialog::InterpreterDialog( QWidget *parent, const char *name, 
				      bool modal )
  :KDialogBase( parent, name, modal, i18n("Configure Interpreter"),
		Ok|Cancel|Default, Ok )
{
  QFrame *page = makeMainWidget();
  QVBoxLayout *topLayout = new QVBoxLayout( page, 0, spacingHint() );

  QGroupBox *groupBox = new QGroupBox( i18n("&General"), page );
  topLayout->addWidget( groupBox );
  mInterpreterEdit = new QLineEdit( groupBox );
  QLabel *label = new QLabel( i18n("Interpreter name"), groupBox );
  mAntiAliasCheck = new QCheckBox( i18n("Antialiasing"), groupBox );
  connect (mAntiAliasCheck, SIGNAL (toggled(bool)),
	   this, SLOT (slotAaToggled(bool)));
  mPlatformFontCheck = new QCheckBox( i18n("Platform fonts"), groupBox );
  mMessagesCheck = new QCheckBox( i18n("Messages"), groupBox );
  QGridLayout *glay = new QGridLayout( groupBox, 3, 3, spacingHint() );
  glay->addRowSpacing(0, groupBox->fontMetrics().lineSpacing() );
  glay->setColStretch( 2, 1 );
  glay->addWidget( label, 1, 0 );
  glay->addMultiCellWidget( mInterpreterEdit, 1, 1, 1, 2 );
  glay->addWidget( mAntiAliasCheck, 2, 0 );	
  glay->addWidget( mPlatformFontCheck, 2, 1 );
  glay->addWidget( mMessagesCheck, 2, 2, AlignLeft );

  QButtonGroup *buttonGroup = new QButtonGroup( i18n("&Palette"), page );
  buttonGroup->setExclusive( true );
  topLayout->addWidget( buttonGroup );
  mMonoRadio = new QRadioButton( i18n("Monochrome"), buttonGroup );
  connect (mMonoRadio, SIGNAL (toggled(bool)),
	   this, SLOT(slotPaletteChange(bool)));
  mGrayscaleRadio = new QRadioButton( i18n("Grayscale"), buttonGroup );
  mColorRadio = new QRadioButton( i18n("Color"), buttonGroup );
  glay = new QGridLayout( buttonGroup, 2, 3, spacingHint() );
  glay->setColStretch( 2, 1 );
  glay->addRowSpacing( 0, buttonGroup->fontMetrics().lineSpacing() );
  glay->addWidget( mMonoRadio, 1, 0 );
  glay->addWidget( mGrayscaleRadio, 1, 1 );
  glay->addWidget( mColorRadio, 1, 2, AlignLeft );

  buttonGroup = new QButtonGroup( i18n("&Backing"), page );
  buttonGroup->setExclusive( true );
  topLayout->addWidget( buttonGroup, 10, AlignTop );
  mPixmapRadio = new QRadioButton( i18n("Pixmap"), buttonGroup );
  mBackingStoreRadio = new QRadioButton( i18n("Backing store"), buttonGroup );
  glay = new QGridLayout( buttonGroup, 3, 2, spacingHint() );
  glay->setColStretch( 1, 1 );
  glay->setRowStretch( 2, 1 );
  glay->addRowSpacing( 0, buttonGroup->fontMetrics().lineSpacing() );
  glay->addWidget( mPixmapRadio, 1, 0 );
  glay->addWidget( mBackingStoreRadio, 1, 1, AlignLeft );

  readSettings();
}

void InterpreterDialog::slotAaToggled(bool b)
{
  if (b && mMonoRadio->isChecked())
    mGrayscaleRadio->setChecked(true);
}

void InterpreterDialog::slotPaletteChange(bool)
{
  if (mAntiAliasCheck->isChecked() &&
      mMonoRadio->isChecked())
    mAntiAliasCheck->setChecked(false);
}

void InterpreterDialog::setup()
{
  mInterpreterEdit->setText( mInterpreterPath );
  mAntiAliasCheck->setChecked( mAntialias ); 
  mPlatformFontCheck->setChecked( mPlatformFonts ); 
  mMessagesCheck->setChecked( mShowMessages ); 

  mMonoRadio->setChecked( mPaletteType == MONO_PALETTE );
  mGrayscaleRadio->setChecked( mPaletteType == GRAY_PALETTE ); 
  mColorRadio->setChecked( mPaletteType == COLOR_PALETTE );

  mPixmapRadio->setChecked( mBackingType == PIX_BACKING );
  mBackingStoreRadio->setChecked( mBackingType == STORE_BACKING );
}


QString InterpreterDialog::interpreterPath() const
{
  return mInterpreterPath;
}


bool InterpreterDialog::antiAlias() const
{
  return mAntialias;
}


bool InterpreterDialog::showMessages() const
{
  return mShowMessages;
}


bool InterpreterDialog::platformFonts() const
{
  return mPlatformFonts;
}


int InterpreterDialog::paletteType() const
{
  return mPaletteType;
}


int InterpreterDialog::backingStoreType() const
{
  return mBackingType;
}


void InterpreterDialog::slotOk()
{
  mInterpreterPath  = mInterpreterEdit->text();
  mAntialias	    = mAntiAliasCheck->isChecked();
  mShowMessages	    = mMessagesCheck->isChecked();
  mPlatformFonts    = mPlatformFontCheck->isChecked();
  if( mColorRadio->isChecked() )
    mPaletteType = COLOR_PALETTE;
  else if( mGrayscaleRadio->isChecked() )
    mPaletteType = GRAY_PALETTE;
  else
    mPaletteType = MONO_PALETTE;

  if( mPixmapRadio->isChecked() )
    mBackingType = PIX_BACKING;
  else
    mBackingType = STORE_BACKING;

  writeSettings();
  accept();
}


void InterpreterDialog::slotDefault()
{
  mInterpreterPath  = "gs";
  mAntialias	    = true;
  mShowMessages	    = true;
  mPlatformFonts    = false;
  mPaletteType	    = COLOR_PALETTE;
  mBackingType	    = PIX_BACKING;
  setup();
}



void InterpreterDialog::readSettings()
{
  KConfig &config = *KGVFactory::instance()->config();
  config.setGroup( "General" );
  
  mInterpreterPath = config.readEntry( "Interpreter", "gs" );
  mAntialias = config.readBoolEntry( "Antialiasing", true );
  mShowMessages = config.readBoolEntry( "Messages", false );
  mPlatformFonts = config.readBoolEntry( "Platform fonts", false );

  QString text = config.readEntry( "Palette" );
  if( text.find( "mono" ) == 0 )
    mPaletteType = MONO_PALETTE;
  else if( text.find( "gray" ) == 0 )
    mPaletteType = GRAY_PALETTE;
  else
    mPaletteType = COLOR_PALETTE;
  
  text = config.readEntry( "Backing" );
  if( text.find( "store" ) == 0 )
    mBackingType = STORE_BACKING;
  else
    mBackingType = PIX_BACKING;

  setup();
}

void InterpreterDialog::writeSettings()
{
  //Don't set the group in here.
  KConfig &config = *KGVFactory::instance()->config();

  config.writeEntry( "Interpreter", mInterpreterPath );
  config.writeEntry( "Antialiasing", mAntialias );
  config.writeEntry( "Platform fonts", mPlatformFonts );
  config.writeEntry( "Messages", mShowMessages );

  QString text;
  
  if( mPaletteType == COLOR_PALETTE )
    text = "color";
  else if( mPaletteType == GRAY_PALETTE )
    text = "grayscale";
  else
    text = "monochrome";
  config.writeEntry( "Palette", text );
  
  if( mBackingType == PIX_BACKING )
    text = "pixmap";
  else
    text = "store";
  config.writeEntry( "Backing", text );
  
  config.sync();
}
#include "interpreterdialog.moc"
