/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef CATALOGITEM_H
#define CATALOGITEM_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qstringlist.h>
#include <qlist.h>

class QTextStream;
class EditCommand;

/**
* This class represents an entry in a catalog.
* It contains the comment, the Msgid and the Msgstr.
* It defines some functions to query the state of the entry
* (fuzzy, untranslated, cformat).
*
* @short Class, representing an entry in a catalog
* @author Matthias Kiefer <matthias.kiefer@gmx.de>
*/
class CatalogItem
{

public:
    enum IOStatus{Ok,RecoveredParseError,ParseError,Obsolete};

	/**
	 * flags, what errors this item has
	 * @see checkArgs
	 * @see checkAccel
	 * @see checkEquation
	 * @see checkContext
	 */
	enum Error{None=0, Syntax=1, Args=2, Accel=4, Equation=8, Context=16};

    /** Constructor */
    CatalogItem();
    /** Constructs a catalogitem from a string*/
    CatalogItem(QString item);
    /** Destructor */
    ~CatalogItem();

    CatalogItem(const char *msgid, const char *msgstr,
		const char *comment);

    /**
    * returns true, if the translation is fuzzy, this means
    * if the string ", fuzzy" is contained in the comment
    */
    bool isFuzzy() const;
    /**
    * returns true, if the translation in c-format, this means
    * if the string ", c-format" is contained in the comment
    */
    bool isCformat() const;
    /**
    * returns true, if the Msgid is untranslated, this means the
    * Msgstr is empty
    */
    bool isUntranslated() const;

    bool isValid() const {return _valid;}

	void setSyntaxError(bool);

    /** returns the number of lines, the entry will need in a file */
    int totalLines() const;

    /**
    * removes the string ", fuzzy" from the comment
    * @param doIt if false, the comment is not changed, just the
    * commands for doing it are calculated
    */
    QList<EditCommand> removeFuzzy(bool doIt=true);
 
    /**
    * adds the string ", fuzzy" to the comment
    * @param doIt if false, the comment is not changed, just the
    * commands for doing it are calculated
    */
    QList<EditCommand> addFuzzy(bool doIt=true);


    /** cleares the item */
    void clear();


    /** returns the comment of this entry */
    QString comment() const{return _comment;}
    /** returns the msgid of the entry */
    QString msgid() const{return _msgid;}
    /** returns the msgstr of the entry */
    QString msgstr() const{return _msgstr;}

    QStringList msgidAsList() const;
    QStringList msgstrAsList() const;
    QStringList commentAsList() const;

    void setComment(QString com){_comment=com;}
    void setMsgid(QString msg){_msgid=msg;}
    void setMsgstr(QString msg){_msgstr=msg;}

    void processCommand(EditCommand* cmd, bool undo=false);


    /** returns the complete entry as a string */
    QString asString();

    /** @return a list of tags in the msgid */
    const QStringList& tagList();

	/** 
	* checks, if number and type of arguments (%1, %s,...)
	* are the same in msgid and msgstr
	* @return false, if an error has been found
	*/
	bool checkArgs();
	
	/**
	* checks if msgid and msgstr have the same number of occurences 
	* of accelMarker 
	* @return false, if an error has been found
	*/
	bool checkAccelerator(QChar accelMarker);
	
	/**
	* when the msgid is of the form abc=yxz it checks if abc is 
	* exactly the same in msgstr. Useful for checking KDE's desktop.po
	* files.
	* @return false, if an error has been found
	*/
	bool checkEquation();
	
	/**
	 * checks if the context of a message is translated.
	 * This is a patch to gettext which is used by KDE. It adds one line
	 * in front of the actual text, which starts with _: and then contains
	 * a small description. The msgstr must not translate this.
	 * @param reg a regular expression, that determines what is 
	 * the context information
	 */
	bool checkForContext(const QRegExp& reg);
	
	/**
	 * @return the errors of this item. @see CatalogItem::Error
	 */
	int errors() const { return _error; }

	/**
	 * makes some sanity checks and set status accordingly
	 * @return the new status of this item
	 * @see CatalogItem::Error
	 * @param accelMarker a char, that marks the keyboard accelerators
	 * @param contextInfo a regular expression, that determines what is 
	 * the context information
	 */
	int checkErrors(QChar accelMarker, const QRegExp& contextInfo);
	
    /**
    * reads an entry from the stream
    * @see QTextStream
    */
    IOStatus read(QTextStream& stream);
    /**
    * writes an entry to the stream
    * @see QTextStream
    */
    void write(QTextStream& stream) const;

    void operator=(const CatalogItem& rhs) {
	_comment = rhs._comment;
	_msgid = rhs._msgid;
	_msgstr = rhs._msgstr;
	_valid = rhs._valid;
    }

private:
   QString _comment;
   QString _msgid;
   QString _msgstr;

   QStringList *_tagList;

   bool _valid;

   int _error;

};

#endif // CATALOGITEM_H
