/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *  Copyright (C) 2000-2001 Matthew Tuck
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "dialogs.h"
#include "download-dirs.h"
#include "dialog-common-elements.h"

/* Structure that keeps the widgets with the information of the new download
 * file */
typedef struct 
{
    GtkWidget *dlg;            /* GNOME dialog */
    GtkWidget *url;            /* Entry */
    DlPolicyWidgets dl_policy; /* Download dir, etc */
} NewDlData;

/* Check whether the current URL is a valid one */

static gboolean
valid_url_check (gchar *url, GtkWidget *dlg)
{
    GtkWidget *error;

    g_return_val_if_fail (url != NULL, FALSE);
    g_return_val_if_fail (dlg != NULL, FALSE);

    if (!file_data_url_valid (url)) {
	error = gnome_error_dialog_parented (_ ("Invalid URL."), 
                                             GTK_WINDOW (dlg) );
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
	return FALSE;
    } else {
        return TRUE;
    }
}

/* Call back function used when the OK and CANCEL buttons get pressed */
static void
ok_new_file (GtkWidget *widget, NewDlData *new_dl_data)
{
    FileData *file_data;
    gchar *url;
    DlPolicyData dl_policy;
    GtkWidget *error;

    /* Get the URL data */
    url = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_dl_data->url)));

    /* Make sure it's right */
    if (!valid_url_check (url, new_dl_data->dlg)) {
        g_free (url);
        return;
    }

    /* We don't support downloading directories yet */
    if (!download_not_dir_check (url, new_dl_data->dlg)) {
	g_free (url);
	return;
    }

    /* Get the download policy info from the widgets */
    get_dl_policy_data (&dl_policy, &(new_dl_data->dl_policy));

    /* Create the file data with the information */
    file_data = dl_policy_file_data_create (url, &dl_policy);

    /* Add the new download file to the file list (if not a duplicate) */
    if (!file_list_single_add (GTK_CLIST (dl_file_list), file_data)) {
	/* URL on the file list */
	gchar *error_msg;

	error_msg = g_strconcat (_ ("URL:\n"), url, "\n",
				 _ ("already in the list"), NULL);
        error = gnome_error_dialog_parented (
            error_msg, GTK_WINDOW (new_dl_data->dlg));
	gnome_dialog_run_and_close (GNOME_DIALOG (error));
	file_data_free (file_data);
	g_free (error_msg);
        return;
    }
    /* If dialog changes default download dir update the def dl dir */
    change_default_download_dir (dl_policy.download_dir);

    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (url);
    g_free (new_dl_data);
}

static void
cancel_new_file (GtkWidget *widget, NewDlData *new_dl_data)
{
    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (new_dl_data->dlg);
    g_free (new_dl_data);
}

/* If possible, calculate the directory by applying pattern matching
   to the URL */
static void
match_cb (GtkWidget *widget, NewDlData *new_dl_data)
{
    gchar *url, *dir = NULL;

    url = g_strdup (gtk_entry_get_text (GTK_ENTRY (new_dl_data->url)));

    if (!valid_url_check (url, new_dl_data->dlg))
        return;

    download_dirs_suggest_dir (url, &dir);
    dir_suggested_check (dir, &(new_dl_data->dl_policy), new_dl_data->dlg);

    g_free(url);
}

static void
dialog_new_drag_data_received (GtkWidget *widget, GdkDragContext *context,
                               gint x, gint y, 
                               GtkSelectionData *selection_data,
                               guint info, guint time,
                               NewDlData *new_dl_data)
{
    g_return_if_fail (widget != NULL);
    g_return_if_fail (selection_data->data != NULL);
    gtk_entry_set_text (GTK_ENTRY (new_dl_data->url), selection_data->data);
}

/* Fill the contents of the dialog */
static void
dialog_new_contents (NewDlData *new_dl_data)
{
    GtkWidget *label;
    gchar *default_protocol;

    g_return_if_fail (new_dl_data != NULL);

    switch (gtm_pref.presume_protocol) {
        case 0:
            default_protocol = g_strconcat("", NULL);
            break;
        case 1:
            default_protocol = g_strconcat("http://", NULL);
            break;
        case 2:
            default_protocol = g_strconcat("ftp://", NULL);
            break;
        default:
            default_protocol = g_strconcat("", NULL);
    }
    /* URL */
    label = gtk_label_new (_ ("Please enter URL of the file to download:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			label, FALSE, FALSE, 0);
    new_dl_data->url = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(new_dl_data->url), default_protocol);
    gtk_widget_set_usize (new_dl_data->url, 300, -2);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (new_dl_data->dlg)->vbox), 
			new_dl_data->url, FALSE, FALSE, 0);
    gtk_widget_grab_focus (new_dl_data->url);

    /* Download Policy */
    add_download_policy_widget (new_dl_data->dlg,
                                &(new_dl_data->dl_policy),
				gtm_pref.def_download_dir,
                                GTK_SIGNAL_FUNC (match_cb),
                                new_dl_data);
}    

/* Dialog used to add a new file to download */
void
dialog_new (void)
{
    NewDlData *new_dl_data = g_new (NewDlData, 1);

    /* Create dialog box */
    new_dl_data->dlg = 
	gnome_dialog_new (_ ("New file to download"),
			  GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
			  NULL);
    gnome_dialog_set_parent (GNOME_DIALOG (new_dl_data->dlg),
			     GTK_WINDOW (gtm_app));

    /* Fill the contents */
    dialog_new_contents (new_dl_data);

    /* Set up OK/Cancel callbacks, defaults, etc */
    configure_ok_cancel_dialog (GNOME_DIALOG (new_dl_data->dlg),
                                GTK_SIGNAL_FUNC (ok_new_file),
                                GTK_SIGNAL_FUNC (cancel_new_file),
				(gpointer) new_dl_data);

    /* Set drag 'n drop stuff */
    gtk_drag_dest_set (new_dl_data->url,
                       GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
                       target_table (), target_table_count (),
                       GDK_ACTION_COPY | GDK_ACTION_MOVE);

    /* Set the signal handlers for the content */
    gtk_signal_connect (GTK_OBJECT (new_dl_data->url), "activate",
			GTK_SIGNAL_FUNC (ok_new_file), 
			(gpointer) new_dl_data);
    gtk_signal_connect (GTK_OBJECT (new_dl_data->url), "drag_data_received",
                        GTK_SIGNAL_FUNC (dialog_new_drag_data_received),
                        (gpointer) new_dl_data);

    /* Show the dialog box and every thing it contains */
    gtk_widget_show_all (new_dl_data->dlg);
}
