/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team (pan@rebelbase.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <stdio.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>

#include <gnome.h>

#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <pwd.h>

#include <pan/debug.h>
#include <pan/pan-glib-extensions.h>

#include <pan/globals.h>
#include <pan/prefs.h>
#include <pan/util.h>

/***
****
****  GUI
****
***/

void
pan_info_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);
	pan_lock();
	gnome_ok_dialog (str);
	pan_unlock();
	g_free (str);
}

void
pan_error_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);
	pan_lock();
	gnome_error_dialog (str);
	pan_unlock();
	g_free (str);
}



static pthread_t has_lock_thr = 0;

void
pan_lock_unconditional_from (const gchar * file, int line)
{
	pthread_t thr = pthread_self();
	if (thr == has_lock_thr) {
		g_warning (_("ERROR thread %lu attempted double lock! %s:%d"),
			(gulong)thr, file, line);
		pan_warn_if_reached ();
	}
	gdk_threads_enter();
	has_lock_thr = thr;
	debug3 (DEBUG_LOCK,"thread %lu entered gdk_threads from %s %d", thr, file, line);
}

void
pan_lock_from (const gchar * file, int line)
{
	pthread_t thr = pthread_self();

	if (thr==Pan.main_t && !has_lock_thr)
	{
		debug3 (DEBUG_LOCK,"mainthread %lu attempted lock from %s %d", thr, file, line);
	}
	else
	{
		pan_lock_unconditional_from (file, line);
	}
}

void
pan_unlock_unconditional_from (const gchar * file, int line)
{
	has_lock_thr = (pthread_t)0;
	gdk_threads_leave();
	debug3 (DEBUG_LOCK,"thread %lu left gdk_threads from %s %d", pthread_self(), file, line);
}

void
pan_unlock_from (const gchar* file, int line)
{
	pthread_t thr = pthread_self();

	if (thr==Pan.main_t && !has_lock_thr)
	{
		debug3 (DEBUG_LOCK,"mainthread %lu attempted unlock from %s %d", thr, file, line);
	}
	else
	{
		pan_unlock_unconditional_from (file, line);
	}
}



/***
****
****  HOSTNAME
****
***/

char*
get_host_name (void)
{
	char *ptr;
	char hostname[256] = { '\0' };

	if ((gethostname(hostname, sizeof(hostname))) == -1) {
		if ((ptr = getenv("HOST")) != NULL)
			strncpy(hostname, ptr, MAXHOSTNAMELEN);
		else if ((ptr = getenv("HOSTNAME")) != NULL)
			strncpy(hostname, ptr, MAXHOSTNAMELEN);
		else
			hostname[0] = '\0';
	}
	hostname[255] = '\0';
	ptr = strtok (hostname, ".");

	return g_strdup (ptr);
}

/* Returns the fully qualified domain name */
char*
get_fqdn (const char * host)
{
	char name[512] = { '\0' };
	char line[1025];
	char fqdn[1024] = { '\0' };
	char * cp;
	char * domain = NULL;
	FILE * inf;
	struct hostent * hp;
	struct in_addr in;

	if (host) {
		if (strchr(host, '.'))
			return g_strdup(host);
		strncpy(name, host, MAXHOSTNAMELEN);
	}
	else {
		if (gethostname(name, MAXHOSTNAMELEN))
			return NULL;
	}

	if (isdigit((int)*name)) {
		in.s_addr = inet_addr (name);
		if ((hp = gethostbyaddr((char *) &in.s_addr, 4, AF_INET)))
			in.s_addr = (*hp->h_addr);
		return g_strdup((hp && strchr(hp->h_name, '.') ? hp->h_name : (char *) inet_ntoa(in)));
	}
	if ((hp = gethostbyname(name)) && !strchr(hp->h_name, '.'))
		if ((hp = gethostbyaddr(hp->h_addr, hp->h_length, hp->h_addrtype)))
			in.s_addr = (*hp->h_addr);

	sprintf(fqdn, "%s", hp ? strchr(hp->h_name, '.')
		? hp->h_name : (char *) inet_ntoa(in)
		: "");
	if (!*fqdn || (fqdn[strlen(fqdn) - 1] <= '9')) {
		*fqdn = '\0';
		inf = fopen("/etc/resolv.conf", "r");
		if (inf) {
			while (fgets(line, 1024, inf)) {
				line[1024] = '\0';
				g_strchug(line);
				g_strchomp(line);
				if (strncmp(line, "domain ", 7) == 0) {
					domain = line + 7;
					break;
				}
				if (strncmp(line, "search ", 7) == 0) {
					domain = line + 7;
					cp = strchr(domain, ' ');
					if (cp)
						*cp = '\0';
					break;
				}
			}
			if (domain)
				sprintf(fqdn, "%s.%s", name, domain);
			fclose(inf);
		}
	}
	return g_strdup (fqdn);
}

/***
****
****  VERSION CHANGE
****
***/


#define DB_FORMAT_LAST_CHANGED_IN_VERSION "0.9.4"
#define DB_FORMAT_VERSION 3

void
check_and_warn_if_version_change (void)
{
	int old_db_version =
		gnome_config_get_int ("/Pan/General/DBFormatVersion=0");
	gchar * username = gnome_config_get_string ("/Pan/User/Full_Name");
	const gboolean new_user = !is_nonempty_string (username);

	/* 0.8.1 beta 1 used a different key for finding db format version */
	if (old_db_version < 1)
	{
		gchar * s = gnome_config_get_string ("/Pan/General/AppVersion");
		if (!pan_strcmp(s, "0.8.1beta1"))
			old_db_version = 1;
		g_free (s);
	}

	/* do something if the versions differ... */
	if (old_db_version != DB_FORMAT_VERSION)
	{
	       	/* not a new user, let them know what's happening... */
		if (!new_user)
		{
			gchar * msg = g_strdup_printf (
_("Thanks for upgrading to Pan %s!"
"\n"
"\nThe database formats changed in version %s of Pan."
"\nYou need to delete your Pan database files (located in the directory"
"\n``.pan/data'' in your home directory, by default) before using this"
"\nversion of Pan."
"\n"
"\nFor more information, see the installation section of the"
"\nPan FAQ at <http://pan.rebelbase.com/faq.html>."),
				VERSION,
				DB_FORMAT_LAST_CHANGED_IN_VERSION);
			GtkWidget * dialog = gnome_message_box_new (msg,
				GNOME_MESSAGE_BOX_INFO,
				GNOME_STOCK_BUTTON_CANCEL,
				GNOME_STOCK_BUTTON_NEXT,
				NULL);
			int button = gnome_dialog_run_and_close (
				GNOME_DIALOG(dialog));
			if (button == 0)
				exit (0);
			g_free (msg);
		}

		/* update the config so that the user won't be warned
		   the next time they run Pan. */
		gnome_config_set_int (
			"/Pan/General/DBFormatVersion", DB_FORMAT_VERSION);
	}

	g_free (username);
}

/***
****
****  LINKIFY TEXT
****
***/

/* This is pretty much taken straight from
   slrn's art.c, written by John E. Davis */
static const gchar*
find_url (const gchar  * in,
          guint        * setme_url_len)
{
	const gchar * ptr;

	while ((ptr = strstr (in, "://")) != NULL)
	{
		const gchar * begin;
		const gchar * end;
		gboolean again;

		/* walk back to the beginning of the word */
		begin = ptr;
		while ((begin>in) && isalpha((int)begin[-1]))
			--begin;

		/* all registered and reserved scheme names are >= 3 chars long */
		if (begin+3 > ptr) {
			in = ptr + 3;
			continue;
		}

		end = ptr;
		again = TRUE;
		while (again)
		{
			gchar ch = *end;

			switch (ch)
			{
				case ',':
				case ';':
				case '.':
					ch = *++end;
					if (isspace((int)ch)) {
						--end;
						again = FALSE;
					}
					break;

				case ' ':
				case '\t':
				case '\n':
				case '\"':
				case '}':
				case '{':
				case ')':
				case '>':
				case '<':
				case 0:
					again = FALSE;
					break;

				default:
					++end;
					break;
			}
		}

		*setme_url_len = (guint)(end - begin);
		if (*setme_url_len < 7)
			continue;

		if (!strncmp (end-3, "://", 3))
			continue;

		return begin;
	}

	return NULL;
}

gchar*
linkify_text (const gchar * text)
{
	guint len = 0;
	gchar * retval;
	GString * str;
	const gchar * prev = text;
	const gchar * pch;
	debug_enter ("linkify_text");

       	str = g_string_new (NULL);
	while ((pch = find_url(prev,&len)) != NULL)
	{
		gchar * url = g_strndup (pch, len);
		if (pch != prev)
			pan_g_string_append_len (str, prev, (guint)(pch-prev));

		g_string_append (str, "<a href=\"");
		g_string_append (str, url);
		g_string_append (str, "\">");
		g_string_append (str, url);
		g_string_append (str, "</a>");

		prev = pch + len;
		g_free (url);
	}
	g_string_append (str, prev);

	retval = str->str;
	g_string_free (str, FALSE);

	debug_exit ("linkify_text");
	return retval;
}

#if 0
static void
test_linkify_text (void)
{
	const gchar * test_type = "linkify_text";
	gint test_num = 0;
	const gchar * in;
	const gchar * expect;
	gchar * out;

	in = "Go visit http://pan.rebelbase.com/ right now.";
	expect = "Go visit <a href=\"http://pan.rebelbase.com/\">http://pan.rebelbase.com/</a> right now.";
	out = linkify_text (in);
	regression_test_result_va (!strcmp(out,expect), test_type, ++test_num, "%s", in);
	g_free (out);

	in = "Go visit http://pan.rebelbase.com/. right now.";
	expect = "Go visit <a href=\"http://pan.rebelbase.com/\">http://pan.rebelbase.com/</a>. right now.";
	out = linkify_text (in);
	regression_test_result_va (!strcmp(out,expect), test_type, ++test_num, "%s", in);
	g_free (out);

	in = "The url is http://pan.rebelbase.com/; foo";
	expect = "The url is <a href=\"http://pan.rebelbase.com/\">http://pan.rebelbase.com/</a>; foo";
	out = linkify_text (in);
	regression_test_result_va (!strcmp(out,expect), test_type, ++test_num, "%s", in);
	g_free (out);
}
#endif


/***
****
****  UNSORTED
****
***/


gchar*
get_default_author_address (void)
{
	gchar * email = gnome_config_get_string ("/Pan/User/Email");

	if (email != NULL)
		g_strstrip (email);

	/* if no email in prefs, do the best we can */
	if (!is_nonempty_string(email)) {
		gchar * host = get_host_name();
		gchar * fqdn = get_fqdn (host);
		replace_gstr (&email,
			g_strdup_printf ("%s@%s", g_get_user_name(), fqdn));
		g_free (host);
		g_free (fqdn);
	}

	return email;
}

gchar*
get_default_author_from (void)
{
	gchar * retval;
	gchar * real = gnome_config_get_string ("/Pan/User/Full_Name");
	gchar * email = get_default_author_address ();

	g_strstrip (real);

	/* if no name give a default; if not quoted, quote it */
	if (!is_nonempty_string (real))
		replace_gstr (&real, g_strdup_printf ("Dave Rhodes"));
	if (real!=NULL && *real!='"')
		replace_gstr (&real, g_strdup_printf ("\"%s\"", real));

	retval = g_strdup_printf ("%s <%s>", real, email);

	g_free (real);
	g_free (email);

	return retval;
}

gchar*
get_date_display_string (time_t date, const gchar * fmt)
{
	struct tm tm_date;
	gchar date_buf[256];

        if (display_article_dates_in_local_time)
                localtime_r (&date, &tm_date);
        else
                gmtime_r (&date, &tm_date);

	strftime (date_buf, sizeof(date_buf), fmt, &tm_date);
	date_buf[sizeof(date_buf)-1] = '\0';
	return g_strdup (date_buf);
}

void
open_outside_file (const char *fname)
{
	const char * mime_type = NULL;
	const char * mime_prog = NULL;
	char **argv;
	int argc;

	if (!((mime_type = gnome_mime_type (fname))))
		return;
	if (!((mime_prog = gnome_mime_program (mime_type))))
		return;
	if (!strstr (mime_prog, "%f"))
		return;

	argv = g_strsplit (mime_prog, " ", -1);
	for (argc=0; argv[argc]; argc++)
		if (!strcmp(argv[argc], "%f"))
			replace_gstr (argv+argc, g_strdup(fname));

	gnome_execute_async (NULL, argc, argv);

	g_strfreev (argv);
}            
