/******************************************************************************
  File:     $Id: hpdj.h,v 1.12 1999-09-02 08:06:08+02 Martin Rel $
  Contents: Ghostscript device driver "hpdj":
	    Common types and internal interfaces
  Author:   Martin Lottermoser, Metzgerfeldweg 9, 85737 Ismaning, Germany.
	    E-mail: martin.lottermoser@mch20.sbs.de

*******************************************************************************
*									      *
*	Copyright (C) 1996, 1997, 1998 by Martin Lottermoser		      *
*	All rights reserved						      *
*									      *
*******************************************************************************

  See gdevhpdj.c for introductory comments and the license information.

******************************************************************************/

#ifndef _hpdj_h	/* Inclusion protection */
#define _hpdj_h

/* Configuration management identification */
#pragma ident "@(#)$Id: hpdj.h,v 1.12 1999-09-02 08:06:08+02 Martin Rel $"

/*****************************************************************************/

/* Ghostscript header files */
#include "gdevprn.h"	/* This implies including "gxdevice.h". */
/* I'm also using the type 'bool' defined in stdpre.h. */

/* Other headers internal to hpdj */
#include "pclcomp.h"

/*****************************************************************************/

/* Constants used by several files */

/* Conversion factors (exact). These are deliberately floating point numbers. */
#define MM_PER_IN	25.4
#define BP_PER_IN	72.0	/* PostScript units (big points) per inch */

/* Derived values */
#define BP_PER_MM	(BP_PER_IN/MM_PER_IN)

/* Error message prefix */
#define ERRPREF		"? Device hpdj: "
/* Warning message prefix */
#define WARNPREF	"?-W Device hpdj: "

/*****************************************************************************/

/*  Arguments for the PCL Page Size command.

    This driver assumes that media size and media orientation can be chosen
    independently, provided the Page Orientation command is sent after the
    Page Size command. (The description of the Page Orientation command does
    not indicate that its outcome depends on the page size code sent
    previously.) Hence the page size codes can be used to characterize media
    size independent of media orientation.

    Why bother about this point? Because of page size codes 81 and -81 for
    COM10 envelopes: A DeskJet printer switches to landscape orientation on
    receiving 81 but remains in the previous orientation when receiving -81.
    Under the assumption above (verified for 81 on the DeskJet 850C) this and
    possibly existing similar distinctions can be ignored.

    Some values are marked "BPD05446". These are given in the document listing
    the PCL-3 commands for the DeskJet 1120C series printers. This document
    gives only the names of sizes but not their dimension. Unfortunately, the
    names are not always specific enough to determine the size from other
    sources. BPQ04632 is not much help because it lists only sizes where the
    dimension is universally known anyway. In some unambiguous cases I was able
    to supply the size from other sources, in particular from Adobe's PPD 4.3
    specification. Only these entries will be used by hpdj (see hpdjdata.c if
    you want to check this).

    Two values are marked "PS3". These are from Adobe's "LanguageLevel 3
    Specification and Adobe PostScript 3 Version 3010 Product Supplement"
    (dated 1997-10-10), page 352, where eight PCL5 page size codes are listed.
    Six of these agree with PCL-3 page size codes, and the remaining two are
    those inserted here and marked "PS3". My assumption is that PCL-3 and PCL5
    use a common convention for page size codes, but I'm not sufficiently
    certain of this. Hence they are supported by the driver but not mentioned
    in the documentation. I do not know any PCL-3 printer supporting these
    two sizes as discrete sizes.
*/
typedef enum {
  pcl_ps_none = 0,	/* Actually, some HP documents state that this means
    "default size", but the value is used here only as a sentinel. */
  pcl_ps_executive = 1,	/* US Executive (7.25 x 10.5 in).
    According to Adobe's PPD 4.3 specification, media sizes called by this name
    vary by up to 1/2 in. */
  pcl_ps_letter = 2,	/* US Letter (8.5 x 11 in) */
  pcl_ps_legal = 3,	/* US Legal (8.5 x 14 in) */
  pcl_ps_tabloid = 6,	/* US Tabloid (11 x 17 in) or Ledger (17 x 11 in).
			   BPD05446 calls it "US Ledger" and BPQ04632 lists the
			   size as 11 x 17 in. I'm following PPD 4.3. */
  pcl_ps_statement = 15, /* US Statement (5.5 x 8.5 in) (BPD05446) */
  pcl_ps_superB = 16,	/* Super B (305 x 487 mm) (BPD05446) */
  pcl_ps_A6 = 24,	/* ISO/JIS A6 (105 x 148 mm) */
  pcl_ps_A5 = 25,	/* ISO/JIS A5 (148 x 210 mm) */
  pcl_ps_A4 = 26,	/* ISO/JIS A4 (210 x 297 mm) */
  pcl_ps_A3 = 27,	/* ISO/JIS A3 (297 x 420 mm) (BPD05446) */
  pcl_ps_JISB5 = 45,	/* JIS B5 (182 x 257 mm). This is distinct from
			   ISO B5 (176 x 250 mm). */
  pcl_ps_JISB4 = 46,	/* JIS B4 (257 x 364 mm). This is distinct from
			   ISO B4 (250 x 353 mm). (BPD05446) */
  pcl_ps_hagaki = 71,	/* Hagaki card (100 x 148 mm) */
  pcl_ps_A6postcard = 73, /* "ISO A6 Postcard (envelope)" (BPD05446). No idea
			   what is meant by this. The ISO A series is not
			   intended for envelopes. */
  pcl_ps_4x6 = 74,	/* US Index card (4 x 6 in) */
  pcl_ps_5x8 = 75,	/* US Index card (5 x 8 in) */
  pcl_ps_monarch = 80,	/* US Monarch (3.875 x 7.5 in) in PCL-5 (PS3) */
  pcl_ps_com10 = 81,	/* US No. 10 envelope (4.125 x 9.5 in).
    I've used 81 instead of -81 because of the DJ500 and DJ500C models which
    do not accept -81. */
  pcl_ps_DL = 90,	/* ISO DL (110 x 220 mm) */
  pcl_ps_C5 = 91,	/* ISO C5 (162 x 229 mm) in PCL-5 (PS3) */
  pcl_ps_C6 = 92,	/* ISO C6 (114 x 162 mm) */
  pcl_ps_custom = 101,	/* Custom page size */
  pcl_ps_US_A2 = 109,	/* US A2 envelope (4.375 x 5.75 in) */
  pcl_ps_long3 = 110,	/* "NEC Long3 Envelope" (BPD05446) */
  pcl_ps_long4 = 111,	/* "NEC Long4 Envelope" (BPD05446) */
  pcl_ps_kaku = 113	/* "Kaku Envelope" (BPD05446). PPD 4.3 lists two
			   candidates: EnvKaku2 (240 x 332 mm) and EnvKaku3
			   (216 x 277 mm). */
} pcl_page_size;

/* Macro to test for envelope sizes */
#define is_envelope(pcl_size)					\
  ((pcl_size) == pcl_ps_com10 || (pcl_size) == pcl_ps_DL ||	\
   (pcl_size) == pcl_ps_C6    || (pcl_size) == pcl_ps_US_A2)

/*---------------------------------------------------------------------------*/

/*  Media dimensions: the type 'med_dim'

    Actually, this type does not merely associate media dimensions with a PCL
    Page Size code: it distinguishes between width and height and therefore
    also makes a statement about orientation. I find this subject a bit messy
    and shall therefore elaborate.

    Consider a printer with an input tray containing rectangular sheets. There
    are then 8 possible ways to put a sheet into the input tray. If the sheets
    have different width and height, are unmarked and both sides are
    indistinguishable, these 8 possibilities reduce to 2 groups of 4 equivalent
    orientations each. Assuming further that the printer always fetches the
    sheets from the same edge of the tray, these two groups of orientations can
    be distinguished by whether the shorter or the longer edge of the sheet is
    orthogonal to the feeding direction. These input orientations are commonly
    called "short edge first" and "long edge first".

    Now do a thought experiment: Suppose your printer supports a particular
    media size. Choose a PCL code selecting that size (there may be several,
    just pick one and repeat for the others if desired) and choose a particular
    way to put the sheet into the input tray. Send the Page Size code to the
    printer followed by the Page Orientation = "portrait" command and some
    raster graphics data. Take the page as printed and hold it such that each
    pixel line runs from left to right and lines sent later are below those
    sent earlier. The orientation of the sheet in this "PCL space" defined by
    the pixels (think of the sheet as being attached to the pixels instead of
    vice versa) is now determined as a function of the page size code and
    the orientation in the input tray:

    (*)  Page Size code + input orientation => sheet orientation in PCL space

    In the case of only two inequivalent orientations for the sheet, the
    result can be expressed by stating whether the sheet orientation is
    portrait (width <= height) or landscape (width > height) in PCL space.
    If one fixes the page size code, the resulting map associates input
    orientation in the tray (short edge or long edge first) with sheet
    orientation (portrait or landscape) in PCL space.

    DeskJet printers are not able to sense the dimensions of a sheet put into
    the input tray. Although a PostScript page description usually fixes the
    "media size" (two numbers for width and height without a specification
    which is which), a driver normally has no access to the input orientation
    and must therefore make an assumption. The validity of this assumption must
    be ensured by the user, just as the user is also responsible for providing
    a sheet of appropriate size.

    The type 'med_dim' defined here now represents the following assumption:

      If there is an entry of type 'med_dim' with a particular page size code
      and a certain width and height, then sending that page size code and
      raster graphics data as described above will give a sheet orientation in
      PCL space of the specified dimensions.

    In order to ensure the validity of this assumption, (*) has to be inverted
    to find the input orientation based on page size code and orientation in
    PCL space, and the user has to supply a sheet of appropriate size and
    orientation in the input tray. Because the user knows only the size s/he
    has requested, this works reliably only if the list of instances of type
    'med_dim' is such that the media size determines the input orientation
    uniquely. See the comments for the variable 'hpdj_mdim[]' in hpdjdata.c for
    the continuation of this discussion.

    Incidentally, hpdj assumes throughout that PCL space and PostScript device
    space have parallel orientation, i.e., the device space x axis is assumed
    to be horizontal and the y axis to be vertical in PCL space. This is of
    interest for the correct interpretation of some device parameters.
 */
typedef struct {
  pcl_page_size code;	/* PCL Page Size code to be sent to the printer */
  float width, height;	/* Measured in "units in default user space", and in
    hpdj I take that to mean exactly 1/72 in, i.e. 1 bp, regardless of any
    scaling setpagedevice might do in setting up the default user space
    whenever the PageSize Policy permits it. */
} med_dim;

/*---------------------------------------------------------------------------*/

/* Margin description */
typedef struct {
  pcl_page_size code;	/* PCL Page Size code to be sent to the printer */
  float left, bottom, right, top;	/* measured in bp.
    These are the margins enforced by the printer if it is sent 'code'
    as a page size code followed by a "page orientation = portrait"
    command. The designation of each margin refers to PCL space (see above the
    comments on the type 'med_dim'). */
} margin_desc;

/*---------------------------------------------------------------------------*/

/*  Data for the support of custom page sizes.
    The limits and margins are assumed to refer to PCL space for the PCL Page
    Size code "custom page size".
    See the discussion for the type 'med_dim' above and the comments in
    hpdjdata.c where instances of type 'custom_page_desc' are instantiated.
 */
typedef struct {
  float width_min, width_max, height_min, height_max;	/* Limits in bp */
  float left, bottom, right, top;			/* Margins in bp */
} custom_page_desc;

/*****************************************************************************/

/* DeskJet colour capabilities

   This type is used for capabilities as well as for the mode currently chosen.
   A model capable of a particular mode is also capable of all modes with
   smaller numerical values.

   For DeskJet printers, there are at least two kinds of ink cartridges:
   one contains only black ink, and the other the three colours normally used
   for printing: cyan, magenta and yellow (CMY). Not all DeskJet printers are
   colour capable, and not all can hold both kinds of cartridges at the same
   time. I have therefore classified the colour capabilities of DeskJet
   printers in four categories:
   - "mono": Has only a black cartridge.
   - "cmy": Can print in colour mode, but only without black. These are printers
     with room for a single cartridge, which can either be a black or a colour
     one.
   - "cmy_plus_k": Holds a colour and a black cartridge simultaneously, but the
     inks are chemically incompatible and should not be overlayed.
   - "cmyk": Has all four inks available and they can be mixed.
   Apparently, a choice of 'cmy' for a printer capable of 'cmy_plus_k' is
   remapped to the higher mode by the printer. That, at least, happens with a
   DeskJet 850C.

   This classification does not support the new photo cartridges which demand
   a six-component colour space.
*/
typedef enum {mono, cmy, cmy_plus_k, cmyk} colour_type;

/*-----------------------------------------------------------------------------

  Colour Model
  ------------
  PostScript assumes that each device has a "native device colour space" which
  can be one of 'DeviceRGB', 'DeviceCMYK' and 'DeviceGray'. Every colour
  specification with respect to any colour space is ultimately mapped into the
  device's native space.

  Ghostscript describes the colour capabilities of a device internally by the
  'color_info' field in the device structure. The field is of type
  'gx_device_color_info', defined in gxdevice.h, and described in drivers.txt.
  However, I still have difficulties deciding whether my understanding of
  ghostscript's internal behaviour with respect to this structure is correct
  and I have even found discrepancies between what is documented and
  ghostscript's actual behaviour. Hence I'll give here a short summary of my
  assumptions. Any discrepancy between these and ghostscript's actual behaviour
  is likely to show up as a bug in this driver. I do not find it worth the
  effort to disentangle ghostscript's code in order to discover what is really
  going on under which circumstances.

  Ghostscript represents the colour of a pixel by a value of the unsigned
  integer type 'gx_color_index'. These values are not set by ghostscript but
  are determined by calling a colour mapping function defined by the device,
  passing a colour specification in the device's native colour space in the
  arguments. The intensities of the components in this specification are
  represented by values of the unsigned integer type 'gx_color_value'. The
  maximum value is 'gx_max_color_value' which is equal to
  2^gx_max_color_value_bits - 1. In gs 5.50, gx_max_color_value_bits
  is 8*sizeof(unsigned short) and therefore at least 16.

  The fields in the 'gx_device_color_info' structure have the following meaning:

    int num_components;
      This identifies the native device colour space. Permitted values are
      1 (DeviceGray), 3 (DeviceRGB) and 4 (DeviceCMYK). This value determines
      which of a device's colour mapping routines will be called by ghostscript.
      These are map_rgb_color for 1 and 3, and map_cmyk_color for 4. In the case
      of 1 (DeviceGray), map_rgb_color is called only for shades of grey, i.e.,
      with equal values for all RGB components.
    int depth;
      Number of bits used to store a single pixel's colour value (i.e., pixmap
      depth). Legal values are 1, 2, 4, 8, 16, 24 and 32. Of course, a colour
      mapping function must always return a 'gx_color_index' value in the range
      0 to 2^depth - 1.
    gx_color_value max_gray;
    gx_color_value max_color;
      One less than the number of different intensity levels the device can
      display per pixel and colour component, specified independently for
      DeviceGray and the coloured native device colour spaces. This differs
      from the description in drivers.txt which suggests that these values
      refer to grey and non-grey colours to be displayed. Looking into gxcmap.c
      we find however that these values are used alternatively, based on
      whether 'num_components' is larger than 1 or not.
      Ghostscript does not normally do any halftoning if the device can display
      at least 32 intensities according to the relevant max_* value.
    gx_color_value dither_grays;
    gx_color_value dither_colors;
      Number of levels available for dithering. Again, drivers.txt suggests
      that these values are inspected according to the colour to be rendered.
      The only places these values are used are in gxdither.c in the routines
      gx_render_device_gray() and gx_render_device_color(), respectively.
      Closer inspection by debugger in gs 5.10 revealed that although
      gx_render_device_gray() is indeed called if a grey colour is to be
      rendered by a device with an RGB native device colour space (i.e., colour
      mode CMY or CMY+K for hpdj), for DeviceCMYK gx_render_device_color() is
      also called for grey values (RGB are zero, white takes on various values),
      although apparently not exclusively. This is due to the behaviour of
      cmap_cmyk_direct() in gxcmap.c: there is a comment there stating that the
      missing conversion to grey if C=M=Y is deliberate. This even applies to
      the case when all three are zero.


  DeskJets have a simple colour model without dependencies between pixels or
  colours (except a possible incompatibilty between black and the other inks).
  For each pixel, you can choose which colours among those available you wish
  to put there and in which intensity. For the DeskJet 500 series, the only
  choices are "on" and "off", i.e., 1 bit per colour and pixel. Later products
  supporting HP's Color Resolution Enhancement technology (C-REt) offer more
  choices. The DeskJet 850C, for example, permits 2 bits per pixel and hence
  up to 4 different intensities for each colour component.

  The number if intensity levels can at least partially be chosen independently
  for each colour component. Unfortunately, ghostscripts internal colour model
  is not able to express such a situation. Hence gs will produce colours which
  are slightly off whenever a user chooses different and non-zero intensity
  levels for black and CMY. I think I've overstepped the limits of what is
  possible with ghostscript's rendering routines in this respect.

  Of the "colour modes" defined in 'colour_type', 'cmy' and 'cmy_plus_k' do not
  seem to fit ghostscript's internal colour interface as described above.
  Fortunately, however, this impression is wrong: theoretically, CMY is
  equivalent to RGB, each component being the complement of a component in the
  other model. (Black is only needed to correct for the real-world deficiencies
  of inks.) As ghostscript does not interpret 'gx_color_index' values, we can
  lie to it and claim to have an RGB device while actually we have a CMY one.
  And at least for the case of two intensity levels, CMY+K can be treated just
  as CMY with a single modification: replace every "composite black" value (all
  three colours "on") by true black. For more than two levels, hpdj does the
  same but only for gray values (equal intensities for the three components).
  This latter case is also problematical if the number of intensity levels is
  different for black and CMY.


  hpdj uses pixmaps of various depths, depending on the colour mode and the
  number of intensity levels requested at runtime. 'gx_color_index' values are
  divided into 1 (colour mode "mono") or 4 bit fields, each of length
  'bits_per_components' stored in the device structure. From least to most
  significant bit the order is K(CMY), corresponding to the order in which bit
  planes for these colours have to be sent to the printer.
  The '*_INDEX' macros can be used as indices into arrays having the same order.
*/

/* Macros used for non-monochrome modes and a pixmap depth of 4 */
#define BLACK_MASK	1LU
#define CYAN_MASK	2LU
#define MAGENTA_MASK	4LU
#define YELLOW_MASK	8LU

#define BLACK_INDEX	0
#define CYAN_INDEX	1
#define MAGENTA_INDEX	2
#define YELLOW_INDEX	3

/*****************************************************************************/

/*  Lists of supported resolutions.
    To my knowledge, all DeskJets have the property that a resolution which
    is supported for colour is also supported for monochrome.
    Note that this type definition assumes that the permitted horizontal and
    vertical resolutions in PCL space are independent of all other settings,
    in particular of the page size code as sent by hpdj.
 */
typedef struct {
  short h, v;		/* Horizontal and vertical resolution in
			   pixels per inch in PCL space */
  bool for_colour;	/* Is this resolution also supported in colour mode? */
} supported_resolution;

/*****************************************************************************/

/*  Type for boolean parameters where the default is to decide on context.
 */
typedef enum {automatic, requested, declined} context_bool;

/*****************************************************************************/

/*  Identifiers of recognized models. These values double as indices into the
    array 'hpdj_model[]' in order to cheaply get a printable name for a model.
    Care must therefore be taken to ensure that the ordering is the same. If
    you compile without NDEBUG being defined, this is checked in
    hpdj_get_params() at runtime. */
typedef enum {
  hpdj_none = 0,	/* This value is intended *only* as a compile-time
    default, in particular for people distributing compiled versions of hpdj
    and wishing to ensure that the user makes a deliberate choice of a
    particular model. If this model is used, the driver will issue an
    appropriate error message. */
  hpdj500, hpdj500c, hpdj510 /* can be treated like the DeskJet 520 */,
  hpdj520, hpdj540, hpdj550c, hpdj560c,
  hpdj850c, hpdj855c /* is treated like the DeskJet 850C */,
  hpdj_unspec
} Model;

/*---------------------------------------------------------------------------*/

/* Description of printer models */
typedef struct {
  Model model;
    /*	Model identifier */
  const char *name;
    /*	Name of the model as used in the "Model" parameter */
  colour_type colour_capability;
    /*  Highest possible colour mode */
  const margin_desc *margin;
    /*  List of supported media sizes and associated margins for printing in
	black, terminated by an entry with code == pcl_ps_none. This is always
	non-NULL. */
  float bottom_increment;
    /*  Change of bottom margin when printing in colour (given in bp).
	Older DeskJets have a larger bottom margin when printing in colour. */
  const custom_page_desc *custom;
    /*  This may be NULL, in which case the model does not support custom page
	sizes. Otherwise it is a list of entries, terminated with an entry
	having 0 for 'width_max'. The margins refer to printing in black. */
  const supported_resolution *resolutions;
    /*  List of raster graphics resolutions supported by this model. This
	variable is either NULL (all resolutions are accepted by the driver) or
	terminated with an entry having resolution (0, 0). The list may contain
	entries with 'for_colour' being true even if the printer is only
	capable of monochrome, i.e., this list cannot be used to deduce colour
	capabilities. Each pair of resolution values may appear at most once in
	this list. */
} model_description;

/*****************************************************************************/

/* Device structure */
typedef struct hpdj_device_s {
  gx_device_common;		/* Attributes common to all devices */
  gx_prn_device_common;		/* Attributes for all printers */

  Model model;			/* Model identifier */
  int
    media_type,			/* PCL code for media type */
    print_quality,		/* PCL code for print quality */
    dry_time,			/* -1, 0 or dry time in seconds */
    undoc1,			/* ESC & u # D. 0 means "dont't use". */
    black_levels,		/* Number of intensity levels for black ink */
    cmy_levels,			/* Number of intensity levels for CMY inks */
    bits_per_component;		/* Number of bits per component used in
				   'gx_color_index' values. This is stored just
				   for convenience and could be derived from
				   other data when needed. */
  pcl_page_size ps_code;	/* PCL code for page size */
  float
    top_shift,			/* Shift values for the device coordinate */
    left_shift;			/*   origin in device space in bp */
  bool
    landscape;			/* Coordinate rotation for landscape? */
  pcl_compression
    compression_method;		/* Default method to use for compression */
  colour_type colour_mode;	/* Colour mode requested */
  context_bool manualfeed;	/* Require manual feed or not */
  char
    *margin_file		/* Margin file name. Non-NULL if
				   'margin_overrides' is non-NULL. */
#ifndef HPDJ_NO_PAGECOUNTFILE
    ,
    *pagecount_file		/* Name of page count file. May be NULL. */
#endif
    ;
  margin_desc *margin_overrides;
    /*  List of supported media dimensions and corresponding raster mode margins
	for printing in black. This is usually NULL, meaning that the 'margin'
	list from the device's model should be used. If non-NULL, the list must
	be terminated with an entry having 'code' == pcl_ps_none. If this field
	is non-NULL, only the dimensions listed here will be supported. This
	is also true for custom page sizes: if the model supports them ('custom'
	is non-NULL), 'margin_overrides' must contain an entry with page size
	code 'pcl_ps_custom' in order for custom page sizes to be supported by
	the device.
    */
  bool
    initialized,		/* Have we called init()? */
    is_valid;			/* Hack. See hpdj_put_params(). */
} hpdj_device;

/*****************************************************************************/

/* Type to describe an association between a string and an integer */
typedef struct {
  const char *name;
  int value;
} StringToInt;

/*****************************************************************************/

/* Interface offered by gdevhpdj.c */
extern hpdj_device gs_hpdj_device;	/* Interface to ghostscript kernel */
extern int hpdj_set_page_layout(hpdj_device *dev);

/* Interface offered by hpdjdata.c */
extern const med_dim hpdj_mdim[];
  /* This list is terminated by an entry with a 'code' of 'pcl_ps_none'. */
extern const model_description hpdj_model[];
  /* This list is terminated by an entry with a NULL 'name'. */

/* Interface offered by hpdjparm.c */
extern dev_proc_get_params(hpdj_get_params);
extern dev_proc_put_params(hpdj_put_params);
int hpdj_read_margins(hpdj_device *dev);
extern const StringToInt hpdj_colour_mode_list[];
  /* This list is terminated by an entry with a NULL 'name'. */

/* Interface offered by hpdjprn.c */
extern dev_proc_print_page(hpdj_print_page);

/* Interface offered by pagecount.c */
extern int pcf_getcount(const char *filename, unsigned long *count);
extern int pcf_inccount(const char *filename, unsigned long by);

/*****************************************************************************/

#endif	/* Inclusion protection */
