/******************************************************************************
  File:     $Id: gdevhpdj.c,v 2.27 1999-09-05 08:28:43+02 Martin Rel $
  Contents: Ghostscript device driver "hpdj":
	    Device structure and some device procedures
  Author:   Martin Lottermoser, Metzgerfeldweg 9, 85737 Ismaning, Germany.
	    E-mail: martin.lottermoser@mch20.sbs.de

*******************************************************************************
*									      *
*	Copyright (C) 1996, 1997, 1998 by Martin Lottermoser		      *
*	All rights reserved						      *
*									      *
*******************************************************************************

  hpdj can be used under the terms of the GNU Library General Public License
  (LGPL), Version 2 (June 1991). For the purpose of the license, all files
  distributed with hpdj together count as one "library", except that the
  pclcomp.* files form a sub-library which can be used independently.

  USE THIS AT YOUR OWN RISK.

  See also the file README.hpdj in the hpdj distribution.

*******************************************************************************

Driver description
==================

Applicability
-------------
The ghostscript device "hpdj" is intended as an interface to Hewlett-Packard
DeskJet printers, in particular the following models:

  - HP DeskJet 500
  - HP DeskJet 500C
  - HP DeskJet 510 and 520
  - HP DeskJet 540
  - HP DeskJet 550C
  - HP DeskJet 560C
  - HP DeskJet 850C and 855C

These models accept various dialects of HP's Printer Command Language (PCL),
Level 3 (PCL-3). Unfortunately, there is no "universal" version of PCL.
Apart from adhering to one of the main groups of the PCL language, in particular
PCL-3 and PCL-5, each PCL printer usually implements a model-specific subset
of all PCL commands.

However, given any two PCL printers, the overlap between their command
sets is usually large. The upshot of this is that in many and in particular
the simple cases a driver for a specific model works also for some other
printers, possibly with some loss of quality in the printout.

For the models listed above, the information about their PCL command sets
has been taken from the following references:

  - Hewlett-Packard
    "Technical Reference Guide for the HP DeskJet 500 Series Printers"
    First edition, October 1994
    Manual Part Number: C2170-90099
    (Quoted as "TRG" for short.)
  - Hewlett-Packard
    "HP DeskJet 800 Series Printer Commands"
    1st February 1996
    HP FIRST #: 2998
    Obtainable from http://www.hp.com under the name BPD02926.

Neither document is exhaustive. In particular, the TRG does not provide the
information necessary to access all features a series 500 DeskJet supports.
The second document is merely a list of basic formatting commands understood by
these printers (and almost identical to commands accepted by printers in the
DeskJet 500 series).

In addition I have used several other documents which are quoted in the
comments at the appropriate point. Anything with a name like "BPDxxxxx" is
available from http://www.hp.com in the directory
cposupport/printers/support_doc and under the name bpdxxxxx.html.

As far as undocumented PCL features are used in this driver, their treatment is
based on experiments I have done with a DeskJet 850C. To some extent, I have
also looked at the output generated by HP's Windows driver for this model.

The only printer model I have continuous access to is a DJ 850C. As far as
model-dependent PCL code or behaviour is concerned, I am therefore somewhat
limited as to testing. However, among the few problem reports I have received
since hpdj 2.0 in March 1997, none was concerned with the generated PCL code.


Runtime Configuration and Limitations
-------------------------------------
See the manual page, gs-hpdj(1).



Compile Time Configuration
--------------------------
The driver code assumes an ISO/ANSI C compiler and library.

The following preprocessor symbols can be used to modify the default behaviour
of the device and can be overridden at runtime:

  HPDJ_DEFAULT_COMPRESSION	The default is "pcl_cm_crdr" (of type
				'pcl_compression' defined in pclcomp.h), i.e.,
				method 9. You should only need to change this
				if your default model is the DJ 500, in which
				case the value should be "pcl_cm_delta".
  HPDJ_DEFAULT_MODEL		The default is "hpdj_none" (of type 'Model'
				defined in hpdj.h). Unless this is overridden
				at runtime by the "-sModel" option, any call to
				hpdj will terminate with an error message
				asking the user to select a model.
  HPDJ_DEFAULT_RESOLUTION	The default is "300" (pixels per inch).
  HPDJ_MARGIN_FILE		If defined, this must be a string giving the
				path name of a margin description file. The
				model "unspec" will in that case not use the
				margin descriptions for the DJ 850C as a
				default but will read from this source instead.

You should ensure that HPDJ_DEFAULT_COMPRESSION, HPDJ_DEFAULT_MODEL and
HPDJ_DEFAULT_RESOLUTION form a valid combination of values.

The following symbols change the behaviour of the device:

  HPDJ_INPUTATTRIBUTES		See hpdjparm.c.

  HPDJ_NO_PAGECOUNTFILE		Define this if you don't want to have the
				page count file feature of hpdj. I suspect that
				you must define this on non-UNIX platforms.

  HPDJ_PRINT_BLANK_ROWS		See hpdjprn.c.

The following preprocessor defines exist to adapt to changes in ghostscript:

  HPDJ_INPUTMEDIA_PRN		See hpdjparm.c.

  HPDJ_MEDIASIZE		Try defining this if the clipping is wrong when
				you are printing in landscape orientation with
				future (> 5.50) versions of ghostscript. I don't
				have much hope that it will work, though (my
				powers of clairvoyance are unfortunately
				non-existing).

  HPDJ_USEPAGESIZE		This must be non-zero for older versions of
				ghostscript (< 3.46) which have a variable
				'PageSize' in the device structure instead of
				'MediaSize'. The default is zero unless
				GS_REVISION is defined and less than 346.

In addition, you can define NDEBUG in the usual way to suppress some assertions.

******************************************************************************/

#ifndef lint
/* Configuration management identification */
static const char
  cm_id[] = "@(#)$Id: gdevhpdj.c,v 2.27 1999-09-05 08:28:43+02 Martin Rel $";
#endif

/*****************************************************************************/

#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE	500
#endif

/* Special Aladdin header, must be included before <sys/types.h> on some
   platforms (e.g., FreeBSD). Here apparently needed because of <stdio.h>. */
#include "std.h"

/* Standard headers */
#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

/* hpdj headers */
#include "hpdj.h"
#include "pclcomp.h"

/* Additional ghostscript headers besides those included from hpdj.h */
#include "gsmatrix.h"

/*****************************************************************************/

/* Items which can be set at compile time */

#ifndef HPDJ_DEFAULT_COMPRESSION
#define HPDJ_DEFAULT_COMPRESSION pcl_cm_crdr /* of type 'pcl_compression' */
#endif

#ifndef HPDJ_DEFAULT_MODEL
#define HPDJ_DEFAULT_MODEL	hpdj_none	/* of type 'Model' (hpdj.h) */
#endif

#ifndef HPDJ_DEFAULT_RESOLUTION
#define HPDJ_DEFAULT_RESOLUTION	300		/* pixels per inch */
#endif


/* Change macros */

#ifndef HPDJ_USEPAGESIZE
#if defined(GS_REVISION) && GS_REVISION < 346
#define HPDJ_USEPAGESIZE	1
#else
#define HPDJ_USEPAGESIZE	0
#endif
#endif

/*****************************************************************************/

#ifndef lint
static const char info_string[] = "@(#)HPDJ_MEDIASIZE is "
#ifndef HPDJ_MEDIASIZE
  "not "
#endif
  "defined.";
#endif	/* lint */

/*****************************************************************************/

/* Forward declarations of local device procedures */
static dev_proc_get_initial_matrix(hpdj_get_initial_matrix);
static dev_proc_map_cmyk_color(hpdj_map_cmyk_color);
static dev_proc_map_cmyk_color(hpdj_map_cmyk_color_flex);
static dev_proc_map_rgb_color(hpdj_map_rgb_color);
static dev_proc_map_rgb_color(hpdj_map_rgb_color_flex);
static dev_proc_open_device(hpdj_open_device);

/*****************************************************************************/

/* Variables */

/* Device procedure table
   This is the actual procedure table only while the device is closed.
   The 'open_device' method of the "prn" device constructs a new procedure
   table from the rendering operations of the "memory" or "clist" device
   and the non-rendering operations in this table. gdev_prn_close() restores
   this table.
 */
static gx_device_procs hpdj_procs = {
  hpdj_open_device,			/* open_device */
  hpdj_get_initial_matrix,		/* get_initial_matrix */
  NULL,					/* sync_output */
  gdev_prn_output_page,			/* output_page */
  gdev_prn_close,			/* close_device */
  hpdj_map_rgb_color,			/* map_rgb_color */
  NULL,					/* map_color_rgb
    drivers.txt states that a printer driver must provide this function, but
    this is apparently not true. It seems rather to be intended for devices
    supporting colour maps. */
  NULL,					/* fill_rectangle */
  NULL,					/* tile_rectangle */
  NULL,					/* copy_mono */
  NULL,					/* copy_color */
  NULL,					/* draw_line */
  NULL,					/* get_bits */
  hpdj_get_params,			/* get_params */
  hpdj_put_params,			/* put_params */
  hpdj_map_cmyk_color			/* map_cmyk_color */
#if 0	/* The following definitions are not presently needed. */
  ,
  NULL,					/* get_xfont_procs */
  NULL,					/* get_xfont_device */
  NULL,					/* map_rgb_alpha_color */
  NULL,					/* get_page_device */
  NULL,					/* get_alpha_bits */
  NULL,					/* copy_alpha */
  NULL,					/* get_band */
  NULL,					/* copy_rop */
  NULL,					/* fill_path */
  NULL,					/* stroke_path */
  NULL,					/* fill_mask */
  NULL,					/* fill_trapezoid */
  NULL,					/* fill_parallelogram */
  NULL,					/* fill_triangle */
  NULL,					/* draw_thin_line */
  NULL,					/* begin_image */
  NULL,					/* image_data */
  NULL,					/* end_image */
  NULL,					/* strip_tile_rectangle */
  NULL,					/* strip_copy_rop */
  NULL					/* get_clipping_box */
#endif
};


/* Device structure instance (device prototype) */
hpdj_device gs_hpdj_device = {
  prn_device_std_body(
    hpdj_device,		/* device type */
    hpdj_procs,			/* device procedures */
    "hpdj",			/* device name */
    DEFAULT_WIDTH_10THS,	/* page width in 0.1" (+/- 1.27 mm) */
    DEFAULT_HEIGHT_10THS,	/* page height in 0.1" (+/- 1.27 mm) */
    HPDJ_DEFAULT_RESOLUTION,	/* horizontal resolution in ppi */
    HPDJ_DEFAULT_RESOLUTION,	/* vertical resolution in ppi */
    0.0, 0.0, 0.0, 0.0,		/* margins around printable area, in inches.
				   See hpdj_set_page_layout(). */
    1,				/* color_info.depth: monochrome */
    &hpdj_print_page		/* page printing procedure called by */
  ),				/*   gdev_prn_output_page() */
  HPDJ_DEFAULT_MODEL,		/* model of device */
  0,				/* media type: plain paper */
  0,				/* print quality: normal */
  -1,				/* set dry time: no */
  0,				/* undoc1: don't generate this command */
  2,				/* black levels */
  0,				/* CMY levels */
  1,				/* bits per colour component */
  pcl_ps_none,			/* page size code (will be reset) */
  0.0,				/* top shift */
  0.0,				/* left shift */
  false,			/* No rotation of coordinates for landscape */
  HPDJ_DEFAULT_COMPRESSION,	/* default compression method */
  mono,				/* default colour mode */
  automatic,			/* manual feed: decide on context */
  NULL,				/* margin file name: none */
#ifndef HPDJ_NO_PAGECOUNTFILE
  NULL,				/* page count file: none */
#endif
  NULL,				/* device-specific margin specifications: no */
  false,			/* initialized */
  true				/* is_valid */
};

/******************************************************************************

  Function: hpdj_get_initial_matrix

  This function returns the initial matrix for the device.

  hpdj's write_page() routine assumes that the device coordinate origin is
  located at CAP = (0, 0) on the PCL "logical page" and that ghostscript's
  'prn' device returns scanlines in top to bottom order and within a line from
  left to right, both as seen from PCL space (the "physical page").

  The logical page consists of the physical page minus unprintable regions at
  the left, right and top. Actually, the TRG does not make clear whether the
  unprintable region at the top is part of the logical page or not. However,
  I have determined experimentally that the DJ 850C for the page sizes A4 and
  DL starts clipping raster rows at a distance from row 0 which corresponds
  to the media height minus the published top and bottom margins. Although
  the size of the top margin is only 1 mm, I still consider this to show
  that the top margin cannot be considered to be part of the logical page.

  To avoid further confusion, I should also point out that there is a top
  margin in PCL which is different from the unprintable region at the top. The
  former can be set by the PCL command "Top Margin" and counts from the top of
  the logical page. hpdj sets it to zero. All unqualified references to any
  "margin" in hpdj refer to an unprintable region.

  The PCL origin (CAP = (0, 0)) is located at the left edge of the logical page
  and downwards from its top edge by the PCL top margin.

  The initial matrix is now determined from placing the device coordinate
  origin in the top left corner of the printable area in PCL space (origin of
  the logical page), with the x axis pointing to the right and the y axis
  downwards, and the default user space coordinate origin either in the lower
  left corner of the page with the usual orientation of the axes or in the
  lower right corner with landscape orientation.

  There is no support here for the special behaviour with envelopes as
  demanded in the PostScript Language Reference Manual Supplement for version
  2017. See the comments in the man page for my reasons.

******************************************************************************/

static void hpdj_get_initial_matrix(gx_device *device, gs_matrix *mptr)
{
  hpdj_device *dev = (hpdj_device *)device;
#ifdef HPDJ_TRACE
  fputs("! hpdj_get_initial_matrix()...\n", stderr);
#endif

  if (dev->landscape) {
    /*
      For landscape orientation, rotate user coordinates by +90 degrees and
      shift the user origin into the lower right corner of PCL space.
      The resulting CTM is uniquely characterized by the following relations
      between coordinates from user to device space:
	(0, 0)		  -> (width, height)
	(0, width in bp)  -> (0, height)
	(height in bp, 0) -> (width, 0)
      I am assuming that the 'width' and 'height' variables in the device
      structure refer to x and y extensions, respectively, in device space.
    */
    mptr->xx = 0;
    mptr->xy = -dev->HWResolution[1]/BP_PER_IN;
      /* The PostScript variable 'HWResolution' is in ppi along the device
         space x and y axes */
    mptr->yx = -dev->HWResolution[0]/BP_PER_IN;
    mptr->yy = 0;
    mptr->tx = dev->width;
    mptr->ty = dev->height;
  }
  else /* Otherwise, choose the usual relation between the spaces */
    gx_default_get_initial_matrix(device, mptr);

  /*  Finally, shift the device space origin to the top-left corner of the
      printable area. I am deliberately not using the corresponding shift
      feature in gx_device_set_margins() because it achieves its effect by
      using the 'Margins' array which should remain at the user's disposal for
      correcting misadjustments. Note in particular that the margins depend on
      the PCL Page Size code and a correction for misalignment is unlikely to
      show the same variations. Hence a user would have to specify different
      'Margins' values for different media sizes. In addition,
      gx_device_set_margins() will not work correctly for landscape orientation
      anyway. See hpdj_set_page_layout() for further remarks.
  */
  {
    gs_matrix translation;

    /*	Translation in device space by top and left margins. I am not using
	HWMargins[] because I'm not sure in which default user space this is to
	be interpreted (setpagedevice can rotate the coordinate system). In
	fact, I've not even found a statement that HWMargins[] is to be
	interpreted in default user space, although it is used in that manner
	in gs_initclip(). That is the reason why I've introduced the otherwise
	logically superfluous values 'left_shift' and 'top_shift' in the device
	structure.
    */
    gs_make_translation(
      -dev->left_shift/BP_PER_IN*dev->HWResolution[0],	/* in pixels */
      -dev->top_shift /BP_PER_IN*dev->HWResolution[1],
      &translation);

    /* Multiply the initial matrix from the right with the translation matrix,
       i.e., in going from user to device space the translation will be applied
       last. */
    gs_matrix_multiply(mptr, &translation, mptr);
  }

  return;
}

/******************************************************************************

  Function: hpdj_map_cmyk_color

  This function returns a colour index for the best approximation the device
  can achieve for the specified CMYK colour.

  This function is assumed to be called only for 2-level CMYK printing or with
  black or white colour specifications.

******************************************************************************/

static gx_color_index hpdj_map_cmyk_color(gx_device *device,
  gx_color_value cyan, gx_color_value magenta, gx_color_value yellow,
  gx_color_value black)
{
  gx_color_index value = 0;
  gx_color_value threshold = gx_max_color_value/2;

#ifdef HPDJ_TRACE
  fprintf(stderr,
    "! hpdj_map_cmyk_color() called for CMYK = (%hu, %hu, %hu, %hu).\n",
      cyan, magenta, yellow, black);
#endif

  if (cyan    > threshold) value |= CYAN_MASK;
  if (magenta > threshold) value |= MAGENTA_MASK;
  if (yellow  > threshold) value |= YELLOW_MASK;
  if (black   > threshold) value |= BLACK_MASK;

  return value;
}

/******************************************************************************

  Function: hpdj_map_cmyk_color_flex

  This is a 'map_cmyk_color' method supporting more intensity levels.

  Note that this routine presently never returns 'gx_no_color_index'. This is,
  of course, only relevant for devices with at least 32 intensity levels and
  I don't know of any PCL-3 printer with this property. Still, should one
  become available one should think about a suitably criterion for returning
  'gx_no_color_index' and modify this routine accordingly. However,
  ghostscript's colour model presents stickier problems than that...

******************************************************************************/

static gx_color_index hpdj_map_cmyk_color_flex(gx_device *device,
  gx_color_value cyan, gx_color_value magenta, gx_color_value yellow,
  gx_color_value black)
{
  gx_color_index value = 0;
  gx_color_value step;
  int level;
  hpdj_device *dev = (hpdj_device *)device;

#ifdef HPDJ_TRACE
  fprintf(stderr,
    "! hpdj_map_cmyk_color_flex() called for CMYK = (%hu, %hu, %hu, %hu).\n",
      cyan, magenta, yellow, black);
#endif

  /*  I can think of at least three methods to extract discrete values from
      a continuous intensity in the range [0, 1]:
      (a) multiply by the number of levels minus 1 and truncate,
      (b) multiply by the number of levels minus 1 and round, and
      (c) multiply by the number of levels and truncate, except for an
	  intensity of 1 in which case one returns the number of levels minus 1.
      For intensity values which can be represented exactly, i.e.,

	intensity = i/(levels-1)	for some non-negative i < levels,

      these three methods are identical. (a) is however inappropriate here
      because for less than 32 levels ghostscript already provides intensity
      values which have been adjusted to a representable level. A rounding
      error could now result in a level which is too small by one. I prefer (c)
      because it gives equal shares to all levels.

      I'm using integer arithmetic here although floating point numbers would
      be more accurate. This routine is, however, called quite frequently, and
      the loss in accuray is acceptable as long as the values determined for
      'step' are large compared to the number of levels. If you consider
      "large" as meaning "10 times as large", the critical boundary is at about
      81 levels. The highest number of intensity levels at present supported by
      DeskJets is apparently 16 (DJ 890C).

      A more accurate implementation would determine 'step' as a floating point
      value, divide the intensity by it, and take the floor (entier) of the
      result as the component intensity.
  */

  /* The order has to be (YMC)K from left to right */
  if (dev->colour_mode != mono) {
    step = gx_max_color_value/dev->cmy_levels;

    level = yellow/step;
    if (level >= dev->cmy_levels) level = dev->cmy_levels - 1;
    value = level << dev->bits_per_component;
    level = magenta/step;
    if (level >= dev->cmy_levels) level = dev->cmy_levels - 1;
    value = (value | level) << dev->bits_per_component;
    level = cyan/step;
    if (level >= dev->cmy_levels) level = dev->cmy_levels - 1;
    value = (value | level) << dev->bits_per_component;
  }
  if (black != 0) {
    step = gx_max_color_value/dev->black_levels;
    level = black/step;
    if (level >= dev->black_levels) level = dev->black_levels - 1;
    value |= level;
  }

#ifdef HPDJ_TRACE
  fprintf(stderr, "  returning colour index %lX.\n", (unsigned long)value);
#endif
  return value;
}

/******************************************************************************

  Function: hpdj_map_rgb_color_flex

  This is a flexible 'map_rgb_color' method.

******************************************************************************/

static gx_color_index hpdj_map_rgb_color_flex(gx_device *device,
  gx_color_value red, gx_color_value green, gx_color_value blue)
{
  hpdj_device *dev = (hpdj_device *)device;

#ifdef HPDJ_TRACE
  fprintf(stderr,
    "! hpdj_map_rgb_color_flex() called for RGB = (%hu, %hu, %hu).\n",
    red, green, blue);
#endif

  /* Treat pure grey levels differently */
  if (dev->colour_mode != cmy && red == green && green == blue)
    return hpdj_map_cmyk_color_flex(device, 0, 0, 0, gx_max_color_value - red);

  return hpdj_map_cmyk_color_flex(device, gx_max_color_value - red,
    gx_max_color_value - green, gx_max_color_value - blue, 0);
  /* This implies that for CMY+K only "true" grey shades will be printed with
     black ink, all others will be mixed from CMY. */
}

/******************************************************************************

  Function: hpdj_map_rgb_color

  This function returns a colour index for the best approximation the device
  can produce for the specified RGB value.

  The function is assumed to be called only for at most 2 intensity levels per
  component and not for CMYK.

******************************************************************************/

static gx_color_index hpdj_map_rgb_color(gx_device *device, gx_color_value red,
  gx_color_value green, gx_color_value blue)
{
  gx_color_value half = gx_max_color_value/2;
  gx_color_index value = (CYAN_MASK | MAGENTA_MASK | YELLOW_MASK);
  hpdj_device *dev = (hpdj_device *)device;

#ifdef HPDJ_TRACE
  fprintf(stderr, "! hpdj_map_rgb_color() called for RGB = (%hu, %hu, %hu).\n",
    red, green, blue);
#endif

  assert(dev->colour_mode == mono && red == green && green == blue &&
      (blue == 0 || blue == gx_max_color_value) ||
    dev->colour_mode == cmy || dev->colour_mode == cmy_plus_k);

  /* Map to CMY */
  if (red   > half) value &= ~CYAN_MASK;
  if (green > half) value &= ~MAGENTA_MASK;
  if (blue  > half) value &= ~YELLOW_MASK;

  /* Remap composite black to true black if available */
  if (dev->colour_mode != cmy &&
      value == (CYAN_MASK | MAGENTA_MASK | YELLOW_MASK))
    value = BLACK_MASK;

  return value;
}

/******************************************************************************

  Function: hpdj_set_page_layout

  This function identifies the PCL Page Size code to send to the printer,
  the margins the printer will use in that case, and it will set up the data
  for the correct initialization of the default user coordinate system.

  With DeskJet printers, the correct value for the margins is important for
  two reasons:

  - The PCL interface places the first pixel sent in the top-left corner of the
    printable area ("PCL device coordinate origin"), not in the top-left corner
    of the medium.

    Hence knowledge of the top and left margins is necessary in order to
    properly position the output on the page.

  - The print head should not move off the page in order to prevent it being
    caught at the medium's edge when moving back, or even printing on the
    printer instead of on the medium.

    As one cannot move the print head in the negative direction horizontally
    and should not move it in the negative direction vertically, this is no
    problem with the top and left margins, but one should ensure a safe
    distance from the right and bottom margins. This seems to be only a problem
    for custom page sizes because as one can determine experimentally (I have
    not found a corresponding statement from HP), if the printer is told that a
    specific media size is used, it enforces clipping boundaries at the right
    and bottom margins. The PCL Page Size code "Custom Page Size"
    ('pcl_ps_custom') apparently switches this off. It is, however, not
    accepted by all DeskJet printers.

  Of course the printer has no knowledge of the dimensions of the medium
  actually fed into the printer. This permits printing on media of arbitrary
  size, but one still must
  (a) give the printer some page size in order to fix the device coordinate
      origin (either custom page size or a size having at least the same
      printable area as the requested size), and
  (b) make sure the print head does not move off the page.

  The function returns zero iff the printer supports the requested size.
  It sets in '*dev' 'code', 'landscape', 'top_shift', 'left_shift' and the
  margins.

  If HPDJ_INPUTATTRIBUTES is non-zero, this function should never return an
  error because setpagedevice will already have determined whether the printer
  supports the size or not, based on the InputAttributes dictionary (see
  hpdj_get_params()).

******************************************************************************/

int hpdj_set_page_layout(hpdj_device *dev)
{
  bool rotate = false;
  float
    /*  Page width and height in PCL space in bp.

	Note that nothing can be gained in precision from trying to use
	MediaSize[] instead because gx_device_set_width_height() (gsdevice.c)
	assigns MediaSize[] exactly these values.

	And why, then, do I not use MediaSize[]? Because I can't be sure what
	its orientation is, i.e., which component of the array is the horizontal
	and which the vertical dimension in device space.
	gx_device_set_width_height() assumes that, e.g., MediaSize[0] is to be
	measured along the same axis as 'width'. Hence one would conclude that
	MediaSize[] is to be interpreted with the same orientation as device
	space. Looking into gs_initclip(), however, one discovers that
	MediaSize[] is interpreted in default user space!

	Unfortunately, ghostscript's documentation does not state what the
	reference system for MediaSize[] is (the same is true for several more
	parameters defined in gxdevice.h). L. Peter Deutsch, on applied to,
	replied that he hadn't looked at that code for a long time but he
	assumed that all parameters in the device structure should be
	interpreted with respect to device space (mail of 1997-12-02). This, as
	just mentioned, is in conflict with the usage of MediaSize[] and also
	HWMargins[] in gs_initclip(), and leads demonstrably to the wrong
	clipping (that, in fact, is how I discovered the problem).

	Taking all this into account, the best solution is to try and rely as
	far as possible on parameters with an umambiguous interpretation (like
	'width' and 'height'), if necessary to introduce them (like 'top_shift'
	and 'left_shift'), and ultimately to hope that future versions of
	ghostscript will not introduce incompatible changes in those of the
	ambiguous parameters this routine must touch.
    */
    width  = ((float)dev->width) /dev->HWResolution[0]*BP_PER_IN,
    height = ((float)dev->height)/dev->HWResolution[1]*BP_PER_IN,
    /* PCL space margins found */
    margins[4];
  int j;
  pcl_page_size size_requested;

#ifdef HPDJ_TRACE
  fputs("! hpdj_set_page_layout()...\n", stderr);
#endif

  /* Initialization */
  dev->ps_code = pcl_ps_none;
  dev->landscape = false;

  /* Identify media size. We use the same tolerance for matching requested
     size to media size as setpagedevice, i.e., 5 bp (roughly 1.8 mm), or rather
     5.0001 bp in order to prevent rounding errors to let setpagedevice accept
     the size while this routine refuses it.
     One might distinguish here at the outset between portrait and landscape
     by the relative size of 'width' and 'height', but the code as given here
     works also if our media size list contains entries where the PCL code
     given followed by a request to switch to portrait orientation actually
     results in landscape orientation in PCL space.
     See the comments on the type 'med_dim'.
   */
  j = 0;
  while (hpdj_mdim[j].code != pcl_ps_none &&
    (fabs(height - hpdj_mdim[j].height) > 5.0001 ||
     fabs(width  - hpdj_mdim[j].width ) > 5.0001)) j++;
  /* If it didn't work one way, try the other */
  if (hpdj_mdim[j].code == pcl_ps_none) {
    j = 0;
    while (hpdj_mdim[j].code != pcl_ps_none &&
      (fabs(width  - hpdj_mdim[j].height) > 5.0001 ||
       fabs(height - hpdj_mdim[j].width ) > 5.0001)) j++;
    if (hpdj_mdim[j].code != pcl_ps_none) {
      float temp;
      rotate = true;
      temp = width; width = height; height = temp;
    }
  }
  if (hpdj_mdim[j].code != pcl_ps_none) size_requested = hpdj_mdim[j].code;
  else size_requested = pcl_ps_none;

  /* Get model-specific information for recognized media sizes */
  if (size_requested != pcl_ps_none) {
    const margin_desc *mp;

    /* Determine whether the device supports this page size */
    if (dev->margin_overrides == NULL) mp = hpdj_model[dev->model].margin;
    else mp = dev->margin_overrides;
    while (mp->code != pcl_ps_none && mp->code != size_requested) mp++;

    if (mp->code != pcl_ps_none) {
      dev->ps_code = size_requested;
      margins[0] = mp->left;
      margins[1] = mp->bottom;
      margins[2] = mp->right;
      margins[3] = mp->top;
    }
  }

  /* If we have not been able to identify the media size or if the printer
     does not support it, try for custom page size. */
  if (dev->ps_code == pcl_ps_none && hpdj_model[dev->model].custom != NULL) {
    const custom_page_desc *cp = hpdj_model[dev->model].custom;

    /* Exchange width and height for landscape orientation.
       I am assuming here that paper with an unrecognized geometry should
       always be fed into the printer such that the result is portrait
       orientation in PCL space.
    */
    if (width > height) {
      float temp;
      temp = width; width = height; height = temp;
      rotate = true;
    }
    else rotate = false;

    /* Check limits on custom sizes. We take the first entry which fits. */
    while (cp->width_max > 0 &&
      (width  < cp->width_min  || cp->width_max  < width ||
       height < cp->height_min || cp->height_max < height)) cp++;
    if (cp->width_max <= 0) {
      fprintf(stderr, ERRPREF
	"This document requests a sheet size of %d x %d bp.\n"
	/* Note: Not "page size" because we might have rotated already. */
	"  This exceeds the custom page size limits for the DeskJet %s.\n",
	   (int)(width + 0.5), (int)(height + 0.5),
	hpdj_model[dev->model].name);
      return -1;
    }
    /* OK, page size is supported as a custom page size */

    /* Find the margin values */
    if (dev->margin_overrides != NULL) {
      margin_desc *mp = dev->margin_overrides;
      while (mp->code != pcl_ps_none && mp->code != pcl_ps_custom) mp++;
      if (mp->code != pcl_ps_custom) {
	fprintf(stderr, ERRPREF
	  "This document requests a sheet size of %d x %d bp\n"
	  "  but there is no entry for this size in the margin file\n"
	  "  %s.\n"
	  "  The size could, e.g., be supported as a custom page size.\n",
	  (int)(width + 0.5), (int)(height + 0.5), dev->margin_file);
	return -1;
      }
      dev->ps_code  = pcl_ps_custom;
      margins[0] = mp->left;
      margins[1] = mp->bottom;
      margins[2] = mp->right;
      margins[3] = mp->top;
    }
    else {
      dev->ps_code  = pcl_ps_custom;
      margins[0] = cp->left;
      margins[1] = cp->bottom;
      margins[2] = cp->right;
      margins[3] = cp->top;
    }
  }

  if (dev->ps_code == pcl_ps_none) {
    /* Requested size is unsupported */
    fprintf(stderr, ERRPREF
      "This document requests a sheet size of %d x %d bp.\n",
	 (int)(width + 0.5), (int)(height + 0.5));
    if (dev->margin_overrides != NULL)
      fputs("  The margin file does not contain an entry for this size.\n",
	stderr);
    else
      fprintf(stderr, "  This size is not supported by the DeskJet %s.\n",
	hpdj_model[dev->model].name);
    return -1;
  }

  if (rotate) {
    /*
      All DeskJet printers support landscape orientation, but not all do so
      in raster mode, and even for the others this feature does not seem to be
      intended for rotating a full page. The best solution is, however, anyway
      to rotate the image within ghostscript.

      If HPDJ_INPUTATTRIBUTES has been defined, this branch should only be
      taken until setpagedevice has been extended to react to a request to
      interpret custom page size limits as limits on portrait orientation.
      See hpdj_get_params().
    */
#ifdef HPDJ_TRACE
    fputs("! hpdj_set_page_layout(): Rotation is necessary.\n", stderr);
#endif

    /* Remember that the initial matrix must be rotated */
    dev->landscape = true;

    /* Exchange width and height in the device */
    gx_device_set_width_height((gx_device *)dev,
      width /BP_PER_IN*dev->HWResolution[0] + 0.499,
      height/BP_PER_IN*dev->HWResolution[1] + 0.499);
    /*  The values "0.499" are copied from gx_device_set_media_size() which I
	do not want to use because of the uncertainty in the interpretation of
	MediaSize[] which does not permit to predict the resulting effect on
	the 'width' and 'height' members in the device structure. And the
	latter are more important.
	But now I have to adjust MediaSize[]. At present (gs 5.50), the values
	have to be interchanged at this point. I can't say what is going to
	happen with future versions of ghostscript, though, hence the
	preprocessor statements. This measure does, however, not protect
	against all possible modifications.
    */
#ifndef HPDJ_MEDIASIZE
    {
      /* Exchange the MediaSize[] components */
      float temp;
#if HPDJ_USEPAGESIZE
#undef MediaSize
#define MediaSize PageSize	/* old gs versions use 'PageSize' */
#endif
      temp = dev->MediaSize[0]; dev->MediaSize[0] = dev->MediaSize[1];
      dev->MediaSize[1] = temp;
    }
#endif	/* HPDJ_MEDIASIZE */

    /* If the device is open at this point, reallocate storage */
    if (dev->is_open) {
      int rc;

#ifdef HPDJ_TRACE
      fputs("! hpdj_set_page_layout(): Device is open on rotation.\n", stderr);
#endif

      /* Unfortunately, gdev_prn_free() and gdev_prn_alloc() are static. */
      gdev_prn_close((gx_device *)dev);		/* ignore the result */
      rc = gdev_prn_open((gx_device *)dev);
      if (rc < 0) {
	fprintf(stderr, ERRPREF "Failure of gdev_prn_open(), code is %d.\n",
	  rc);
	return rc;
      }
    }
  }

  /* Increase the bottom margin for coloured modes */
  if (dev->colour_mode != mono)
    margins[1] += hpdj_model[dev->model].bottom_increment;

  /* Store the top and left margins in the device structure for use by
     hpdj_get_initial_matrix() */
  dev->top_shift  = margins[3];
  dev->left_shift = margins[0];

  /*  Set the margins of the printable area.
      gx_device_set_margins() (see gsdevice.c) copies the margins[] array to
      HWMargins[] which is presumably to be interpreted in default user space
      (see gs_initclip() in gspath.c), and if its second argument is true it
      also modifies the offset variable Margins[] which must be interpreted in
      canonical device units (i.e., resolution-independent) with respect to
      device space. Hence gx_device_set_margins() can only be used if device
      space and default user space have the usual relation for print devices in
      ghostscript. In addition, I prefer to reserve 'Margins' for the user as
      intended by PostScript.
  */
  if (rotate) {
    /* The left default user space margin is the bottom PCL space margin etc.
       At least this is true for the default user space as set up by
       hpdj_get_initial_matrix().
    */
    for (j = 0; j < 4; j++) dev->HWMargins[j] = margins[(j+1)%4];
  }
  else {
    /* Convert to inches */
    for (j = 0; j < 4; j++) margins[j] /= BP_PER_IN;

    gx_device_set_margins((gx_device *)dev, margins, false);
    /* Of course, I could set HWMargins[] directly also in this case. This way
       is however less prone to break on possible future incompatible changes
       to ghostscript. */
  }

  return 0;
}

/******************************************************************************

  Function: final_checks

  This function performs final consistency checks on the device structure.
  The validity of individual values is assumed to have been checked in the
  put_params routine or in whatever routine sets them.

  I am not particularly happy with this construction, but I do not know what
  all the possible sources of modifications of the device structure are and
  at which points one should require the device instance to be valid. If the
  structure is only updated by calls to the put_params routine, one might put
  these checks there instead. On the other hand, that would mean that no
  modification consisting of setting several values could be split into two
  calls to put_params. I just don't want to bother about this, hence this final
  check is delayed until here.

  The function returns zero on success and a non-zero ghostscript error value
  otherwise. In the latter case, an error message will have been issued on
  stderr.

******************************************************************************/

static int final_checks(hpdj_device *dev)
{
  int
    rh = dev->HWResolution[0] + 0.5,
    rv = dev->HWResolution[1] + 0.5;
  const supported_resolution *rp = hpdj_model[dev->model].resolutions;

  /* First check for model "none" */
  if (dev->model == hpdj_none) {
    fputs(ERRPREF, stderr);
    if (HPDJ_DEFAULT_MODEL == hpdj_none)
      fputs("This instance of hpdj has been compiled without\n"
	"  choosing a default printer model. You must therefore explicitly\n"
	"  select a model by means of the 'Model' option. Consult the manual\n"
	"  page gs-hpdj(1) for permissible values.\n",
      stderr);
    else
      fputs("Setting the printer model to \"none\" is not\n"
	"  particularly profitable. Please choose another model.\n",
	stderr);
    return_error(gs_error_rangecheck);
  }

  /* Colour capability */
  if (hpdj_model[dev->model].colour_capability < dev->colour_mode) {
    fprintf(stderr, ERRPREF "The DeskJet %s does not support colour mode %s.\n",
      hpdj_model[dev->model].name,
      hpdj_colour_mode_list[dev->colour_mode].name);
    return_error(gs_error_rangecheck);
  }

  /* Checks on resolution */
  if (rp != NULL) {
    /* Find the resolution entry */
    while (rp->h != 0 && (rp->h != rh || rp->v != rv)) rp++;
    if (rp->h == 0) {
      fprintf(stderr, ERRPREF "A resolution of %d", rh);
      if (rh != rv) fprintf(stderr, "x%d", rv);
      fprintf(stderr, " ppi is not supported\n  by the DeskJet %s.\n",
	hpdj_model[dev->model].name);
      return_error(gs_error_rangecheck);
    }

    /* Check for values illegal in coloured modes */
    if (dev->colour_mode != mono && !rp->for_colour) {
      fprintf(stderr, ERRPREF "A resolution of %d", rh);
      if (rh != rv) fprintf(stderr, "x%d", rv);
      fprintf(stderr,
	" ppi is not supported\n  by the DeskJet %s when printing in colour.\n",
	hpdj_model[dev->model].name);
      return_error(gs_error_rangecheck);
    }
  }

  /* Checks concerned with intensity levels */
  if ((dev->colour_mode == mono) != (dev->cmy_levels == 0) ||
      dev->colour_mode == cmy && dev->black_levels != 0) {
    fprintf(stderr, ERRPREF "Colour mode (%s) and intensity levels (%d, %d) "
	"are inconsistent.\n",
      hpdj_colour_mode_list[dev->colour_mode].name, dev->black_levels,
      dev->cmy_levels);
    return_error(gs_error_rangecheck);
  }
  if (dev->black_levels > 2 || dev->cmy_levels > 2) {
    if (dev->model != hpdj850c && dev->model != hpdj855c &&
	dev->model != hpdj_unspec) {
      fprintf(stderr, ERRPREF
	"More than 2 intensity levels are not supported\n"
	"  for the DeskJet %s.\n", hpdj_model[dev->model].name);
      return_error(gs_error_rangecheck);
    }

    /* Constraints for individual models */
    if (dev->model != hpdj_unspec) {
      if (rh != 300) {
	/* This is an experimentally determined constraint. I've not been able
	   to achieve more than two intensities on a DJ 850C except at 300 ppi.
	 */
	fprintf(stderr, ERRPREF
	  "For the DeskJet %s, more than 2 intensity levels are only\n"
	  "  supported at 300 ppi.\n",
	  hpdj_model[dev->model].name);
	return_error(gs_error_rangecheck);
      }
      if (dev->colour_mode == cmy) {
	/* This is an experimentally determined constraint. My attempts to
	   print with colour mode CMY on a DJ 850C lead to the colour changing
	   instead of the intensity.
	 */
	fprintf(stderr, ERRPREF
	  "For the DeskJet %s, more than 2 intensity levels are not\n"
	  "  supported for the colour mode CMY.\n",
	  hpdj_model[dev->model].name);
	return_error(gs_error_rangecheck);
      }
      if (dev->black_levels > 4 || dev->cmy_levels > 4) {
	/* See BPD02652 for the justification */
	fprintf(stderr, ERRPREF
	  "The DeskJet %s permits at most 4 intensity levels.\n",
	  hpdj_model[dev->model].name);
	return_error(gs_error_rangecheck);
      }
      if (dev->cmy_levels > 2 && dev->black_levels < 4) {
	/*  This is an experimentally determined constraint. On a DJ 850C,
	    setting the black levels to 2 or 3 lead to the colours being
	    ignored or scrambled. (Note that we can't have CMY here.)
	 */
	fprintf(stderr, ERRPREF "With more than 2 CMY intensity levels,\n"
	  "  you must set BlackLevels to 4 for the DeskJet %s.\n",
	  hpdj_model[dev->model].name);
	return_error(gs_error_rangecheck);
      }
      if (dev->cmy_levels == 2 && dev->black_levels != 4) {
	/*  This is an experimentally determined constraint. On a DJ 850C,
	    setting the black levels to 3 for 2 CMY levels lead to the colours
	    being ignored or scrambled. (Note that at least one of the levels
	    is larger than 2.)
	 */
	fprintf(stderr, ERRPREF "If you want to increase BlackLevels with\n"
	  "  colour mode CMYK, you must set it to 4 for the DeskJet %s.\n",
	  hpdj_model[dev->model].name);
	return_error(gs_error_rangecheck);
      }
      if (dev->print_quality == -1) {
	/* This is an experimentally determined constraint. On a DJ 850C and in
	   draft quality, the low intensity bit was ignored.
	 */
	fprintf(stderr, ERRPREF
	  "Printing with more than 2 intensity levels is\n"
	  "  useless in draft quality on the DeskJet %s.\n",
	  hpdj_model[dev->model].name);
	return_error(gs_error_rangecheck);
      }
    }
  }

  /* Compression methods */
  if (dev->model == hpdj500 && dev->compression_method == pcl_cm_crdr) {
    fputs(ERRPREF "The DeskJet 500 does not support compression method 9.\n",
      stderr);
    return_error(gs_error_rangecheck);
  }

  /* Dry time */
  if (dev->dry_time >= 0 && (dev->model == hpdj500 || dev->model == hpdj500c)) {
    fprintf(stderr, ERRPREF
      "The DeskJet %s does not support setting a dry time.\n",
      hpdj_model[dev->model].name);
    return_error(gs_error_rangecheck);
  }

  /* Media types */
  if (dev->media_type == 5 && dev->model != hpdj_unspec)
    fprintf(stderr, WARNPREF
      "The DeskJet %s has no knowledge of photo paper.\n",
      hpdj_model[dev->model].name);

  return 0;
}

/******************************************************************************

  Function: hpdj_open_device

  This function "opens" the device. According to drivers.txt, the 'open_device'
  functions are called before any output is sent to the device, and they must
  ensure that the device instance is valid, possibly by doing suitable
  initialization.

  This function checks for consistency between values, determines derived
  values, and "opens" the parts defined by the base class 'prn',

  The function returns zero on success and a ghostscript error value otherwise.

******************************************************************************/

static int hpdj_open_device(gx_device *device)
{
  hpdj_device *dev = (hpdj_device *)device;
  int rc;

#ifdef HPDJ_TRACE
  fputs("! hpdj_open_device()...\n", stderr);
#endif

#ifdef HPDJ_MARGIN_FILE
  /* Change default margin descriptions for 'unspec' */
  if (dev->model == hpdj_unspec && dev->margin_file == NULL) {
    /* Copy the name into the device structure */
    dev->margin_file = (char *) gs_malloc(strlen(HPDJ_MARGIN_FILE) + 1,
      /* At present, one could also use sizeof() instead of strlen() here.
	 I find this safer, though, concerning possible changes. */
      sizeof(char), "hpdj_open_device");
    if (dev->margin_file == NULL) {
      fputs(ERRPREF
	"Memory allocation failure from gs_malloc() in hpdj_open_device().\n",
	stderr);
      return_error(gs_error_VMerror);
    }
    strcpy(dev->margin_file, HPDJ_MARGIN_FILE);

    /* Read the margin file */
    if ((rc = hpdj_read_margins(dev)) != 0) {
      gs_free(dev->margin_file, strlen(dev->margin_file) + 1, sizeof(char),
	"hpdj_open_device");
      dev->margin_file = NULL;
      return rc;
    }
  }
#endif

  /* Determine derived values */
  if (hpdj_set_page_layout(dev) != 0) return_error(gs_error_rangecheck);
  if (dev->black_levels > 2 || dev->cmy_levels > 2) {
    set_dev_proc(dev, map_cmyk_color, &hpdj_map_cmyk_color_flex);
    set_dev_proc(dev, map_rgb_color,  &hpdj_map_rgb_color_flex);
  }
  /* It's not necessary to restore the old values because the new functions
     contain the functionality of the old ones. */

#ifndef HPDJ_NO_PAGECOUNTFILE
  /* Read the page count value */
  if (dev->pagecount_file != NULL) {
    unsigned long count = 0;
    if (pcf_getcount(dev->pagecount_file, &count) != 0) {
      /* pcf_getcount() has issued an error message. */
      fputs(
        "  No further attempts will be made to access the page count file.\n",
	stderr);
      gs_free(dev->pagecount_file, strlen(dev->pagecount_file) + 1,
	sizeof(char), "hpdj_open_device");
      dev->pagecount_file = NULL;
    }
    dev->PageCount = count;	/* unsigned to signed. The C standard permits
      an implementation to generate an overflow indication if the value is too
      large. I consider this to mean that the type of 'PageCount' is
      inappropriate :-). Note that hpdj does not use 'PageCount' for updating
      the file. */
  }
#endif

  /* Open the "prn" device part */
  if ((rc = gdev_prn_open(device)) != 0) return rc;

  /* Final checks */
  rc = final_checks(dev);

  return rc;
}
