/*
 * @(#)ActionEvent.java	1.21 00/02/02
 *
 * Copyright 1996-2000 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the proprietary information of Sun Microsystems, Inc.  
 * Use is subject to license terms.
 * 
 */

package java.awt.event;

import java.awt.AWTEvent;
import java.awt.Event;

/**
 * A semantic event which indicates that a component-defined action occured.
 * This high-level event is generated by a component (such as a Button) when
 * the component-specific action occurs (such as being pressed).
 * The event is passed to every every <code>ActionListener</code> object
 * that registered to receive such events using the component's
 * <code>addActionListener</code> method.
 * <P>
 * The object that implements the <code>ActionListener</code> interface
 * gets this <code>ActionEvent</code> when the event occurs. The listener
 * is therefore spared the details of processing individual mouse movements
 * and mouse clicks, and can instead process a "meaningful" (semantic)
 * event like "button pressed".
 *  
 * @see ActionListener
 * @see <a href="http://java.sun.com/docs/books/tutorial/post1.0/ui/eventmodel.html">Tutorial: Java 1.1 Event Model</a>
 * @see <a href="http://www.awl.com/cp/javaseries/jcl1_2.html">Reference: The Java Class Libraries (update file)</a>
 *
 * @author Carl Quinn
 * @version 1.21 02/02/00
 * @since 1.1
 */
public class ActionEvent extends AWTEvent {

    /**
     * The shift modifier. An indicator that the shift key was held
     * down during the event.
     */
    public static final int SHIFT_MASK		= Event.SHIFT_MASK;

    /**
     * The control modifier. An indicator that the control key was held
     * down during the event.
     */
    public static final int CTRL_MASK		= Event.CTRL_MASK;

    /** 
     * The meta modifier. An indicator that the meta key was held
     * down during the event.
     */
    public static final int META_MASK		= Event.META_MASK;

    /** 
     * The alt modifier. An indicator that the alt key was held
     * down during the event.
     */
    public static final int ALT_MASK		= Event.ALT_MASK;


    /**
     * The first number in the range of ids used for action events.
     */
    public static final int ACTION_FIRST		= 1001;

    /**
     * The last number in the range of ids used for action events.
     */
    public static final int ACTION_LAST		        = 1001;

    /**
     * This event id indicates that a meaningful action occured.
     */
    public static final int ACTION_PERFORMED	= ACTION_FIRST; //Event.ACTION_EVENT

    /**
     * The nonlocalized string that gives more details
     * of what actually caused the event.
     * This information is very specific to the component
     * that fired it.

     * @serial
     * @see getActionCommand()
     */
    String actionCommand;
    /**
     * This represents the key modifier that was selected,
     * and is used to determine the state of the selected key.
     * If no modifier has been selected it will default to
     * zero.
     *
     * @serial
     * @see getModifiers()
     */
    int modifiers;

    /*
     * JDK 1.1 serialVersionUID 
     */
    private static final long serialVersionUID = -7671078796273832149L;

    /**
     * Constructs an <code>ActionEvent</code> object.
     *
     * @param source  the object that originated the event
     * @param id      an integer that identifies the event
     * @param command a string that may specify a command (possibly one 
     *                of several) associated with the event
     */
    public ActionEvent(Object source, int id, String command) {
        this(source, id, command, 0);
    }

    /**
     * Constructs an <code>ActionEvent</code> object with modifier keys.
     *
     * @param source    the object that originated the event
     * @param id        an integer that identifies the event
     * @param command   a string that may specify a command (possibly one 
     *                  of several) associated with the event
     * @param modifiers the modifier keys held down during this action
     */
    public ActionEvent(Object source, int id, String command, int modifiers) {
        super(source, id);
        this.actionCommand = command;
        this.modifiers = modifiers;
    }

    /**
     * Returns the command string associated with this action.
     * This string allows a "modal" component to specify one of several 
     * commands, depending on its state. For example, a single button might
     * toggle between "show details" and "hide details". The source object
     * and the event would be the same in each case, but the command string
     * would identify the intended action.
     *
     *@return the string identifying the command for this event
     */
    public String getActionCommand() {
        return actionCommand;
    }

    /**
     * Returns the modifier keys held down during this action event.
     * 
     * @return the integer sum of the modifier constants
     */
    public int getModifiers() {
        return modifiers;
    }

    /**
     * Returns a parameter string identifying this action event.
     * This method is useful for event-logging and for debugging.
     * 
     * @return a string identifying the event and its associated command 
     */
    public String paramString() {
        String typeStr;
        switch(id) {
          case ACTION_PERFORMED:
              typeStr = "ACTION_PERFORMED";
              break;
          default:
              typeStr = "unknown type";
        }
        return typeStr + ",cmd="+actionCommand;
    }

}
