/*
 *  MonKT - a monitor for LinKT
 *  Copyright (C) 1998-2000 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "dxwin.h"
#include "global.h"
#include "toolbox.h"
#include "main.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <kapp.h>
#include <kmessagebox.h>

#include <qlayout.h>
#include <qwidget.h>
#include <qlabel.h>


#define BUTTONWIDTH 65
#define BUTTONHEIGHT 25



extern TopLevel *toplevel;


DXData::DXData( QListView *parent, char *dxcall, char *str_qrg, char *str_time, char *comment, char *sender )
				: QListViewItem( parent, dxcall, str_qrg, str_time, comment, sender )
{
	this->dxcall = (char *) strdup(dxcall);
}


DXData::~DXData()
{
	delete dxcall;
}


const char * DXData::getDxCall()
{
	return dxcall;
}


DXWin::DXWin( QWidget *parent, QWidget *tl )
                    : QWidget(parent)
{
   tabbar = new QTabBar( this );
   connect(tabbar, SIGNAL(selected(int)), this, SLOT(changeTab(int)));

   dxviewroot = NULL;
   akttab = -1;
   aktview = NULL;


   kontextmenu = new QPopupMenu();
   kontextmenu->insertItem( klocale->translate("&New view"), (TopLevel *)tl, SLOT(slotNewView()));
   kontextmenu->insertSeparator( -1 );
   kontextmenu->insertItem( klocale->translate("&Delete view"), (TopLevel *)tl, SLOT(slotDeleteView()));


   loadViewsList();
}


void DXWin::resizeEvent( QResizeEvent *e )
{
   s_dxviews *views;

   QWidget::resizeEvent(e);

   views = dxviewroot;

   while (views != NULL)
   {
      views->lview->setGeometry(0, tabbar->height(), width(), height()-tabbar->height());
      views = views->next;
   }

   QWidget::resizeEvent( e );
}


s_dxviews * DXWin::addDXView(char *label, double lowerqrg, double upperqrg)
{
   s_dxviews *tmp;
   bool found=false;
   struct QTab *tab;


   // Gucken, ob noch ein View "von frueher" existiert, also einer, bei
   // dem label auf NULL steht.
   tmp = dxviewroot;
   while (tmp != NULL && !found)
   {
      if (tmp->label == NULL)
         found = true;
      else
         tmp = tmp->next;
   }


   if (!found)
   {
      if (dxviewroot == NULL)
      {
         // Erster Eintrag
         dxviewroot = (s_dxviews *)malloc(sizeof(s_dxviews));
         tmp = dxviewroot;
      }
      else
      {
         tmp = dxviewroot;
         while (tmp->next != NULL) tmp = tmp->next;

         tmp->next = (s_dxviews *)malloc(sizeof(s_dxviews));
         tmp = tmp->next;
      }

      tmp->next = NULL;
   }
   tmp->lowerqrg = lowerqrg;
   tmp->upperqrg = upperqrg;
   tmp->label = (char *) strdup(label);

   if (!found)
   {
      tab = new QTab();
      tab->label = label;
      tmp->tabid = tabbar->addTab( tab );
   }
   else
   {
      tab = tabbar->tab(tmp->tabid);
      tab->label = label;
      tabbar->setTabEnabled( tmp->tabid, true );
   }
   tabbar->resize(tabbar->sizeHint());

   if (!found)
   {
		tmp->lview = new QListView( this );
      tmp->lview->setGeometry(0, tabbar->height(), width(), height()-tabbar->height());
      tmp->lview->setColumnWidth(tmp->lview->addColumn(klocale->translate("DX-Call")), 80);
      tmp->lview->setColumnWidth(tmp->lview->addColumn(klocale->translate("QRG")), 100);
      tmp->lview->setColumnWidth(tmp->lview->addColumn(klocale->translate("Time")), 50);
      tmp->lview->setColumnWidth(tmp->lview->addColumn(klocale->translate("Comment")), 200);
      tmp->lview->setColumnWidth(tmp->lview->addColumn(klocale->translate("Sender")), 80);
      tmp->itemcount = 0;
      tmp->lview->setAllColumnsShowFocus( true );
      tmp->lview->setSorting( -1 );

      connect(tmp->lview, SIGNAL(rightButtonClicked(QListViewItem *, const QPoint &, int)), this, SLOT(slotShowKontextMenu(QListViewItem *, const QPoint &, int)));
      connect(tmp->lview, SIGNAL(selectionChanged(QListViewItem *)), this, SLOT(slotSelectLine(QListViewItem *)));
	}
   tmp->lview->show();


   if (akttab != -1)
   {
      tmp->lview->hide();
      akttab = tmp->tabid;
      aktview = tmp;
   }

   return tmp;
}


void DXWin::addDXMessage(char *dx_call, char *absender, float qrg, int zeit, char *bemerkung)
{
	char str_qrg[50], str_time[50];
   s_dxviews *views;
   bool ausgeben;


   views = dxviewroot;

   while (views != NULL)
   {
      ausgeben = true;

      if (qrg < views->lowerqrg) ausgeben = false;
      if (qrg > views->upperqrg) ausgeben = false;

      if (ausgeben)
      {
		   sprintf(str_qrg, "%.1f", qrg);
		   sprintf(str_time, "%.4i", zeit);

         views->lview->ensureItemVisible(
         new DXData( views->lview, dx_call, str_qrg, str_time, bemerkung, absender )
         );
			views->itemcount++;

         while (views->itemcount > config->tblsize)
         {
         	views->itemcount--;
            delete views->lview->firstChild();
			}
      }

      views = views->next;
   }
}


void DXWin::changeTab(int id)
{
   s_dxviews *views;
   bool found=false;


   views = dxviewroot;

   while (views != NULL && !found)
   {
      if (views->tabid == akttab)
      {
         found = true;
         views->lview->hide();
      }
      else views = views->next;
   }

   views = dxviewroot;

   while (views != NULL)
   {
      if (views->tabid == id)
      {
         views->lview->show();
         akttab = id;
         aktview = views;
         return;
      }
      views = views->next;
   }
}


void DXWin::slotShowKontextMenu( QListViewItem *, const QPoint & p, int )
{
   kontextmenu->popup(p);
}


void DXWin::mousePressEvent( QMouseEvent *e )
{
   if (e->button() == RightButton)
   {
      kontextmenu->popup(QCursor::pos());
      return;
   }

   QWidget::mousePressEvent(e);
}


void DXWin::delDXView( const char  *label )
{
   s_dxviews *tmp;

   tmp = dxviewroot;
   while (tmp != NULL)
   {
      if (!strcmp(tmp->label, label))
      {
         // Ok, dieser View soll versteckt werden...
         tmp->lview->clear();
         tmp->lview->hide();

         // Tab verstecken
         tabbar->setTabEnabled( tmp->tabid, false );

         free(tmp->label);
         tmp->label = NULL;

         if (akttab == tmp->tabid)
         {
            akttab = -1;
            aktview = NULL;
         }

         return;
      }
      tmp = tmp->next;
   }
}


void DXWin::slotSaveViewList()
{
   s_dxviews *tmp;
   char str[1000];
   int fd;


   sprintf(str, "%s/dxviews.list", config->maindir);
   if ((fd = open(str, O_CREAT|O_WRONLY|O_TRUNC)) == -1) return;

   fchmod(fd,S_IRUSR|S_IWUSR);

   tmp = dxviewroot;
   while (tmp != NULL)
   {
      if (tmp->label != NULL)
      {
         sprintf(str, "%.1f;%.1f;%s\n",
                      tmp->lowerqrg, tmp->upperqrg, tmp->label);

         write(fd, str, strlen(str));
      }

      tmp = tmp->next;
   }

   ::close(fd);
}


void DXWin::loadViewsList()
{
   FILE *f;
   char str[1000];
   int i;
   double low,up;
   char label[500];


   sprintf(str, "%s/dxviews.list", config->maindir);
   if ((f = fopen(str, "r")) == NULL) return;

   while (fgets(str, 999, f) != NULL)
   {
      if ((i = POS('\n', str)) != -1) str[i] = '\0';

      if (sscanf(str, "%lf;%lf;%s", &low, &up, label) == 3)
      {
         strcpy(label, str+lPOS(';', str)+1);
         addDXView( label, low, up );
      }
   }

   fclose(f);
}


void DXWin::slotSelectLine( QListViewItem *entry )
{
   if (aktview == NULL || entry == NULL)
      toplevel->updateStatusbar( "" );
   else
      toplevel->updateStatusbar( ((DXData *)entry)->getDxCall() );
}


void DXWin::resizeTabBar()
{
   tabbar->resize(tabbar->sizeHint());
}


s_dxviews * DXWin::getViewsRoot()
{
   return dxviewroot;
}



//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////



DlgAddView::DlgAddView() : QTabDialog( 0, "", true )
{
   setCaption(klocale->translate("MonKT: Add new view..."));

   setFixedSize(350, 250);

   setupWindowGeneral();
   addTab(winGeneral, klocale->translate("&General"));

   setupWindowQRG();
   addTab(winQRG, klocale->translate("&QRG"));

   readData();

   setCancelButton();
   connect( this, SIGNAL(applyButtonPressed()), SLOT(saveData()));
}


DlgAddView::~DlgAddView()
{
}


void DlgAddView::setupWindowQRG()
{
   winQRG = new QWidget( this );

   QLabel *l = new QLabel( winQRG );
   l->setGeometry( 10, 10, width()-20, 20 );
   l->setText( klocale->translate("You can set the minimum- and the maximum-QRG") );

   l = new QLabel( winQRG );
   l->setGeometry( 10, 30, width()-20, 20 );
   l->setText( klocale->translate("for this view here:") );

   l = new QLabel( winQRG );
   l->setGeometry( 10, 60, width()-20, 20 );
   l->setText("minimum");

   minQRG = new QLineEdit( winQRG );
   minQRG->setGeometry( 10, 80, 80, 20);

   l = new QLabel( winQRG );
   l->setGeometry( 100, 80, 40, 20 );
   l->setText("khz");

   l = new QLabel( winQRG );
   l->setGeometry( 160, 60, width()-20, 20 );
   l->setText("maximum");

   maxQRG = new QLineEdit( winQRG );
   maxQRG->setGeometry( 160, 80, 80, 20);

   l = new QLabel( winQRG );
   l->setGeometry( 250, 80, 40, 20 );
   l->setText("khz");
}


void DlgAddView::setupWindowGeneral()
{
   winGeneral = new QWidget( this );

   QLabel *l = new QLabel( winGeneral );
   l->setGeometry( 10, 40, width()-20, 20 );
   l->setText( klocale->translate("Short description:") );

   name = new QLineEdit( winGeneral );
   name->setGeometry( 10, 60, 100, 20);
   name->setMaxLength( 20 );
}


void DlgAddView::readData()
{
   // Voreinstellungen
   name->setText("new view");
   minQRG->setText("0.0");
   maxQRG->setText("99999999.9");
}


void DlgAddView::saveData()
{
   // Check, ob die Variablen korrekte Werte haben
   lowerqrg = atof(minQRG->text());
   upperqrg = atof(maxQRG->text());

   if (lowerqrg >= upperqrg)
   {
      KMessageBox::error( this, klocale->translate("The min-QRG must be lower than the max-QRG!") );
      return;
   }

   accept();
}


void DlgAddView::getData( double & min, double & max, char *desc )
{
   min = lowerqrg;
   max = upperqrg;
   strcpy(desc, name->text());
}


//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////


DlgDelView::DlgDelView( s_dxviews *views ) : QDialog( 0, "", true )
{
   s_dxviews *tmp;

   setCaption(klocale->translate("MonKT: Delete view..."));

   setFixedSize(250, 200);

   dxviews = views;


   QLabel *l = new QLabel( this );
   l->setGeometry( 10, 10, width()-20, 20 );
   l->setText( klocale->translate("Which view do you want to delete?") );

   liste = new QListBox( this );
   liste->setGeometry( 10, 30, width()-20, 100 );


   // OK- und Abbruch-Knopf
   int abstand = (width()-(2*BUTTONWIDTH))/3;
   ok = new QPushButton( this );
   ok->setText( klocale->translate ("&Ok") );
   ok->setGeometry(abstand, height()-BUTTONHEIGHT-10, BUTTONWIDTH, BUTTONHEIGHT);
   connect( ok, SIGNAL(clicked()), this, SLOT(slotOkClicked()) );

   abort = new QPushButton( this );
   abort->setText( klocale->translate ("&Abort") );
   abort->setGeometry(abstand*2+BUTTONWIDTH, height()-BUTTONHEIGHT-10, BUTTONWIDTH, BUTTONHEIGHT);
   connect( abort, SIGNAL(clicked()), this, SLOT(reject()) );


    tmp = views;
    while (tmp != NULL)
    {
       liste->insertItem( tmp->label );
       tmp = tmp->next;
    }
}


DlgDelView::~DlgDelView()
{
}


void DlgDelView::slotOkClicked()
{
   s_dxviews *tmp;

   if (liste->currentItem() == -1)
   {
      KMessageBox::error( this, klocale->translate("You must select a view.") );
      return;
   }

   tmp = dxviews;
   while (tmp != NULL)
   {
      if (!strcmp(tmp->label, liste->text(liste->currentItem())))
      {
         accept();
         return;
      }
      tmp = tmp->next;
   }

   KMessageBox::error( this, klocale->translate("You must select a view.") );
}


const char * DlgDelView::getListName()
{
   return liste->text(liste->currentItem());
}


