/*
 *  comp.c - procedures for (de)compression with a static huffman-table
 *
 *  Copyright (C) 1999 Johann Hanne, DH3MB. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "comp.h"

static struct huff_comp_table_struct huff_comp_table[] = {
#include "table_comp"
};

static struct huff_decomp_tree_struct huff_decomp_tree[] = {
#include "table_decomp"
};

/* Compresses srclen bytes, beginning at *src; the result is stored at *dest
 *
 * Note: If you want to compress 1 byte, then set srclen to 1, not to 0!
 *
 * Only up to 255 bytes can be compressed, but you have to allocate
 * 256 bytes at *dest!
 *
 * Return value: Length of the compressed data, i.e. of the data
 * stored at *dest, or 0 if the parameters are invalid
 */
unsigned short int comp_stat_huff(char *src, unsigned short int srclen,
                                  char *dest)
{
  unsigned short int destlen=1;
  unsigned short int bytes_left;
  unsigned char bitmask=0x01;
  unsigned short int bitmask16=0x0001;
  unsigned short int codeend=0x0000;
  unsigned short huffcode=0x0000;

  if(srclen==0 || srclen>255)
    return(0);

  *dest=(char)(srclen-1); /* First byte = length of uncompressed data - 1 */

  src--; /* Will be incremented again later */

  bytes_left=srclen+1;

  for(;;)
  {
    bitmask16>>=1;
    if(bitmask16==codeend)
    {
      if(--bytes_left==0)
        break;
      src++;
      huffcode=huff_comp_table[(unsigned char)*src].code;
      codeend=0x8000>>huff_comp_table[(unsigned char)*src].len;
      bitmask16=0x8000;
    }

    bitmask>>=1;
    if(bitmask==0)
    {
      destlen++;
      if(destlen>=srclen)
      {
        dest-=(destlen-2);
        src-=(srclen-bytes_left);
        dest[0]=255;
        memcpy(dest+1,src,srclen);
        return(srclen+1);
      }
      dest++;
      *dest=0;
      bitmask=0x80;
    }

    if(huffcode&bitmask16)
      *dest|=bitmask;
  }

  return(destlen);
}

/* Decompresses a frame, which is stored at *src and srclen bytes long; the
 * result is stored at *dest
 *
 * Note: If the compressed frame is 1 byte long, then set srclen to 1,
 *       not to 0!
 *
 * The compressed frame may be up to 256 bytes long, but due to the
 * limitations of this compression method the uncompressed frame the
 * will never become longer than 255 bytes
 *
 * Return value: Length of the uncompressed data, i.e. of the data
 * stored at *dest; 0 if the parameters are invalid or if the decompression
 * failed (this happens if the frame was not compressed)
 */
unsigned short int decomp_stat_huff(char *src, unsigned short int srclen,
                                    char *dest)
{
  unsigned short int destlen;
  unsigned short int bytes_left;
  unsigned char bitmask;
  signed short int treepos=0;

  if(srclen<2 || srclen>256)
    return(0);

  destlen=(unsigned short int)((unsigned char)(*src)+1);

  if(destlen==256)
  {
    /* frame is not really compressed */
    memcpy(dest, src+1, srclen-1);
    return(srclen-1);
  }

  bytes_left=destlen;

  do
  {
    src++;
    if(--srclen==0)   /* Decompression not complete, but      */
      return(0);      /* no data left => no compressed frame! */

    for(bitmask=0x80;bitmask>0;bitmask>>=1)
    {
      if(*src&bitmask)
        treepos=huff_decomp_tree[treepos].node2;
      else
        treepos=huff_decomp_tree[treepos].node1;

      /* If this happens, we have found a bit-sequence, which is
         not used (101010101100010 or 100111111110010) */
      if(treepos==-1)
        return(0);

      if(huff_decomp_tree[treepos].node2==0)
      {
        *(dest++)=(char)huff_decomp_tree[treepos].node1;
        if(--bytes_left==0)
          break; /* Decompression complete */
        treepos=0;
      }
    }

  }
  while(bytes_left);

  if(srclen>1)   /* Decompression complete, but some       */
    return(0);   /* bytes are left => no compressed frame! */

  /* Now check, if all remaining bits are set to 0; if not, it's
     no compressed frame (well, hopefully...) */
  while(bitmask>0)
  {
    bitmask>>=1;
    if(*src&bitmask)
      return(0);
  }

  return(destlen);
}
