/*
 * ckraid.c : Utility for the Linux Multiple Devices driver
 *            Copyright (C) 1997 Ingo Molnar, Miguel de Icaza, Gadi Oxman,
 *            Bradley Ward Allen
 *
 * This utility checks Linux MD RAID 1/4/5 arrays.
 *
 * This source is covered by the GNU GPL, the same as all Linux kernel
 * sources.
 */

#include "common.h"
#include "parser.h"
#include "raid_sb.h"

md_cfg_entry_t *cfg_head = NULL, *cfg = NULL;
int force_flag = 1;
static int do_fix_flag = 0, do_quiet_flag = 0, do_continue = 0;
static char source_disk[MAX_LINE_LENGTH];
static unsigned long skip=0, suggest_failed_mask=0, errcount=0;
static time_t begin_t;
static __u32 blocks, current = 0;

static progress() {
	time_t left_t, elapsed_t;

	if((elapsed_t=(time(NULL)-begin_t))) {
		left_t=(time_t) ( (float) (elapsed_t) * ( (float) ( (float) blocks/ (float) current) - (float) 1 ) );
		fprintf(stderr,"block %ld=0x%lX (%2d%% done; ~%d:%02d left %d:%02d elapsed) ",
			current, current,
			(current / (blocks / 100)),
			(int) left_t/60, (int) left_t%60,
			(int) elapsed_t/60, (int) elapsed_t%60);
		fflush(stderr);
	}
}

static void xor_block(char *dst, char *src, int size)
{
	int *dstp = (int *) dst, *srcp = (int *) src, words = size / sizeof(int), i;
	for (i = 0; i < words; i++)
		*dstp++ ^= *srcp++;
}

static inline int fixrow5(md_cfg_entry_t *p, md_superblock_t *sb, int fd[MD_SB_DISKS], int current, int failed_disk, char buffer[MD_SB_DISKS][MD_BLK_SIZ], char zero_buffer[MD_BLK_SIZ])
{
	char test_buffer[MD_BLK_SIZ];
	int i;

	memset(test_buffer, 0, MD_BLK_SIZ);
	for (i = 0; i < sb->nr_disks; i++)
		xor_block(test_buffer, buffer[i], MD_BLK_SIZ);
	if (!memcmp(zero_buffer, test_buffer, MD_BLK_SIZ))
		return 0;
	progress();
	memset(test_buffer, 0, MD_BLK_SIZ);
	if (!do_quiet_flag) fprintf(stderr, "row #%ld=0x%lX incorrect\n", current, current);
	if (!do_fix_flag || failed_disk == -1) {
		if (!do_quiet_flag) fprintf( stderr, " ... not synced.\n");
		if(do_fix_flag && failed_disk==-1) {
			fprintf(stderr, "no failed disk known or specified so cannot fix\n");
		}
		return 0;
	}
	if (!do_quiet_flag) fprintf(stderr, " ... synced.\n");
	for (i = 0; i < sb->nr_disks; i++) {
		if (i == failed_disk)
			continue;
		xor_block(test_buffer, buffer[i], MD_BLK_SIZ);
	}
	lseek(fd[failed_disk],-MD_BLK_SIZ,SEEK_CUR);
	if (write(fd[failed_disk], test_buffer, MD_BLK_SIZ) != MD_BLK_SIZ) {
		fprintf(stderr, "write error on device file %s, block #%ld=0x%lX\n",
			p->device_name[failed_disk], current,current);
		return 1;
	}
	return 0;
}

static inline int fixrow1(md_cfg_entry_t *p, md_superblock_t *sb, int fd[MD_SB_DISKS], int current, int source, char buffer[MD_SB_DISKS][MD_BLK_SIZ])
{
	int i;

	for (i = 0; i < sb->nr_disks; i++) {
		if (i == source)
			continue;
		if (!memcmp(buffer[source], buffer[i], MD_BLK_SIZ))
			continue;
		progress();
		if (!do_quiet_flag) fprintf(stderr, "block #%ld=0x%lX on %s differs", current,current, p->device_name[i]);
		if (!do_fix_flag) {
			if (!do_quiet_flag) fprintf( stderr, " ... not synced.\n");
			continue;
		}
		if (!do_quiet_flag) fprintf( stderr, " ... synced.\n");
		memcpy(buffer[i], buffer[source],MD_BLK_SIZ);
		lseek(fd[i],-MD_BLK_SIZ,SEEK_CUR);
		if (write(fd[i], buffer[i], MD_BLK_SIZ) != MD_BLK_SIZ) {
			fprintf(stderr, "write error on device file %s, block #%ld=0x%lX\n", p->device_name[i], current,current);
			return 1;
		}
	}
	return 0;
}

static int ckraid (md_cfg_entry_t *p)
{
	md_superblock_t *sb = &p->sb, *sb_old = &p->sb_old;
	int i, fd[MD_SB_DISKS], source = -1, failed_disk = -1, nr_failed = 0,
	exit_status=0;
	md_descriptor_t *descriptor;
	char buffer[MD_SB_DISKS][MD_BLK_SIZ];
	char test_buffer[MD_BLK_SIZ], zero_buffer[MD_BLK_SIZ];
	unsigned char read_error;

	blocks = sb->size;
	time(&begin_t);
	errcount=0;
	printf("checking raid level %d set %s\n", sb->level, p->md_name);

	for (i = 0; i < sb->nr_disks; i++)
		fd[i] = -1;
	memset(zero_buffer, 0, MD_BLK_SIZ);
	print_sb(&p->sb_old);
	for (i = 0; i < sb->nr_disks; i++) {
		unsigned char this_failed=0, do_source_thingie=0;
		if ((fd[i] = open(p->device_name[i], O_RDWR)) == -1) {
			fprintf(stderr, "couldn't open device file %s\n", p->device_name[i]);
			exit_status=1; goto abort;
		}
		descriptor = &p->sb_old.disks[i];
		if (p->sb_present[i] && (!(descriptor->state & (1 << MD_FAULTY_DEVICE))) &&
		    (descriptor->state & (1 << MD_ACTIVE_DEVICE)) &&
		    (descriptor->state & (1 << MD_SYNC_DEVICE))) {
			++do_source_thingie;
		} else {
			++this_failed;
			fprintf(stderr, "failed device %s\n", p->device_name[i]);
		}
		if((1<<i)&suggest_failed_mask) {
			++this_failed;
			fprintf(stderr, "suggested failed device %s, disk %d\n", p->device_name[i], i);
		}
		if(this_failed) {
			failed_disk = i;
			nr_failed++;
		} else if (do_source_thingie) {
		    	if (source == -1)
				source = i;
		}
		if (strcmp(p->device_name[i], source_disk) == 0)
			source = i;
	}
	printf("array size: %ldkB=0x%lX\n", blocks,blocks*MD_BLK_SIZ);
	if (sb->level == 4 || sb->level == 5) {
		if (nr_failed > 1) {
			fprintf(stderr, "Unrecoverable raid level %d set (%d failed disks)\n", sb->level, nr_failed);
			exit_status=2; goto abort;
		}
		if (nr_failed == 1)
			fprintf(stderr, "Disk %d (%s) not in sync with the raid set\n", failed_disk, p->device_name[failed_disk]);
	} else
		printf("source disk %d (%s)\n", source, p->device_name[source]);

	if(skip) {
		fprintf(stderr, "skipping %lu=0x%lX blocks first ...\n", skip,skip);
		for (i = 0; i < sb->nr_disks; i++) {
			if(-1==lseek(fd[i], (off_t)skip*MD_BLK_SIZ, SEEK_SET)) {
				perror("executing skip option");
				fprintf(stderr, "on disk %d\n", i);
				exit_status=7; goto abort;
			}
		}
	}
	for (current = skip; current < blocks; current++) {
		static time_t last_status_t=0;
		time_t now_t;

		read_error=0;
		if (!(current%0x100)) {
			time(&now_t);
			if(now_t>last_status_t) {
				progress(); putc('\r', stderr);
				last_status_t=now_t;
			}
		}
		/*
		 * Pass 1: read all blocks in a row
		 */
		for (i = 0; i < sb->nr_disks; i++) {
			static readamount;
			if ((readamount=read(fd[i], buffer[i], MD_BLK_SIZ)) != MD_BLK_SIZ) {
				fprintf(stderr, "read error on device file %s, block #%ld=0x%lX\n", p->device_name[i], current,current);
				if (!do_continue) {
					exit_status=3;
					goto abort;
				} else {
					static off_t seek_status;
					++errcount;
/*					if(readamount<0) readamount=0;
					if(-1==(seek_status=lseek(fd[i], MD_BLK_SIZ-readamount, SEEK_CUR))) {*/
					if(-1==(seek_status=lseek(fd[i], (off_t)(current+1)*MD_BLK_SIZ, SEEK_SET))) {
						perror("ckraid: recover from block read error seek");
						exit_status=6;
						goto abort;
					}
					++read_error;
					fprintf(stderr, "debug: continue: seek is now %ld=0x%lX (block %ld=0x%lX)\n", seek_status, seek_status, seek_status/MD_BLK_SIZ, seek_status/MD_BLK_SIZ);
					fprintf(stderr, "continuing ...\n");
				}
			}
		}
		if(read_error) {
			fprintf(stderr, "skipping block %ld=0x%lX because of read error\n", current, current);
		} else {
			if (sb->level == 5 || sb->level == 4) {
				if (fixrow5(p, sb, fd, current, failed_disk, buffer, zero_buffer)) {
					if (!do_continue) {
						exit_status=4;
						goto abort;
					} else {
						++errcount;
						fprintf(stderr, "continuing with next row\n");
					}
				}
			}
			if (sb->level == 1) {
				if (fixrow1(p, sb, fd, current, source, buffer)) {
					if (!do_continue) {
						exit_status=5;
						goto abort;
					} else {
						++errcount;
						fprintf(stderr, "continuing with next row\n");
					}
				}
			}
		}
	}
abort:
	for (i = 0; i < sb->nr_disks; i++)
		if (fd[i] != -1)
			close(fd[i]);
	if(errcount)
		fprintf(stderr, "ckraid: Error count: %lu\n", errcount);
	else
		fprintf(stderr, "ckraid: No errors\n");
	return exit_status;
}

int parse_switches(int argc, char *argv[])
{
	int i, name_index = 0;

	*source_disk = 0;
	for (i = 1; i < argc; i++) {
		if(strncmp(argv[i], "--h", 3) == 0 ||
		   strncmp(argv[i], "-h", 2) == 0) {
			return 0;	/* causes error which prints usage */
		} else if(strcmp(argv[i], "--fix") == 0) {
			do_fix_flag = 1;
			continue;
		} else if (strcmp(argv[i], "--quiet") == 0) {
			do_quiet_flag = 1;
			continue;
		} else if (strcmp(argv[i], "--force-continue") == 0) {
			do_continue = 1;
			continue;
		} else if (strcmp(argv[i], "--force-source") == 0) {
			strncpy(source_disk, argv[++i], MAX_LINE_LENGTH);
			continue;
		} else if (strcmp(argv[i], "--force-skip-blocks") == 0) {
			skip=strtoul(argv[++i],NULL,0);
			continue;
		} else if (strcmp(argv[i], "--suggest-failed-disk-mask") == 0) {
			suggest_failed_mask=strtoul(argv[++i],NULL,0);
			continue;
		} else {
			if (name_index)
				return 0;
			name_index = i;
		}
	}
	return name_index;
}

void usage() {
	fprintf(stderr, "usage: ckraid [--fix] [--quiet] [--force-continue] [--force-source device] [--force-skip-blocks blockcount] [--suggest-failed-disk-mask mask] conf_file\n");
}

int main (int argc, char *argv[])
{
	FILE *fp = NULL;
	int name_index;
	md_cfg_entry_t *p;

	name_index = parse_switches(argc, argv);
	if (!name_index)
		goto print_usage;
	fp = fopen(argv[name_index], "r");
	if (fp == NULL) {
		fprintf(stderr, "Couldn't open %s -- %s\n", argv[name_index], strerror(errno));
		goto abort;
	}
	srand((unsigned int) time(NULL));
	printf("ckraid version %d.%d.%d\n", MKRAID_MAJOR_VERSION, MKRAID_MINOR_VERSION, MKRAID_PATCHLEVEL_VERSION);
	if (parse_config(fp))
		goto abort;
	if (analyze_sb())
		goto abort;
	if (read_sb())
		goto abort;
	for (p = cfg_head; p; p = p->next) {
		if (check_active(p))
			goto abort;
		if (ckraid(p))
			goto abort;
	}
	if (do_fix_flag) {
		if (write_sb(1))
			goto abort;
	} else
		fprintf(stderr, "not fixing RAID superblock\n");
	printf("ckraid: completed\n");
	fclose(fp);
	return 0;
print_usage:
	usage();
abort:
	fprintf(stderr, "ckraid: aborted\n");
	if (fp)
		fclose(fp);
	return 1;
}
