/* 
 *  Copyright (C) 1999-2000 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */

#include <qstringlist.h>
#include <qfile.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <kapp.h>

#include "misc.h"
#include "ldapsite.h"


LdapSite::LdapSite()
{
    inputs.setAutoDelete(true);
}


LdapSite::~LdapSite()
{}


QString LdapSite::makeQueryUrl(const QString &query)
{
    QString url = action;

    QListIterator<LdapInputTag> it(inputs);
    for (; it.current(); ++it)
        {
            url += it.current()->name;
            url += "=";
            if (it.current()->user)
                url += encodeString(query);
            else 
                url += it.current()->value;
        }
    url += "??sub";
    //    qDebug("url: %s", url.latin1());
    //    url = "file:///home/bernd/kdecvs2.0/kdenetwork/libsearch/ldap/bigfoot.out";
    //    url = "file:///home/bernd/kdecvs2.0/kdenetwork/libsearch/ldap/hu.out";
    //    url = "file:///home/bernd/blubb.ldif";
    return url;
}


LdapSite *LdapSiteFactory::create(const QString &filename)
{
    qDebug("create %s", filename.latin1());
    QFile file(filename);
    if (!file.open(IO_ReadOnly))
        return 0;

    LdapSite *site = newLdapSite();
    QDomDocument doc(&file);
    qDebug("Constructed doc");
    QDomElement docel = doc.documentElement();
    site->name = docel.attribute("name");
    qDebug("Name: %s", site->name.latin1());
    site->action = docel.attribute("action");
    qDebug("Action: %s", site->action.latin1());

    QDomElement childel = docel.firstChild().toElement();
    while (!childel.isNull())
        {
            qDebug("dom tag: %s", childel.tagName().latin1()); 
            if (childel.tagName() == "input")
                {
                    qDebug("Input");
                    input(site, childel);
                }
            else if (childel.tagName() == "ldapinterpret")
                {
                    qDebug("Interpret");
                    interpret(site, childel);
                }
            childel = childel.nextSibling().toElement();
        }
    return site;
}


LdapSite *LdapSiteFactory::newLdapSite()
{
    return new LdapSite;
}


void LdapSiteFactory::input(LdapSite *site, const QDomElement &el)
{
    LdapInputTag *tag = new LdapInputTag();
    tag->name = el.attribute("name");
    tag->value = el.attribute("value");
    tag->user = el.attribute("user") == "true" || el.attribute("user") == "TRUE";
    site->inputs.append(tag);
    qDebug("Input name: %s value: %s user: %i",
           tag->name.latin1(), tag->value.latin1(), (int)tag->user);
}


void LdapSiteFactory::interpret(LdapSite *, const QDomElement &)
{
}


LdapSiteManager::LdapSiteManager(const QString &sitetype, LdapSiteFactory &factory)
{
    QStringList files = findNewestResources("data", "libksearch/" + sitetype);
    
    dict.setAutoDelete(true);
    for (QStringList::ConstIterator it = files.begin(); it != files.end(); ++it)
        {
            LdapSite *site = factory.create(*it);
            if (site)
                dict.insert(site->name, site);
        }
}
