/* 
 *  Copyright (C) 1999-2000 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */


#include <stdlib.h>
#include <stdio.h>
#include <qregexp.h>

#include "misc.h"
#include "ldapsite.h"
#include "ldapsearchjob.h"


static QString decodeBase64(const QString &str)
{
    return str;
}


LdapSearchJob::LdapSearchJob(LdapSite *site, const QString &searchtext)
    : TransferJob(KURL(site->makeQueryUrl(searchtext)), 'C',
                       makeArgs(site, searchtext), QByteArray(), false)
{
    m_site = site;
    
    connect( this, SIGNAL(data(KIO::Job*, const QByteArray &)),
             this, SLOT(gotData(KIO::Job*, const QByteArray &)) );
}


LdapSearchJob::~LdapSearchJob()
{}

    
QByteArray LdapSearchJob::makeArgs(LdapSite *site, const QString &searchtext)
{
    QByteArray a;
    QDataStream stream(a, IO_WriteOnly);
    KURL url(site->makeQueryUrl(searchtext));
    stream << url.path() << url.query() << Q_INT8(0);
    return a;
}


void LdapSearchJob::slotFinished()
{
    qDebug("Finished");
    dumpEntry();
    TransferJob::slotFinished();
}


void LdapSearchJob::gotData(KIO::Job *, const QByteArray &data)
{
    QCString cdata(data);
    m_buf += cdata;
    for (;;)
        {
            int pos = m_buf.find('\n');
            if (pos == -1)
                break;
            QString line = m_buf.left(pos);
            m_buf.remove(0, pos+1);

            if (!line.isEmpty())
                {
                    QChar c = line[0];
                    // Comment:
                    if (c == '#')
                        continue;
                    // Continuation line:
                    if (c == ' ' || c == '\t' && !entry.isEmpty())
                        {
                            line.remove(0, 1);
                            entry.last().append(line);
                        }
                    // Normal lines:
                    entry.append(line);
                }
            else
                dumpEntry();
        }
}


void LdapSearchJob::dumpEntry()
{
    if (entry.isEmpty())
        return;
    LdapItem item;
    QStringList::ConstIterator it;
    for (it = entry.begin(); it != entry.end(); ++it)
        {
            int pos = (*it).find(':');
            if (pos == -1)
                continue;
            QString key = (*it).left(pos);
            QString value = (*it).mid(pos+1);
            bool asbase64 = false;
            if (value[0] == ':')
                asbase64 = true;
            else if (value[0] != ' ')
                continue;
            for (pos = 0; pos < (int)value.length() && value[pos] == ' '; ++pos);
            value.remove(0, pos);
            if (asbase64)
                value = decodeBase64(value);
            LdapAttribute x;
            x.key = key;
            x.value = value;
            item.append(x);
        }
    parseItem(item);
    entry.clear();
}
