/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <stdio.h>
#include <iostream.h>

#include "resbaseconf.h"


ResourceBaseConfig::ResourceBaseConfig()
{
  setAutoDelete(true);
  clear();
}


/*
 * Public methods
 *****************/

/*
 * Try to build a list of alternatives for the given 
 * resource setting
 ****************************************************/
ResourceList  *ResourceBaseConfig::alternativeResources(const Resource *setting)
{
  debug("Find alternatives to %s %lx", setting->resourceLowerName().ascii(),
  				       setting->resourceValue());
                                      
  for (ResourceList *reslist = first(); reslist; reslist = next())
  {
    printf("Traversing list %i [%s", at(), reslist->first()->resourceUpperName().local8Bit());
 
    for (Resource *res = reslist->first(); res; res = reslist->next())
      printf(" %lx", res->resourceValue());
 
    printf("]\n");
 
    if (reslist->find(setting) >= 0) return reslist;
  }
  
  debug("No alternatives found");
  return 0L;
} 
 

/*
 * Check whether the given set of resources appears in this
 * resource base configuration
 ***********************************************************/
bool  ResourceBaseConfig::contains(QList<Resource>& setlist)
{
  QList<ResourceList>  matchlist;
  ResourceList         *reslist;
  
  matchlist.clear();
  
  for (Resource *setting = setlist.first(); setting; setting = setlist.next())
  {  
    for (reslist = first(); reslist; reslist = next())
    {
      if (matchlist.findRef(reslist) < 0)	// Search this resource list once only
      {
        if (reslist->find(setting) >= 0)
        {
          matchlist.append(reslist);    	// Next time ignore this resource list
          break;
        }
      }
    }
    
    if (!reslist) return false;	 // Setting not found in any resource list of this base configuration
  }
  
  return true;
}


/* 
 * Dump this resource base configuration to the console
 * (debugging purposes)
 *******************************************************/
void  ResourceBaseConfig::dump()
{
  for (ResourceList *reslist = first(); reslist; reslist = next())
  {
    cout << "Resource list " << at() << endl << endl;
    
    for (Resource *res = reslist->first(); res; res = reslist->next())
      printf("%s\t = 0x%lx\n", res->resourceUpperName().local8Bit(), res->resourceValue());
  
    cout << endl;
  }
}

/*
 * Build a list of resources that fits the actual
 * settings best, based on this configuration
 *************************************************/
QList<Resource>  ResourceBaseConfig::nearestMatch(QList<Resource>& setlist)
{
  QList<Resource>      matchlist;
  ResourceList         *reslist;
  QList<ResourceList>  seenlist;
  
  // Search for resources not covered by the list of actual settings
  
  for (Resource *setting = setlist.first(); setting; setting = setlist.next())
  {
    for (reslist = first(); reslist; reslist = next())
    {
      if (reslist->find(setting) >= 0)
      {
        seenlist.append(reslist);
        matchlist.append(setting);
        break;
      }
    }
  }
  
  // Take the first possible value for resources not covered by the actual settings
  
  for (reslist = first(); reslist; reslist = next())
    if (seenlist.findRef(reslist) < 0) matchlist.append(reslist->first());
  
  return matchlist;
}


/*
 * Determine range for the given resource; this comes handy
 * when range information is missing from scanning due to
 * non-existent entries in /proc/ioports (no driver loaded)
 ***********************************************************/
ulong  ResourceBaseConfig::resourceRange(const Resource *setting)
{
  ResourceList  *reslist = alternativeResources(setting);
  
  return (reslist) ? reslist->first()->resourceRange() : 0L;
}

