/*********************************************************************
 *
 * This is based on code created by Peter Harvey,
 * (pharvey@codebydesign.com).
 *
 * Modified and extended by Nick Gorham
 * (nick@easysoft.com).
 *
 * Any bugs or problems should be considered the fault of Nick and not
 * Peter.
 *
 * copyright (c) 1999 Nick Gorham
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **********************************************************************
 *
 * $Id: SQLGetConnectAttr.c,v 1.2 1999/12/10 01:50:34 harvey Exp $
 *
 * $Log: SQLGetConnectAttr.c,v $
 * Revision 1.2  1999/12/10 01:50:34  harvey
 * Updated with current sources from unixODBC cvs.
 *
 * Revision 1.8  1999/11/13 23:40:59  ngorham
 *
 * Alter the way DM logging works
 * Upgrade the Postgres driver to 6.4.6
 *
 * Revision 1.7  1999/10/24 23:54:18  ngorham
 *
 * First part of the changes to the error reporting
 *
 * Revision 1.6  1999/09/21 22:34:24  ngorham
 *
 * Improve performance by removing unneeded logging calls when logging is
 * disabled
 *
 * Revision 1.5  1999/09/19 22:24:33  ngorham
 *
 * Added support for the cursor library
 *
 * Revision 1.4  1999/07/10 21:10:16  ngorham
 *
 * Adjust error sqlstate from driver manager, depending on requested
 * version (ODBC2/3)
 *
 * Revision 1.3  1999/07/04 21:05:07  ngorham
 *
 * Add LGPL Headers to code
 *
 * Revision 1.2  1999/06/30 23:56:55  ngorham
 *
 * Add initial thread safety code
 *
 * Revision 1.1.1.1  1999/05/29 13:41:07  sShandyb
 * first go at it
 *
 * Revision 1.1.1.1  1999/05/27 18:23:17  pharvey
 * Imported sources
 *
 * Revision 1.2  1999/04/30 16:22:47  nick
 * Another checkpoint
 *
 * Revision 1.1  1999/04/25 23:06:11  nick
 * Initial revision
 *
 *
 **********************************************************************/

#include "drivermanager.h"

static char const rcsid[]= "$RCSfile: SQLGetConnectAttr.c,v $ $Revision: 1.2 $";

SQLRETURN SQLGetConnectAttr( SQLHDBC connection_handle,
           SQLINTEGER attribute,
           SQLPOINTER value,
           SQLINTEGER buffer_length,
           SQLINTEGER *string_length )
{
    DMHDBC connection = (DMHDBC)connection_handle;
    int type = 0;
    char *ptr;
    SQLCHAR s0[ 20 ], s1[ 100 ];

    /*
     * check connection
     */

    if ( !__validate_dbc( connection ))
    {
        return SQL_INVALID_HANDLE;
    }

    function_entry( connection );

    if ( connection -> log_handle )
    {
        sprintf( connection -> msg, "\n\t\tEntry:\
            \n\t\t\tConnection = %p\
            \n\t\t\tAttribute = %s\
            \n\t\t\tValue = %p\
            \n\t\t\tBuffer Length = %d\
            \n\t\t\tStrLen = %p",
                connection,
                __con_attr_as_string( s1, attribute ),
                value, 
                (int)buffer_length,
                (void*)string_length );

        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                connection -> msg );
    }

    thread_protect( SQL_HANDLE_DBC, connection );

    if ( connection -> state == STATE_C3 )
    {
        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: HY010" );

        __post_internal_error( &connection -> error,
                ERROR_HY010, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    if ( connection -> state == STATE_C2 )
    {
        switch ( attribute )
        {
          case SQL_ATTR_ACCESS_MODE:
          case SQL_ATTR_AUTOCOMMIT:
          case SQL_ATTR_LOGIN_TIMEOUT:
          case SQL_ATTR_ODBC_CURSORS:
          case SQL_ATTR_TRACE:
          case SQL_ATTR_TRACEFILE:
            break;

          default:
            dm_log_write( connection -> log_handle, 
                    __get_pid( s0 ), 
                    __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: 08003" );

            __post_internal_error( &connection -> error,
                    ERROR_08003, NULL,
                    connection -> environment -> requested_version );

            thread_release( SQL_HANDLE_DBC, connection );

            return function_return( connection, SQL_ERROR );
        }
    }

    switch ( attribute )
    {
      case SQL_ATTR_ACCESS_MODE:
        /*
         * if connected, call the driver
         */
        if ( connection -> state != STATE_C2 )
        {
            type = 0;
        }
        else
        {
            *((SQLINTEGER*)value) = connection -> access_mode;
            type = 1;
        }
        break;

      case SQL_ATTR_AUTOCOMMIT:
        /*
         * if connected, call the driver
         */
        if ( connection -> state != STATE_C2 )
        {
            type = 0;
        }
        else
        {
            *((SQLINTEGER*)value) = connection -> autocommit_mode;
            type = 1;
        }
        break;

      case SQL_ATTR_LOGIN_TIMEOUT:
        /*
         * if connected, call the driver
         */
        if ( connection -> state != STATE_C2 )
        {
            type = 0;
        }
        else
        {
            *((SQLINTEGER*)value) = connection -> login_timeout;
            type = 1;
        }
        break;

      case SQL_ATTR_ODBC_CURSORS:
        *((SQLINTEGER*)value) = connection -> cursors;
        type = 1;
        break;

      case SQL_ATTR_TRACE:
        *((SQLINTEGER*)value) = connection -> trace;
        type = 1;
        break;

      case SQL_ATTR_TRACEFILE:
        ptr = connection -> tracefile;
        type = 2;
        break;

      default:
        break;
    }

    /*
     * if type has been set we have already set the value,
     * so just return
     */
    if ( type )
    {
        SQLRETURN ret = SQL_SUCCESS;
        if ( type == 1 )
        {
            if ( string_length )
            {
                *string_length = sizeof( SQLUINTEGER );
            }
        }
        else
        {
            if ( string_length )
            {
                *string_length = strlen( ptr );
            }
            if ( value )
            {
                if ( buffer_length > strlen( ptr ) + 1 )
                {
                    strcpy( value, ptr );
                }
                else
                {
                    memcpy( value, ptr, buffer_length - 1 );
                    ((char*)value)[ buffer_length - 1 ] = '\0';
                    ret = SQL_SUCCESS_WITH_INFO;
                }
            }
        }

        sprintf( connection -> msg, 
                "\n\t\tExit:[%s]",
                    __get_return_status( ret ));

        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                connection -> msg );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, ret );
    }
    else
    {
        SQLRETURN ret;

        /*
         * call the driver
         */
        if ( !CHECK_SQLGETCONNECTATTR( connection ))
        {
            if ( CHECK_SQLGETCONNECTOPTION( connection ))
            {
                ret = SQLGETCONNECTOPTION( connection,
                    connection -> driver_dbc,
                    attribute,
                    value );
            }
            else
            {
                dm_log_write( connection -> log_handle, 
                        __get_pid( s0 ), 
                        __FILE__, 
                        __LINE__, 
                        LOG_INFO, 
                        LOG_INFO, 
                        "Error: IM001" );

                __post_internal_error( &connection -> error,
                        ERROR_IM001, NULL,
                        connection -> environment -> requested_version );

                thread_release( SQL_HANDLE_DBC, connection );

                return function_return( connection, SQL_ERROR );
            }
        }
        else
        {
            ret = SQLGETCONNECTATTR( connection,
                connection -> driver_dbc,
                attribute,
                value,
                buffer_length,
                string_length );
        }

        if ( connection -> log_handle )
        {
            sprintf( connection -> msg, 
                    "\n\t\tExit:[%s]",
                        __get_return_status( ret ));

            dm_log_write( connection -> log_handle, 
                    __get_pid( s0 ), 
                    __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    connection -> msg );
        }

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, ret );
    }
}
