#include "simple_view.h"
#include "konq_progressproxy.h"

#include <klocale.h>
#include <kio_job.h>
#include <kio_cache.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <qtextstream.h>

/*
 * First lets do the factory class. This factory is used by KParts in order
 * to transparently allocate generic pointers to different components.
 * This will be almost identical in all parts, with just the type of class
 * you are allocating being different.
 */

/*
 * KParts uses this to allocate a pointer to the factory for your class.
 * Simply return a pointer to whatever class you named your factory here.
 */
extern "C"
{
    void* init_libksimpleview()
    {
        return new KSimpleFactory;
    }
};

/*
 * This is the pointer that gets used in the static KLibFactory::instance()
 * method. Remember this is a shared library so we need to create our own
 * instance.
 */
KInstance *KSimpleFactory::s_instance = 0L;

/*
 * You won't usually have to do anything when creating a factory.
 */
KSimpleFactory::KSimpleFactory()
{
}

/*
 * Delete the instance for the factory.
 */
KSimpleFactory::~KSimpleFactory()
{
    if ( s_instance )
        delete s_instance;
    s_instance = 0;
}

/*
 * The factory method that KParts calls when allocating your view class.
 * Make sure to emit the objectCreated signal here.
 */
QObject* KSimpleFactory::create(QObject *, const char *, const char *,
                                const QStringList & )
{
    QObject *obj = new KSimpleKonqView;
    emit objectCreated(obj);
    return obj;
}

/*
 * The static method used to return the factory instance.
 */
KInstance *KSimpleFactory::instance()
{
    if (!s_instance)
        s_instance = new KInstance("ksimpleview");
    return s_instance;
}

/*
 * ------------------------------------------------------------
 * Now lets do the actual browser view part class.
 * ------------------------------------------------------------
 */

/*
 * Do any initalization required in the constructor.
 */
KSimpleKonqView::KSimpleKonqView()
{
    /*
     * Clear the ID used by KIOJob.
     */
    jobId = 0;

    /*
     * This will be the main widget for the view.
     */
    list = new QListBox(this);

    /*
     * We have one action in this example, a "clear list" toolbar icon.
     */
    clearAction = new KAction(i18n("Clear List"),
                              QIconSet(BarIcon("exit",
                              KSimpleFactory::instance())),
                              0, this, SLOT(slotClear()), this);

    /*
     * Append the action to the list of actions the browser class maintains
     * for this view.
     */
    actions()->append(BrowserView::ViewAction(clearAction,
                                              BrowserView::MenuEdit |
                                              BrowserView::ToolBar));
}

/*
 * This is called when the clearAction toolbar action is activated.
 */
void KSimpleKonqView::slotClear()
{
    list->clear();
}

/*
 * Reset and clear anything we did before.
 */
KSimpleKonqView::~KSimpleKonqView()
{
    stop();
}

/*
 * openURL is called when the user tries to open a URL (either local or remote)
 * associated with this type of view. This is where the data transfer should
 * start.
 */
void KSimpleKonqView::openURL(const QString &url, bool reload,
                              int, int)
{
    /*
     * Make sure we reset any previous data.
     */
    stop();

    /*
     * Create a new KIOJob to process the local or remote file.
     */
    KIOCachedJob *job = new KIOCachedJob;

    /*
     * We use the Konq progess bar to show our status.
     */
    job->setGUImode(KIOJob::NONE);

    /*
     * Connect to the various KIOJob signals that we use.
     */
    connect(job, SIGNAL(sigFinished(int)),
            this, SLOT(slotFinished(int)));
    connect(job, SIGNAL(sigRedirection(int, const char *)),
            this, SLOT(slotRedirection(int, const char *)));
    connect(job, SIGNAL(sigData(int, const char *, int)),
            this, SLOT(slotData(int, const char *, int)));
    connect(job, SIGNAL(sigError(int, int, const char *)),
            this, SLOT(slotError(int, int, const char *)));

    /*
     * KonqProgressProxy is a proxy class to update the statusbar in Konq.
     * One of the constructors takes a KIOJob as an argument. You don't have
     * to do anything else as it will delete itself when needed.
     */
     (void)new KonqProgressProxy(this, job);

     /*
      * Keep the KIOJob id and URL string around so we can do stuff with it
      * later.
      */
     jobId = job->id();
     urlStr = url;
     
     /*
      * Start the KIOJob data transfer from the given URL.
      */
     job->get(url, reload);

     /*
      * Tell Konq we have started loading the document.
      */
     emit started();

}

/*
 * Return whatever URL was specified in openURL() here.
 */
QString KSimpleKonqView::url()
{
    return(urlStr);
}

/*
 * If you require an offset for placement, specify it in these two methods.
 * We don't so return 0.
 */
int KSimpleKonqView::xOffset()
{
    return(0);
}

int KSimpleKonqView::yOffset()
{
    return(0);
}

/*
 * If for any reason we need to stop the data transfer, we call this method.
 * This isn't actually part of the browser API, we just use it for
 * convienece.
 */
void KSimpleKonqView::stop()
{
    if(jobId){
        KIOJob *job = KIOJob::find(jobId);
        if (job)
            job->kill();
        jobId = 0;
    }
    dataBuffer.resize(0);
}

/*
 * This slot gets called when KIOJob has incoming data from a local or remote
 * URL. We place the data into a QByteArray buffer to process when
 * slotFinished is called.
 */
void KSimpleKonqView::slotData(int, const char *data, int len)
{
    /*
     * Append incoming data to the QByteArray buffer.
     */

    /*
     * Get the next free index of the buffer.
     */
    int bufferIndex = dataBuffer.count();
    int dataIndex = 0;

    /*
     * Copy the data.
     */
    dataBuffer.resize(dataBuffer.count()+len);
    for(; dataIndex < len; ++dataIndex, ++bufferIndex)
        dataBuffer[bufferIndex] = data[dataIndex];
}
    

/*
 * This slot gets called when the KIOJob is complete. If you need to do
 * anything special (usually you will), do so here.
 */
void KSimpleKonqView::slotFinished(int)
{
    jobId = 0;

    /*
     * We have buffered the I/O from the URL into this QByteArray. Now lets
     * convert it into a stream and insert it into our QListBox.
     */
    QTextStream stream(dataBuffer, IO_ReadOnly);
    QString dataStr;
    while(!stream.atEnd()){
        dataStr = stream.readLine();
        list->insertItem(dataStr);
    }
    
    /*
     * Make the gear stop spinning :)
     */
    emit completed();
}

/*
 * The URL is redirected. We don't have to do anything special here as KIOJob
 * will take care of this, but should update the location bar.
 */
void KSimpleKonqView::slotRedirection(int, const char *url )
{
    emit setLocationBarURL(QString(url));
}

/*
 * An error occured in the KIOJob data transfer. You can present a dialog or
 * something similar here, we just cancel the operation.
 */
void KSimpleKonqView::slotError(int, int, const char *)
{
  stop();
  emit canceled();
}

/*
 * Since we only have one widget we don't do any real layout management but
 * instead just resize the QListBox to our current size.
 */
void KSimpleKonqView::resizeEvent(QResizeEvent *)
{
    list->setGeometry(0, 0, width(), height());
}

#include "simple_view.moc"

