/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-2000 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "preferences.h"
#include "main.h"
#include "ahuf.h"

#include "global.h"
#include "toolbox.h"
#include "dialog.h"
#include "routerdlg.h"
#include "version.h"

#include <kmsgbox.h>
#include <kwmmapp.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#include <stdlib.h>
#include "main.moc"


KApplication *mykapp;
TopLevel *toplevel;


// aus global.h
s_config *config;
QWidget *focusWidget;
s_boxcheck *boxcheckroot;


/*
 *  TopLevel::TopLevel( QWidget *parent=0, const char *name=0 )
 * Konstruktor des Hauptfensters
 */
TopLevel::TopLevel( const char *name )
             : KTopLevelWidget(name)
{
   setMinimumSize (200, 100);
//   resize(600,350);

   // Lokale Variablen dieser Klasse
   chanlist = NULL;
   currentChannel = NULL;
   qsoWinGeometryFlag = false;
//   position = NULL;

   // Initialisierung der Random-Geschichte
   srand((int)time(NULL));

   init_ax25();

   setCaption(klocale->translate("LinKT - the Linux KDE PR-Terminal"));


   setupMenuBar();
   chanListWidget = new QWidget( this );
   chanListTable = new ChanList( chanListWidget );
   connect( chanListTable, SIGNAL(signalFunctionKey(int)), SLOT(slotCallKey(int)) );
   setMainwinSize();

   router = new Router();

   // Routing-Informationen einlesen
   router->readRoutes();

   setView(chanListWidget);


   // Audio-Server connecten
#ifndef WITHOUT_AUDIO
   if (config->sound->enabled)
      if (KAServer.serverStatus())
      {
         printf("Cannot contact the audio server\n");
         config->sound->enabled = false;
      }
      else
      {
         KAServer.setSignals( false );
      }
#endif
}


TopLevel::~TopLevel()
{
}


void TopLevel::setMainwinSize()
{
   int width = chanListTable->useWidth();
   int height = chanListTable->useHeight();

   chanListWidget->setFixedSize( width, height );
   chanListTable->resize( width, height );

   show();
}


/*
 *  void TopLevel::setupMenuBar()
 * Das Hauptmenue wird aufgebaut und an das KTopLevelWidget angedockt.
 */
void TopLevel::setupMenuBar()
{
   int i;

   pref = new QPopupMenu();

   pref->insertItem( klocale->translate("&Calls"), this, SLOT(slotPrefCalls()));
   pref->insertItem( klocale->translate("&Ports"), this, SLOT(slotPrefPorts()));
   pref->insertItem( klocale->translate("&Screen"), this, SLOT(slotPrefScreen()));
   pref->insertItem( klocale->translate("&Sound"), this, SLOT(slotPrefSound()));
   pref->insertItem( klocale->translate("&Away"), this, SLOT(slotPrefAway()));
   pref->insertItem( klocale->translate("S&hortcuts"), this, SLOT(slotPrefShortcut()));
   pref->insertItem( klocale->translate("Call&Key"), this, SLOT(slotPrefCallKey()));
   pref->insertItem( klocale->translate("&Flags"), this, SLOT(slotPrefFlags()));
   pref->insertItem( klocale->translate("&Directories"), this, SLOT(slotPrefDirs()));
   pref->insertItem( klocale->translate("&Files"), this, SLOT(slotPrefFiles()));


   route = new QPopupMenu ();
   route->insertItem( klocale->translate("&Add"), this, SLOT(slotRouteAdd()));
//   route->insertItem( klocale->translate("&Delete"), this, SLOT(slotRouteDelete()));
   route->insertItem( klocale->translate("&Show List"), this, SLOT(slotRouteShowList()));
//   route->insertItem( klocale->translate("&Show Tree"), this, SLOT(slotRouteShowTree()));


   file = new QPopupMenu ();

   file->insertItem( klocale->translate("&Preferences"), pref );
   file->insertItem( klocale->translate("Away"), this, SLOT(slotAway()), ALT+Key_A);
   file->insertItem( klocale->translate("&Route"), route );
   file->insertSeparator( -1 );
   file->insertItem( klocale->translate("E&xit"), this, SLOT(slotBeenden()), ALT+Key_X);

   channel = new QPopupMenu ();
   i = channel->insertItem( klocale->translate("&Connect"), this, SLOT(slotConnect()), ALT+Key_C);


   help = new QPopupMenu ();
   help->insertItem( klocale->translate("&Index"), this, SLOT(slotHelp()));
   help->insertItem( klocale->translate("&About"), this, SLOT(slotAbout()));



   menubar = new KMenuBar (this);
   menubar->insertItem( klocale->translate("&File"), file );
   menubar->insertItem( klocale->translate("Cha&nnel"), channel );
   menubar->insertSeparator( -1 );
   menubar->insertItem( klocale->translate("&Help"), help );

   setMenu(menubar);
}


/*
 *  void TopLevel::slotBeenden()
 * Wird aufgerufen, wenn LinKT beendet werden soll. (Sicherheitsabfrage)
 */
void TopLevel::slotBeenden()
{
   close();
}


/*
 *  void TopLevel::slotAway()
 * Hier kann man einstellen, ob man da oder weg ist.
 */
void TopLevel::slotAway()
{
   unsigned char old;
   char text[100], text2[100], oldtext[100];
   s_chanlist *tmp;


   AwayDlg *dlg = new AwayDlg(0);

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   if (dlg->exec() == 1)
   {
      // Ok wurde angeklickt
      old = config->away.wo;
      strcpy(oldtext, config->away.text);

      dlg->getInfos(config->away.wo, text);
      config->away.awaytime = time(NULL);

      if (old == config->away.wo)
         if (!(old == 3 && (strcmp(config->away.text, oldtext))))
         {
            delete dlg;
            return;
         }

      tmp = chanlist;
      while (tmp != NULL)
      {
         switch (config->away.wo)
         {
            case 0: strcpy(text, config->away.back); break;
            case 1: strcpy(text, config->away.phone); break;
            case 2: strcpy(text, config->away.bed); break;
            case 3: strcpy(text, config->away.text); break;
         }

         tmp->channel->macroLine(text2, text);

         switch (tmp->channel->userinfo->getType())
         {
            case TYPE_TERMINAL:
                 if (config->away.wo == 0)
                    sprintf(text,"## %s ##\r", text2);
                 else
                    sprintf(text,"## SysOp is away: %s ##\r", text2);
                 tmp->channel->sendString( text );
                 break;
            case TYPE_CONVERS:
                 if (config->away.wo == 0)
                    strcpy(text, "/a\r");
                 else
                    sprintf(text,"/a %s\r", text2);
                 tmp->channel->sendString( text );
                 break;
         }
         tmp = tmp->next;
      }
   }

   delete dlg;
}


//   bool incSSID(char *call)
//
// Die SSID des Rufzeichens wird inkrementiert. Wenn die inkrementierte
// SSID groesser als 15 waehre, kommt eine Fehlermeldung.
bool TopLevel::incSSID(char *call)
{
   char tmp[50];
   int i,len,ssid;


   // SSID abtrennen
   if ((i = POS('-', call)) == -1)
   {
      // Keine SSID vorhanden. SSID = 0
      ssid = 0;
   }
   else
   {
      len = strlen(call)-i-1;
      memcpy(tmp, call+i+1, len);
      tmp[i] = '\0';
      ssid = atoi(tmp);
      call[i] = '\0';
   }

   // SSID inkrementieren und gucken, ob sie > 15 wird.
   ssid++;
   if (ssid > 15)
      return false;

   sprintf(call, "%s-%i", call, ssid);

   return true;
}



/*
 *  void TopLevel::slotConnect()
 * Wird aufgerufen, wenn eine Verbindung nach Aussen aufgebaut werden soll.
 */
void TopLevel::slotConnect()
{
   char digis[500],*port=0,*tmp,*mycall=0, call[500], porttmp[500];
   int i,fd,error,len;
   bool ready=false, changed;

   Connect *dlg = new Connect(0);

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   if (dlg->exec() == 1)
   {
      // Connect aufbauen
      strcpy(digis, dlg->getText());
      port = (char *) strdup(dlg->getPort());
      mycall = (char *) malloc(strlen(dlg->getMycall())+10);
      strcpy(mycall, dlg->getMycall());

      if (mycall[0] == '\0')
      {
         KMsgBox::message( 0,
                           klocale->translate("Connecting"),
                           klocale->translate("Cannot connect - no MyCall given."));
         return;
      }

      if (digis[0] == '\0')
      {
         free(port);
         free(mycall);
         return;
      }

      if (port[0] == '\0')
      {
         KMsgBox::message( 0,
                           klocale->translate("Connecting"),
                           klocale->translate("Cannot connect - no ax.25-port active."));
         return;
      }

      tmp = (char *) malloc( strlen(digis)+1 );

      // Ueberfluessige Leerzeichen weg machen
      OnlyOneSpace(tmp, digis);
      KillSpacesLeft(tmp);
      KillSpacesRight(tmp);

      // Connectpfad in die Liste eintragen
      addConnectList( tmp );

      Gross(tmp);


      // Wenn nur das Rufzeichen (ohne Pfad) uebergeben wurde, wird im
      // Router geguckt, ob ein Pfad bekannt ist.
      strcpy(call, tmp);
      router->getPath( call, changed );

      if (changed)
         if ((i = POS(' ', call)) > -1)
         {
            memcpy(porttmp, call, i);
            porttmp[i] = '\0';
            len = strlen(call)-i-1;
            memmove(call, call+i+1, len);
            call[len] = '\0';

            free(tmp);
            tmp = (char *) strdup(call);
            free(port);
            port = (char *) strdup(porttmp);
         }


      if ((i = POS(' ',tmp)) > -1)
      {
         COPY(digis, tmp, i+1, strlen(tmp)-i-1);
         tmp[i] = '\0';
      }
      else
          digis[0] = '\0';

      Gross(tmp);
      Gross(digis);
      Gross(mycall);

      do
      {

         fd = makeConnect( tmp, digis, mycall, port, &error );

         if (fd == -1)
         {
            switch (error)
            {
               case 1: // invalid parameter
                       KMsgBox::message(0,klocale->translate("AX.25-Error"),klocale->translate("invalid parameter"));
                       ready = true;
                       break;
               case 2: // cannot open socket
                       KMsgBox::message(0,klocale->translate("AX.25-Error"),klocale->translate("cannot open socket"));
                       ready = true;
                       break;
               case 3: // cannot bind socket
                       KMsgBox::message(0,klocale->translate("AX.25-Error"),klocale->translate("cannot bind socket"));
                       ready = true;
                       break;
               case 4: // cannot connect twice
                       if (!config->autoIncSSID)
                       {
                          KMsgBox::message(0,klocale->translate("AX.25-Error"),klocale->translate("cannot connect twice"));
                          ready = true;
                       }
                       else
                          if (!incSSID(mycall))
                          {
                             KMsgBox::message(0,klocale->translate("AX.25-Error"),klocale->translate("cannot increment ssid"));
                             ready = true;
                          }
                       break;
               case 0xFF: // unknown port name
                       KMsgBox::message(0, klocale->translate("AX.25-Error"), klocale->translate("The given port is not known"));
                       ready = true;
                       break;
            }
         }
         else
         {
            if (currentChannel != NULL && !((Channel *)currentChannel)->isConnected())
            {
               ((Channel *)currentChannel)->makeNewConnect( port,
                                                            mycall,
                                                            tmp,
                                                            digis,
                                                            fd );
               addChannelList()->channel = (Channel *)currentChannel;
            }
            else
            {
               addChannelList()->channel = new Channel( 0,
                                                        port,
                                                        mycall,
                                                        tmp,
                                                        digis,
                                                        fd,
                                                        false);
               if (config->hideWinOnSwitch)
                  if (currentChannel != NULL)
                     currentChannel->hide();
            }
            ready = true;
         }
      }
      while (!ready);
      free(tmp);
//      free(digis);
      free(port);
      free(mycall);
   }

   delete dlg;
}


void TopLevel::slotAbout()
{
   AboutDlg *dlg = new AboutDlg(0);

   QPoint point = this->mapToGlobal (QPoint (0,0));

   QRect pos = this->geometry();
   int y = point.y() + pos.height()/2 - dlg->height()/2;
   if (y < 0) y = 0;
   dlg->setGeometry(point.x() + pos.width()/2  - dlg->width()/2,
             y,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


/*
 *  int TopLevel::makeConnect( char *call, char *digis, char *mycall, char *port, int * error )
 * Versucht, die ausgewaehlte Gegenstation zu connecten.
 *
 * Error-Codes: FF = Portname nicht vorhanden
 */
int TopLevel::makeConnect( char *call, char *digis, char *mycall, char *port, int * error )
{
   s_ports *tmp;
   bool found=false;

   // Port suchen
   tmp = portlist;
   while (tmp != NULL && !found)
   {
      if (!strcmp(port, tmp->name))
         found = true;
      else
         tmp = tmp->next;
   }
   if (!found)
   {
      *error = 0xFF;
      return -1;
   }

   return tmp->ax25->makeConnect( call, digis, mycall, port, error);
}



/*
 *  void TopLevel::addChannelList()
 * Macht einen Eintrag in die Kanal-Liste und gibt den Pointer auf
 * die neue Struktur zurueck.
 */
s_chanlist * TopLevel::addChannelList()
{
   s_chanlist *tmp;

   if (chanlist == NULL)
   {
      // Erster Eintrag
      chanlist = (s_chanlist *)malloc(sizeof(s_chanlist));
      tmp = chanlist;
   }
   else
   {
      tmp = chanlist;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_chanlist *)malloc(sizeof(s_chanlist));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->channel = NULL;

   return tmp;
}


/*
 *  void TopLevel::deleteChannelList( Channel *chan )
 * Loescht den uebergebenen Kanal aus der Liste 'raus.
 */
void TopLevel::deleteChannelList( Channel *chan )
{
   s_chanlist *tmp,*last;
   bool found=false;


   if (chanlist == NULL) return;

   if (chanlist->channel == chan)
   {
      tmp = chanlist;
      chanlist = chanlist->next;
      free(tmp);
      return;
   }

   tmp = chanlist->next;
   last = chanlist;
   while (tmp != NULL && !found)
   {
      if (tmp->channel == chan)
      {
         last->next = tmp->next;
         found = true;
         free(tmp);
      }
      last = tmp;
      if (!found) tmp = tmp->next;
   }

   tmp = chanlist;
}


s_chanlist * TopLevel::getChanListPtr()
{
   return chanlist;
}


void TopLevel::setupChanListWidget()
{
   chanListTable = new ChanList( this );

//   chanListWidget->setFixedSize( chanListTable->useWidth(), chanListTable->useHeight() );
}


void TopLevel::closeEvent(QCloseEvent *e)
{
   KConfig *kconfig;
   connCall *savecall;
   int i=1;
   char tmp[10];

   if (chanlist != NULL)
   {

      if (KMsgBox::yesNo(NULL,
                         klocale->translate("Quit LinKT?"),
                         klocale->translate("Some channels are still connected. Do you really want to quit?"),
                         0,
                         klocale->translate("&Yes"),
                         klocale->translate("&No")) != 1)
      {
         e->ignore();
         return;
      }
   }

   // Die letzten 15 connecteten Rufzeichen abspeichern
   kconfig = mykapp->getConfig();

   kconfig->setGroup( "LastConnectCalls" );

   savecall = config->connCallsList->first();
   i = 0;
   while (savecall != NULL)
   {
      sprintf(tmp, "%i", i);
      i++;
      kconfig->writeEntry(tmp, savecall->call());
      savecall = config->connCallsList->next();
   }

   kconfig->sync();


   // Routing-Informationen abspeichern
   router->saveRoutes();

   e->accept();
}


void TopLevel::addConnectList( const char *connline )
{
   connCall *call;
   call = config->connCallsList->first();

   if (call == NULL)
   {
      config->connCallsList->append( new connCall(connline) );
      return;
   }

   do
   {
      if (!strcmp(connline, call->call()))
      {
         config->connCallsList->remove();
         config->connCallsList->append( new connCall(connline) );
         return;
      }
   } while ((call = config->connCallsList->next()) != NULL);

   // Diese Zeile ist noch nicht vorhanden. Anhaengen und eventuell die
   // erste loeschen.
   config->connCallsList->append( new connCall(connline) );

   if (config->connCallsList->count() > 15)
      config->connCallsList->removeFirst();
}


void TopLevel::slotHelp()
{
   mykapp->invokeHTMLHelp( "", "" );
}


void TopLevel::readLastConnectedCalls()
{
   int i;
   KConfig *kconfig;
   char tmp[10],*tmp2;


   kconfig = mykapp->getConfig();

   kconfig->setGroup( "LastConnectCalls" );

   i = 0;
   sprintf(tmp,"%i", i);
   while ((tmp2 = kconfig->readEntry(tmp).data()) != NULL)
   {
      addConnectList(tmp2);
      i++;
      sprintf(tmp,"%i", i);
   }

}


void TopLevel::slotPrefCalls()
{
   CfgCalls *dlg = new CfgCalls();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotPrefPorts()
{
   CfgPorts *dlg = new CfgPorts();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotPrefSound()
{
   bool enabled = config->sound->enabled;
   CfgSound *dlg = new CfgSound();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();
   delete dlg;

   // Das Audio-Zeugs wird aktiviert
#ifndef WITHOUT_AUDIO
   if (!enabled && config->sound->enabled)
   {
      if (KAServer.serverStatus())
      {
         KMsgBox::message( 0,
                           klocale->translate("Sound-Error"),
                           klocale->translate("Cannot contact the audio server. Audio disabled."));
         config->sound->enabled = false;
      }
   }
#else
   KMsgBox::message( 0,
                     klocale->translate("Sound"),
                     klocale->translate("Sound disabled ('#define WITHOUT_AUDIO' in main.h."));
#endif
}


void TopLevel::slotPrefScreen()
{
   int a, b, buttonbar;
   char *txfont, *rxfont, *bcfont;
   bool rxbold, txbold;
   s_chanlist *tmp;
   CfgScreen *dlg = new CfgScreen();
   int colTxText, colRxText, colStatusText, colBackground;
   int colMarkColor, colMarkBack, bcsize;
   bool txWinOben;



   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   a = config->qsofontsize;
   b = config->txfontsize;
   bcsize = config->bcfontsize;
   txfont = (char *) strdup(config->txfontstr);
   rxfont = (char *) strdup(config->rxfontstr);
   bcfont = (char *) strdup(config->bcfontstr);
   txbold = config->colors->txwin_bold;
   rxbold = config->colors->qsowin_bold;
   colTxText = config->colors->txText;
   colRxText = config->colors->rxText;
   colStatusText = config->colors->statusText;
   colBackground = config->colors->background;
   colMarkColor = config->colors->qsoWinMarkColor;
   colMarkBack = config->colors->qsoWinMarkBack;
   txWinOben = config->txWinOben;
   buttonbar = config->buttonbar;

   dlg->exec();

   if (( a != config->qsofontsize) || (b != config->txfontsize) ||
       strcmp(txfont, config->txfontstr) || strcmp(rxfont, config->rxfontstr) ||
       (rxbold != config->colors->qsowin_bold) || (txbold != config->colors->txwin_bold) ||
       colTxText != config->colors->txText || colRxText != config->colors->rxText ||
       colStatusText != config->colors->statusText || colBackground != config->colors->background ||
       colMarkColor != config->colors->qsoWinMarkColor || colMarkBack != config->colors->qsoWinMarkBack)
   {
      // Schrifgroessen oder Schriftarten haben sich geaendert - updaten
      for (tmp=chanlist; tmp; tmp=tmp->next)
			tmp->channel->updateFont();
	}

	if (bcsize != config->bcfontsize || strcmp(bcfont, config->bcfontstr))
		for (tmp=chanlist; tmp; tmp=tmp->next)
			tmp->channel->updateBCSize();

	if (txWinOben != config->txWinOben)
		for (tmp=chanlist; tmp; tmp=tmp->next)
			tmp->channel->updateWindowPos();

	if (buttonbar != config->buttonbar)
		for (tmp=chanlist; tmp; tmp=tmp->next)
			tmp->channel->updateButtonbar();

	free(txfont);
	free(rxfont);
	free(bcfont);

	delete dlg;
}


void TopLevel::slotPrefAway()
{
   CfgAway *dlg = new CfgAway();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotPrefShortcut()
{
   CfgShortcut *dlg = new CfgShortcut();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotPrefCallKey()
{
   CfgCallKey *dlg = new CfgCallKey();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotPrefFiles()
{
   CfgFiles *dlg = new CfgFiles();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


//   void TopLevel::slotCallKey(int id);
//
// Wird aufgerufen, wenn im Vorschreibfenster die Funktionstasten benutzt
// werden.
void TopLevel::slotCallKey(int id)
{
   char rxcall[20];
   s_chanlist *tmp;
   int i;
   QWidget *win,*akt;


   if (config->callKey[id] == NULL)
   {
      QApplication::beep();
      return;
   }

   // Gucken, ob dieses Rufzeichen irgendwo eingeloggt ist
   tmp = chanlist;
   while (tmp != NULL)
   {
      strcpy(rxcall, tmp->channel->getCall());
      if (!config->callKey[id]->ssid)
         if ((i = POS('-', rxcall)) > -1)
           rxcall[i] = '\0';

      if (!strcmp(rxcall, config->callKey[id]->call))
      {
         win = tmp->channel;

         if (win->isActiveWindow()) return;
         akt = toplevel->currentChannel;
         activateChannel(win, akt, true);

         // Den aktuellen Kanal verstecken
/*         if (config->hideWinOnSwitch)
            if (akt != NULL)
               if (akt != win)
                  akt->hide();*/

         chanListTable->auswahlLoeschen();
         return;
      }

      tmp = tmp->next;
   }

   QApplication::beep();
}



void TopLevel::slotPrefFlags()
{
   CfgFlags *dlg = new CfgFlags();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotPrefDirs()
{
   CfgDirs *dlg = new CfgDirs();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::activateChannel(QWidget *win, QWidget *oldwin, bool show)
{
/*   QPoint *pos;

   if (toplevel->currentChannel != NULL)
      pos = new QPoint(currentChannel->pos());
   else
      pos = NULL;*/
//   QSize size(currentChannel->size());

   if ((win == oldwin) && !show) return;

   KWM::activate(win->winId());
   win->show();
   ((Channel *)win)->getVorschreibPtr()->setFocus();

   if (currentChannel != NULL)
      if (config->hideWinOnSwitch)
         KWM::move(win->winId(), currentChannel->pos());

   if (config->hideWinOnSwitch)
      if (oldwin != NULL)
         oldwin->hide();

   currentChannel = win;

//   if (pos != NULL) delete pos;
}


void TopLevel::setQsoWinGeometry( const QRect rect )
{
   qsoWinGeometry = rect;
}


QRect & TopLevel::getQsoWinGeometry()
{
   return qsoWinGeometry;
}


bool TopLevel::qsoWinGeometrySet()
{
   return qsoWinGeometryFlag;
}


void TopLevel::checkForFirstRun()
{
   s_ports *tmp;
   char text[500];
   char mycall[50];


   mycall[0] = '\0';
   if (config->ports == NULL)
   {
      // Alle Ports aktivieren
      text[0] = '\0';
      tmp = portlist;
      while (tmp != NULL)
      {
         // Port an die Portliste anhaengen
         if (text[0] != '\0') strcat(text,",");
         strcat(text, tmp->name);

         if (mycall[0] == '\0') strcpy(mycall, tmp->addr);
         tmp->ax25 = new AX25(tmp->name, tmp->desc, tmp->dev, tmp->baud, tmp->addr);

         tmp = tmp->next;
      }
   }


   if (config->send_mycall[0] == '\0')
   {
      // Das eigene Rufzeichen ist unbekannt. Das Standard-Rufzeichen
      // des ersten Ports wird benutzt.
      if (mycall[0] == '\0')
         strcpy(config->send_mycall, portlist->addr);
      else
         strcpy(config->send_mycall, mycall);
   }
}


void TopLevel::slotRouteAdd()
{
   char call[100], via[100], port[100];
   char c[100], v[100];
   DlgAddRoute *dlg = new DlgAddRoute();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   if (dlg->exec() != 1) return;

   dlg->getRoute( call, via, port );

   delete dlg;

   OnlyOneSpace(c, call);
   OnlyOneSpace(v, via);
   KillSpacesLeft(c);
   KillSpacesRight(c);
   KillSpacesLeft(v);
   KillSpacesRight(v);
   Gross(c);
   Gross(v);

   strcat(port, " ");
   strcat(port, v);
   strcpy(v,port);

   router->addRoute(c, v);
}


void TopLevel::slotRouteDelete()
{
}


void TopLevel::slotRouteShowList()
{
   ListRoutes *dlg = new ListRoutes();

   dlg->setGeometry((mykapp->desktop()->width()-dlg->width())/2,
             (mykapp->desktop()->height()-dlg->height())/2,
          dlg->width(),dlg->height());

   dlg->exec();

   delete dlg;
}


void TopLevel::slotRouteShowTree()
{
   router->showRoutes();
}


void TopLevel::playSound( int id )
{
#ifndef WITHOUT_AUDIO
   switch (id)
   {
      case SOUND_CONNECT:
           if (config->sound->connect != NULL)
              KAServer.play(config->sound->connect);
           break;
      case SOUND_DISCONNECT:
           if (config->sound->disconnect != NULL)
              KAServer.play(config->sound->disconnect);
           break;
      case SOUND_RING:
           if (config->sound->ring != NULL)
              KAServer.play(config->sound->ring);
           break;
      case SOUND_KLACK:
           if (config->sound->klackfile != NULL)
              KAServer.play(config->sound->klackfile);
           break;
   }
#endif
}


void TopLevel::playSound( const char *filename )
{
#ifndef WITHOUT_AUDIO
   KAServer.play( filename );
#endif
}



/*************************************************************************/
/*************************************************************************/
/*************************************************************************/
/*************************************************************************/


int main (int argc, char **argv)
{
//   InterfaceVerwaltung *iface;

   if (argc > 1)
   {
      if (!strcmp(argv[1], "--version"))
      {
         #ifndef WITHOUT_AUDIO
            printf("\nLinKT V%s (%s)\nCopyright (C) 1997-2000 Jochen Sarrazin, DG6VJ\n\n", LINKT_VERSION, DATE);
         #else
            printf("\nLinKT V%s (%s) WITHOUT_AUDIO\nCopyright (C) 1997-2000 Jochen Sarrazin, DG6VJ\n\n", LINKT_VERSION, DATE);
         #endif
         printf("This program is distributed in the hope that it will be useful,\n" \
                "but WITHOUT ANY WARRANTY; without even the implied warranty of\n" \
                "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n" \
                "GNU General Public License for more details.\n\n");
         return 0;
      }
   }

   srand(time(NULL));

#ifndef WITHOUT_AUDIO
   printf("\nLinKT V%s (%s)\nCopyright (C) 1997-2000 Jochen Sarrazin, DG6VJ\n\n", LINKT_VERSION, DATE);
#else
   printf("\nLinKT V%s (%s) WITHOUT_AUDIO\nCopyright (C) 1997-2000 Jochen Sarrazin, DG6VJ\n\n", LINKT_VERSION, DATE);
#endif

   mykapp = new KApplication (argc, argv, "linkt");

   init_globals();
   readBoxCheckFile();

   toplevel = new TopLevel ();
   toplevel->show ();

   mykapp->setMainWidget(toplevel);
   createDirectories();
   toplevel->readLastConnectedCalls();

   toplevel->checkForFirstRun();

   //Interface aktivieren
//   iface = new InterfaceVerwaltung();

   return mykapp->exec ();
}


void getarg( char *line, char *arg )
{
   int i;
   int len;


   if ((i = POS(' ', line)) == -1)
   {
      len = strlen(line);
      memcpy(arg, line, len);
      arg[len] = '\0';
      line[0] = '\0';
      return;
   }

   len = strlen(line)-i-1;
   memcpy(arg, line, i);
   arg[i] = '\0';
   memmove(line, line+i+1, len);
   line[len] = '\0';
}


/*
 *  void createDirectories()
 * Verzechnisse erstellen, wenn es sie noch nicht gibt.
 */
void createDirectories()
{
   char tmp[500];

   mkdir(config->maindir, 448);
   sprintf(tmp, "%s/7plus", config->maindir);
   mkdir(tmp, 448);
   sprintf(tmp, "%s/up", config->maindir);
   mkdir(tmp, 448);
   sprintf(tmp, "%s/down", config->maindir);
   mkdir(tmp, 448);
   sprintf(tmp, "%s/abin", config->maindir);
   mkdir(tmp, 448);
   sprintf(tmp, "%s/userdb", config->maindir);
   mkdir(tmp, 448);
   sprintf(tmp, "%s/log", config->maindir);
   mkdir(tmp, 448);
}


void decodeListenToCalls(char *text)
{
   char tmp[500], str[500];
   int i;
   s_listCalls *tmpcall;

   strcpy(str,text);
   while (str[0] != '\0')
   {
      if ((i = POS(',',str)) != -1)
      {
         COPY(tmp, str, 0, i);
         COPY(str, str, i+1, strlen(str)-i-1);
      }
      else
      {
         strcpy(tmp,str);
         str[0] = '\0';
      }

      // Neuer Eintrag in die Liste
      if (config->listento_calls == NULL)
      {
         config->listento_calls = (s_listCalls *)malloc(sizeof(s_listCalls));
         tmpcall = config->listento_calls;
      }
      else
      {
         tmpcall = config->listento_calls;
         while (tmpcall->next != NULL) tmpcall = tmpcall->next;
         tmpcall->next = (s_listCalls *)malloc(sizeof(s_listCalls));
         tmpcall = tmpcall->next;
      }

      tmpcall->next = NULL;
      tmpcall->call = (char *) strdup(tmp);
   }
}



void decodeDisconnectCalls(char *text)
{
   char tmp[500], str[500];
   int i;
   s_listCalls *tmpcall;

   strcpy(str,text);
   while (str[0] != '\0')
   {
      if ((i = POS(',',str)) != -1)
      {
         COPY(tmp, str, 0, i);
         COPY(str, str, i+1, strlen(str)-i-1);
      }
      else
      {
         strcpy(tmp,str);
         str[0] = '\0';
      }

      // Neuer Eintrag in die Liste
      if (config->discon_calls == NULL)
      {
         config->discon_calls = (s_listCalls *)malloc(sizeof(s_listCalls));
         tmpcall = config->discon_calls;
      }
      else
      {
         tmpcall = config->discon_calls;
         while (tmpcall->next != NULL) tmpcall = tmpcall->next;
         tmpcall->next = (s_listCalls *)malloc(sizeof(s_listCalls));
         tmpcall = tmpcall->next;
      }

      tmpcall->next = NULL;
      tmpcall->call = (char *) strdup(tmp);
   }
}





/*
 *  void init_globals()
 * Globale Variablen aus dem linktrc-File holen und in die config-
 * Struktur packen.
 */
void init_globals()
{
  KConfig *kconfig;
  char *tmp;
  char tmp2[500];
  int i,k;


  config = (s_config *) malloc(sizeof(s_config));
  config->sound = (s_conf_sound *) malloc(sizeof(s_conf_sound));
  config->colors = (s_conf_colors *) malloc(sizeof(s_conf_colors));


  config->ports = NULL;
  config->away.wo = 0;

  config->send_mycall[0] = '\0';
  config->maindir = (char *) malloc( strlen(getenv("HOME"))+50 );
//  sprintf(config->maindir,"%s/share/apps/linkt",mykapp->kdedir().data());
  sprintf(config->maindir,"%s/.kde/share/apps/linkt",getenv("HOME"));
  config->starttime = time(NULL);

  sprintf(tmp2,"%s/linkt/driver/",(const char *)mykapp->kde_datadir());
  config->driverdir = (char *) strdup(tmp2);



  // Konfigurations-File holen
  kconfig = mykapp->getConfig();

  //***** Gruppe AX.25 - Rufzeichen etc.
  kconfig->setGroup( "AX.25" );
  strcpy(config->send_mycall,kconfig->readEntry("sendcall").data());

  if ((tmp = kconfig->readEntry("ports").data()) != NULL)
     if (tmp[0] != '\0') config->ports = (char *) strdup(tmp);

  // Listen-To-Calls
  config->listento_calls = NULL;
  if ((tmp = kconfig->readEntry("listentocalls").data()) != NULL)
     decodeListenToCalls(tmp);
  // Disconnect-Calls
  config->discon_calls = NULL;
  if ((tmp = kconfig->readEntry("disconcalls").data()) != NULL)
     decodeDisconnectCalls(tmp);

  if ((tmp = kconfig->readEntry("autoIncSSID").data()) != NULL)
     if (atoi(tmp) != 0)
        config->autoIncSSID = true;
     else
        config->autoIncSSID = false;
  else config->autoIncSSID = true;       // Default


  if ((tmp = kconfig->readEntry("autosendName").data()) != NULL)
     if (atoi(tmp) != 0)
        config->autosendName = true;
     else
        config->autosendName = false;
  else config->autosendName = true;       // Default

  if ((tmp = kconfig->readEntry("yappResume").data()) != NULL)
     if (atoi(tmp) != 0)
        config->yappResume = true;
     else
        config->yappResume = false;
  else config->yappResume = false;        // Default

  if ((tmp = kconfig->readEntry("extraTransferWin").data()) != NULL)
     if (atoi(tmp) != 0)
        config->extraTransferWin = true;
     else
        config->extraTransferWin = false;
  else config->extraTransferWin = false;         // Default

  if ((tmp = kconfig->readEntry("useLogbook").data()) != NULL)
     if (atoi(tmp) != 0)
        config->useLogbook = true;
     else
        config->useLogbook = false;
  else config->useLogbook = true;         // Default


  // Gruppe Buffers
  kconfig->setGroup("Buffersizes" );
  if ((tmp = kconfig->readEntry("receive").data()) != NULL)
      config->rx_buffersize = atoi (tmp);
  else
      config->rx_buffersize = 1000;

  if ((tmp = kconfig->readEntry("transmit").data()) != NULL)
      config->tx_buffersize = atoi (tmp);
  else
      config->tx_buffersize = 100;


  // Gruppe Colors
  kconfig->setGroup( "Colors" );
  if ((tmp = kconfig->readEntry("receive").data()) != NULL)
     config->colors->rxText = atoi(tmp);
  else
     config->colors->rxText = 12;
  if ((tmp = kconfig->readEntry("transmit").data()) != NULL)
     config->colors->txText = atoi(tmp);
  else
     config->colors->txText = 7;
  if ((tmp = kconfig->readEntry("status").data()) != NULL)
     config->colors->statusText = atoi(tmp);
  else
     config->colors->statusText = 5;
  if ((tmp = kconfig->readEntry("background").data()) != NULL)
     config->colors->background = atoi(tmp);
  else
     config->colors->background = 1;

  if ((tmp = kconfig->readEntry("txwin").data()) != NULL)
     config->colors->txwin = atoi(tmp);
  else
     config->colors->txwin = 0;

  if ((tmp = kconfig->readEntry("txbackground").data()) != NULL)
     config->colors->txbackground = atoi(tmp);
  else
     config->colors->txbackground = 1;


  if ((tmp = kconfig->readEntry("qsoWinBold").data()) != NULL)
     if (atoi(tmp) != 0)
        config->colors->qsowin_bold = true;
     else
        config->colors->qsowin_bold = false;
  else config->colors->qsowin_bold = false;     // Default

  if ((tmp = kconfig->readEntry("txWinBold").data()) != NULL)
     if (atoi(tmp) != 0)
        config->colors->txwin_bold = true;
     else
        config->colors->txwin_bold = false;
  else config->colors->txwin_bold = false;      // Default

  if ((tmp = kconfig->readEntry("qsoMarkColor").data()) != NULL)
     config->colors->qsoWinMarkColor = atoi(tmp);
  else
     config->colors->qsoWinMarkColor = 1;

  if ((tmp = kconfig->readEntry("qsoMarkBackground").data()) != NULL)
     config->colors->qsoWinMarkBack = atoi(tmp);
  else
     config->colors->qsoWinMarkBack = 0;

  if ((tmp = kconfig->readEntry("qsoMarkBold").data()) != NULL)
     if (atoi(tmp) != 0)
        config->colors->mark_bold = true;
     else
        config->colors->mark_bold = false;
  else config->colors->mark_bold = false;      // Default



  // Gruppe Sound
  kconfig->setGroup( "Sound" );

  if ((tmp = kconfig->readEntry("enabled").data()) != NULL)
     if (atoi(tmp) != 0)
        config->sound->enabled = true;
     else
        config->sound->enabled = false;
  else config->sound->enabled = true;

  if ((tmp = kconfig->readEntry("connect").data()) != NULL)
     if (tmp[0] == '\0')
        config->sound->connect = NULL;
     else
        config->sound->connect = (char *) strdup( tmp );
  else
     config->sound->connect = NULL;

  if ((tmp = kconfig->readEntry("disconnect").data()) != NULL)
     if (tmp[0] == '\0')
        config->sound->disconnect = NULL;
     else
        config->sound->disconnect = (char *) strdup( tmp );
  else
     config->sound->disconnect = NULL;

  if ((tmp = kconfig->readEntry("ring").data()) != NULL)
     if (tmp[0] == '\0')
        config->sound->ring = NULL;
     else
        config->sound->ring = (char *) strdup( tmp );
  else
     config->sound->ring = NULL;

  if ((tmp = kconfig->readEntry("klack").data()) != NULL)
     if (atoi(tmp) != 0)
        config->sound->klack = true;
     else
        config->sound->klack = false;
  else config->sound->klack = true;

  if ((tmp = kconfig->readEntry("klack_file").data()) != NULL)
     if (tmp[0] == '\0')
        config->sound->klackfile = NULL;
     else
        config->sound->klackfile = (char *) strdup( tmp );
  else
     config->sound->klackfile = NULL;


  // Gruppe Screen
  kconfig->setGroup( "Screen" );

  if ((tmp = kconfig->readEntry("X").data()) != NULL)
     config->screen_x = atoi( tmp );
  else
     config->screen_x = 600;

  if ((tmp = kconfig->readEntry("Y").data()) != NULL)
     config->screen_y = atoi( tmp );
  else
     config->screen_y = 300;

  // Minimal-Groesse
  if (config->screen_x < 200) config->screen_x = 200;
  if (config->screen_y < 100) config->screen_y = 100;

  // Schriften
  config->rxfontstr = NULL;
  config->txfontstr = NULL;
  config->bcfontstr = NULL;
  if ((tmp = kconfig->readEntry("rxfont").data()) != NULL)
     config->rxfontstr = (char *) strdup(tmp);
  if ((tmp = kconfig->readEntry("txfont").data()) != NULL)
     config->txfontstr = (char *) strdup(tmp);
  if ((tmp = kconfig->readEntry("bcfont").data()) != NULL)
     config->bcfontstr = (char *) strdup(tmp);
  if (config->rxfontstr == NULL)
     config->rxfontstr = (char *) strdup("fixed");
  if (config->txfontstr == NULL)
     config->txfontstr = (char *) strdup("fixed");
  if (config->bcfontstr == NULL)
     config->bcfontstr = (char *) strdup("fixed");

  // Schriftgroessen
  config->qsofontsize = 12;
  if ((tmp = kconfig->readEntry("qsofontsize").data()) != NULL)
  {
     config->qsofontsize = atoi(tmp);
     if (config->qsofontsize < 8)
        config->qsofontsize = 12;
  }

  config->txfontsize = 12;
  if ((tmp = kconfig->readEntry("txfontsize").data()) != NULL)
  {
     config->txfontsize = atoi(tmp);
     if (config->txfontsize < 8)
        config->txfontsize = 12;
  }

  config->bcfontsize = 11;
  if ((tmp = kconfig->readEntry("bcfontsize").data()) != NULL)
  {
     config->bcfontsize = atoi(tmp);
     if (config->bcfontsize < 8)
        config->bcfontsize = 11;
  }


	// ANSI-Codes dekodieren?
	config->allowANSI = true;
	if ((tmp = kconfig->readEntry("allowANSI").data()) != NULL)
		if (tmp[0] == '0')
			config->allowANSI = false;

  config->txWinOben = false;
  if ((tmp = kconfig->readEntry("txWinOben").data()) != NULL)
     if (tmp[0] == '1')
        config->txWinOben = true;

  config->txWinSize = 30;
  if ((tmp = kconfig->readEntry("txWinSize").data()) != NULL)
  {
     config->txWinSize = atoi(tmp);
     if (config->txWinSize == 0 || config->txWinSize > 100)
        config->txWinSize = 30;
  }

  // Kanal-Fenster beim Disconnect schliessen?
  config->closeWinOnDisc = false;
  if ((tmp = kconfig->readEntry("closeWinOnDisc").data()) != NULL)
     if (tmp[0] == '1')
        config->closeWinOnDisc = true;

  // Kanal-Fenster beim Kanalwechsel verstecken?
  config->hideWinOnSwitch = true;
  if ((tmp = kconfig->readEntry("hideWinOnSwitch").data()) != NULL)
     if (tmp[0] == '0')
        config->hideWinOnSwitch = false;

  // Neue Connects sofort anzeigen?
  config->openChannelWin = true;
  if ((tmp = kconfig->readEntry("openChannelWin").data()) != NULL)
     if (tmp[0] == '0')
        config->openChannelWin = false;

  // Schon vorhandene Files ueberschreiben?
  config->overwriteExisting = false;
  if ((tmp = kconfig->readEntry("overwriteExisting").data()) != NULL)
     if (tmp[0] == '1')
        config->overwriteExisting = true;

  // Lokales Echo?
  config->localEcho = true;
  if ((tmp = kconfig->readEntry("localEcho").data()) != NULL)
     if (tmp[0] == '0')
        config->localEcho = false;

   // Piepen, wenn CTRL-G empfangen wurde?
  config->beepCtrlG = false;
  if ((tmp = kconfig->readEntry("beepCtrlG").data()) != NULL)
     if (tmp[0] == '0')
        config->beepCtrlG = false;
     else
        config->beepCtrlG = true;


  // Zeilenumbruch
  config->zeilenumbruch = true;
  if ((tmp = kconfig->readEntry("lineBreak").data()) != NULL)
     if (tmp[0] == '0')
        config->zeilenumbruch = false;

  // Position des Zeilenumbruchs
  config->umbruchpos = 78;
  if ((tmp = kconfig->readEntry("lineBreakPos").data()) != NULL)
  {
     if ((config->umbruchpos = atoi(tmp)) == 0)
        config->umbruchpos = 78;
  }

  // BoxCheck-Liste sortieren
  config->sortBoxCheck = true;
  if ((tmp = kconfig->readEntry("sortBoxCheck").data()) != NULL)
     if (tmp[0] == '0')
        config->sortBoxCheck = false;


  config->rx_font = new QFont(config->rxfontstr, 12);
  config->tx_font = new QFont(config->txfontstr, 12);

  // Buttonbar
  config->buttonbar = BARFLG_SHOW | BARFLG_RIGHT;
  if ((tmp = kconfig->readEntry("buttonBar").data()) != NULL)
     config->buttonbar = atoi(tmp);


  // ----------- Away-Texte -----------
  kconfig->setGroup( "Away" );
  if ((tmp = kconfig->readEntry("phone").data()) != NULL)
     config->away.phone = (char *) strdup( tmp );
  else
     config->away.phone = (char *) strdup( "Ma eben telefonieren... [%a]" );

  if ((tmp = kconfig->readEntry("bed").data()) != NULL)
     config->away.bed = (char *) strdup( tmp );
  else
     config->away.bed = (char *) strdup( "Gute n8! Ab in die Falle... [%a]" );

  if ((tmp = kconfig->readEntry("back").data()) != NULL)
     config->away.back = (char *) strdup( tmp );
  else
     config->away.back = (char *) strdup( "Wiiilmaaaaa!!!!! Ich bin wieder daaaaaaa!!! [%a]" );

  if ((tmp = kconfig->readEntry("away").data()) != NULL)
     config->away.away = (char *) strdup( tmp );
  else
     config->away.away = (char *) strdup( "Mal eben wech, %n... Kurz oder laenger... [%a]" );

   strcpy(config->away.text, config->away.away);


   // Makros
   kconfig->setGroup( "Shortcuts" );
   for (i=0;i<10;i++)
   {
      sprintf(tmp2, "CTRL+%i", i);
      if ((tmp = kconfig->readEntry(tmp2).data()) != NULL)
         config->shortcuts[i] = (char *)strdup(tmp);
      else
         config->shortcuts[i] = NULL;
   }



   kconfig->setGroup("CallKey");
   for (i=0;i<10;i++)
   {
      sprintf(tmp2, "F%i", i);
      if ((tmp = kconfig->readEntry(tmp2).data()) != NULL)
         if (tmp[0] != '\0')
         {
            config->callKey[i] = (s_callKey *) malloc(sizeof(s_callKey));
            strcpy(config->callKey[i]->call, tmp);
            sprintf(tmp2, "ssid%i", i);
            if ((tmp = kconfig->readEntry(tmp2).data()) != NULL)
            {
               if ((k = atoi(tmp)) == 0)
                  config->callKey[i]->ssid = false;
               else
                  config->callKey[i]->ssid = true;
            }
            else config->callKey[i]->ssid = false;
         }
         else
            config->callKey[i] = NULL;
      else
         config->callKey[i] = NULL;
   }


   kconfig->setGroup("Directories");
   if ((tmp = kconfig->readEntry("7plus").data()) != NULL)
      config->dir7plus = (char *) strdup( tmp );
   else
   {
      config->dir7plus = (char *) malloc( strlen(config->maindir)+20 );
      sprintf(config->dir7plus, "%s/7plus", config->maindir);
   }

   if ((tmp = kconfig->readEntry("abin").data()) != NULL)
      config->dirABin = (char *) strdup( tmp );
   else
   {
      config->dirABin = (char *) malloc( strlen(config->maindir)+20 );
      sprintf(config->dirABin, "%s/abin", config->maindir);
   }

   if ((tmp = kconfig->readEntry("down").data()) != NULL)
      config->dirDown = (char *) strdup( tmp );
   else
   {
      config->dirDown = (char *) malloc( strlen(config->maindir)+20 );
      sprintf(config->dirDown, "%s/down", config->maindir);
   }

   if ((tmp = kconfig->readEntry("up").data()) != NULL)
      config->dirUp = (char *) strdup( tmp );
   else
   {
      config->dirUp = (char *) malloc( strlen(config->maindir)+20 );
      sprintf(config->dirUp, "%s/up", config->maindir);
   }


   // Rufzeichen-Liste initialisieren
   config->connCallsList = new QList<connCall>();
}



s_boxcheck * newBoxCheckEntry()
{
   s_boxcheck *tmp;


   if (boxcheckroot == NULL)
   {
      // Erster Eintrag
      boxcheckroot = (s_boxcheck *)malloc(sizeof(s_boxcheck));
      tmp = boxcheckroot;
   }
   else
   {
      tmp = boxcheckroot;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_boxcheck *)malloc(sizeof(s_boxcheck));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->enabled = true;
   tmp->type = -1;
   tmp->desc = NULL;
   tmp->title = NULL;
   tmp->board = NULL;
   tmp->lifetime = NULL;
   tmp->size = NULL;
   tmp->sender = NULL;
   tmp->number = NULL;
   tmp->mustcount = 0;
   tmp->minlen = 0;
   tmp->fields = 0;

   return tmp;
}


void checkBoxCheckEntry( s_boxcheck *bc )
{
	if (bc->desc == NULL)
   {
   	printf("BoxCheck-Error: Description-statement missing. Entry disabled.\n");
      bc->enabled = false;
      return;
   }

   if (bc->type == -1)
   {
   	printf("BoxCheck-Error: Type in '%s' missing. Entry disabled.\n", bc->desc);
      bc->enabled = false;
      return;
   }

   if (bc->title == NULL)
   {
   	printf("BoxCheck-Error: Title in '%s' missing. Entry disabled.\n", bc->desc);
      bc->enabled = false;
      return;
   }

   if (bc->board == NULL)
   {
   	printf("BoxCheck-Error: 'pos board'-statement in '%s' missing.\n                Entry disabled.\n", bc->desc);
      bc->enabled = false;
      return;
   }

   if (bc->number == NULL)
   {
   	printf("BoxCheck-Error: 'pos number'-statement in '%s' missing.\n                Entry disabled.\n", bc->desc);
      bc->enabled = false;
      return;
   }
}


//   void readBoxCheckFile()
//
// Einlesen der Daten, die fuers BoxCheck benoetigt werden
void readBoxCheckFile()
{
   FILE *f;
   char line[500], cmd[100], str[100], ch;
   s_boxcheck *tmp=NULL;
   int len, pos, linenr=0, i;


   boxcheckroot = NULL;

   sprintf(str, "%s/boxcheck.dat", config->maindir);
   if ((f = fopen(str,"r")) == NULL)
   {
      sprintf(str, "%s/linkt/boxcheck.dat", (const char *)mykapp->kde_datadir());
      if ((f = fopen(str,"r")) == NULL) return;
   }


	while (fgets(line, 499, f) != NULL)
	{
   	linenr++;
		KillSpacesLeft(line);
		if (line[0] != '\0' && line[0] != '#' && line[0] != '\n')
		{
      	if ((i = POS('\n', line)) != -1) line[i] = '\0';

			getarg(line, cmd);

			if (!strcmp(cmd, "desc"))
			{
				// Neuer Eintrag
            if (tmp != NULL)
               checkBoxCheckEntry( tmp );
				tmp = newBoxCheckEntry();
            tmp->desc = (char *) strdup(line);
            continue;
         }

         if (tmp == NULL)
         {
         	printf("Error line %i: 'desc'-statement missing.\n", linenr);
				continue;
         }

         if (!strcmp(cmd, "titel"))
         {
         	// Der Titel des alten Eintrags
				if (tmp->title != NULL)
				{
					printf("Error line %i: Entry already has a titel\n", linenr);
					continue;
				}
				if ((i = POS('\"', line)) == -1)
				{
					printf("Error line %i: Wrong syntax\n", linenr);
					continue;
				}
				else
				{
					len = strlen(line)-i-1;
					memmove(line, line+i+1, len);
					line[len] = '\0';
					if ((i = POS('\"', line)) == -1)
					{
						printf("Error line %i: Wrong syntax\n", linenr);
						continue;
					}
					else
					{
						line[i] = '\0';
						tmp->title = (char *) strdup(line);
						continue;
					}
				}
         }

         if (!strcmp(cmd, "type"))
         {
         	if (!strcmp(line, "baybox") || !strcmp(line, "dpbox") || !strcmp(line, "thebox"))
            {
            	tmp->type = BCTYPE_BAYBOX;
            	continue;
            }

         	printf("Error line %i: Unknown type.\n", linenr);
            continue;
         }

			if (!strcmp(cmd, "pos"))
         {
				getarg(line, cmd);
				if (!strcmp(cmd, "board"))
            {
					if (tmp->board != NULL)
					{
               	printf("Error line %i: Entry already has a board-statement.\n", linenr);
                  continue;
					}
					if (sscanf(line, "%i %i", &pos, &len) != 2)
					{
                  printf("Error line %i: Invalid position or length in board-statement\n", linenr);
						continue;
					}
               else
               {
						tmp->board = (s_checkpos *) malloc(sizeof(s_checkpos));
                  tmp->board->pos = pos-1;
                  tmp->board->len = len;
                  if (tmp->minlen < pos) tmp->minlen = pos;
                  tmp->fields |= BCFIELD_BOARD;
						continue;
               }
            }
				if (!strcmp(cmd, "number"))
            {
					if (tmp->number != NULL)
					{
               	printf("Error line %i: Entry already has a number-statement.\n", linenr);
                  continue;
					}
					if (sscanf(line, "%i %i", &pos, &len) != 2)
					{
                  printf("Error line %i: Invalid position or length in number-statement\n", linenr);
						continue;
					}
               else
               {
						tmp->number = (s_checkpos *) malloc(sizeof(s_checkpos));
                  tmp->number->pos = pos-1;
                  tmp->number->len = len;
                  if (tmp->minlen < pos) tmp->minlen = pos;
                  tmp->fields |= BCFIELD_NUMBER;
						continue;
               }
            }
				if (!strcmp(cmd, "lifetime"))
            {
					if (tmp->lifetime != NULL)
					{
               	printf("Error line %i: Entry already has a lifetime-statement.\n", linenr);
                  continue;
					}
					if (sscanf(line, "%i %i", &pos, &len) != 2)
					{
                  printf("Error line %i: Invalid position or length in lifetime-statement\n", linenr);
						continue;
					}
               else
               {
						tmp->lifetime = (s_checkpos *) malloc(sizeof(s_checkpos));
                  tmp->lifetime->pos = pos-1;
                  tmp->lifetime->len = len;
                  if (tmp->minlen < pos) tmp->minlen = pos;
                  tmp->fields |= BCFIELD_LIFETIME;
						continue;
               }
            }
				if (!strcmp(cmd, "size"))
            {
					if (tmp->size != NULL)
					{
               	printf("Error line %i: Entry already has a size-statement.\n", linenr);
                  continue;
					}
					if (sscanf(line, "%i %i", &pos, &len) != 2)
					{
                  printf("Error line %i: Invalid position or size in number-statement\n", linenr);
						continue;
					}
               else
               {
						tmp->size = (s_checkpos *) malloc(sizeof(s_checkpos));
                  tmp->size->pos = pos-1;
                  tmp->size->len = len;
                  if (tmp->minlen < pos) tmp->minlen = pos;
                  tmp->fields |= BCFIELD_SIZE;
						continue;
               }
            }
				if (!strcmp(cmd, "sender"))
            {
					if (tmp->sender != NULL)
					{
               	printf("Error line %i: Entry already has a sender-statement.\n", linenr);
                  continue;
					}
					if (sscanf(line, "%i %i", &pos, &len) != 2)
					{
                  printf("Error line %i: Invalid position or length in sender-statement\n", linenr);
						continue;
					}
               else
               {
						tmp->sender = (s_checkpos *) malloc(sizeof(s_checkpos));
                  tmp->sender->pos = pos-1;
                  tmp->sender->len = len;
                  if (tmp->minlen < pos) tmp->minlen = pos;
                  tmp->fields |= BCFIELD_SENDER;
						continue;
               }
            }
         }

         if (!strcmp(cmd, "char"))
         {
				if (sscanf(line, "%c %i", &ch, &pos) != 2)
            {
            	printf("Error line %i: Syntax error in 'char'-statement.\n", linenr);
            	continue;
            }
            tmp->must[tmp->mustcount].pos = pos-1;
            tmp->must[tmp->mustcount].zeichen = ch;
   			tmp->mustcount++;
				if (tmp->minlen < pos) tmp->minlen = pos;
            continue;
         }

         printf("Error line %i: Unknown statement '%s'.\n", linenr, cmd);
      }
   }

   fclose(f);

   if (tmp != NULL)
		checkBoxCheckEntry( tmp );
}



