/*
 * gnome-print-preview.c: A preview driver for GnomePrint that renders
 * into a GnomeCanvas and a control object for the pages rendered.
 *
 * Author:
 *   Miguel de Icaza (miguel@gnu.org)
 *
 * (C) 1999 International GNOME Support
 */
#include <config.h>
#include <gtk/gtk.h>
#include <string.h>
#include <math.h>

#include <libgnomeui/gnome-canvas-image.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <libgnomeprint/gnome-canvas-bpath.h>
#include <libgnomeprint/gnome-canvas-hacktext.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gt1-parset1.h>
#include <libart_lgpl/art_pixbuf.h>
#include <libgnome/gnome-paper.h>

typedef struct {
	double x, y;
} coord_t;

typedef struct {
	double r, g, b;
	int    opacity;
} color_t;

typedef struct {
	color_t      color;
	double       width;
	int          opacity;
	GdkCapStyle  linecap;
	GdkJoinStyle join;

	int          dash_n_values;
	double      *dash_values;
	double       dash_offset;
	GnomeCanvasBpathDef *bpath;

	GnomeFont    *font;

	double       affine [6];
	double       page_affine [6];
	double       base_affine [6];
	double       startup_page_affine [6];
} GraphicContext;

struct _GnomePrintPreviewPrivate {
	/* List of saved printing contexts */
	GSList *context_list;

	/* Current page displayed */
	int top_page;
	int current_page;

	/* The root group, with a translation setup */
	GnomeCanvasItem *root;

	/* The current page */
	GnomeCanvasItem *page;
	
	/* An array of GnomeCanvasItemGroups */
	GArray *pages;

	GHashTable *font_hash;

	GnomeCanvasItem *auto_hide_page_on_stroke;
};

#define gc(o) ((GraphicContext *)((o)->context_list->data))

static GnomePrintContextClass *print_preview_parent_class;

static void
gc_set_dash (GraphicContext *ctx, int n_values, double *values, double offset)
{
	ctx->dash_n_values = n_values;
	ctx->dash_values   = g_new (double, n_values);
	ctx->dash_offset   = offset;

	memcpy (ctx->dash_values, values, sizeof (double) * n_values);
}

static void
gc_clear_bpath (GraphicContext *ctx)
{
	gnome_canvas_bpath_def_unref (ctx->bpath);
	ctx->bpath = 0;
}

static void
gc_free (GraphicContext *ctx)
{
	if (ctx->dash_values)
		g_free (ctx->dash_values);
	if (ctx->font)
		gtk_object_unref (GTK_OBJECT (ctx->font));
	if (ctx->bpath)
		gc_clear_bpath (ctx);
	g_free (ctx);
}

static void
set_color (color_t *color, double r, double g, double b, int opacity)
{
	color->r = r;
	color->g = g;
	color->b = b;

	if (opacity != -1)
		color->opacity = opacity;
}

static gint32
color_rgba (color_t *color)
{
	return ((((guint)(color->r * 255)) << 24) |
		(((guint)(color->g * 255)) << 16) |
		(((guint)(color->b * 255)) << 8)) |
		color->opacity;
	
}

static void
dump_gc (GnomePrintPreview *pp)
{
	GnomePrintPreviewPrivate *priv = pp->priv;
	GraphicContext *g = gc (priv);
	
	printf ("color:      %08x\n", color_rgba (&g->color));
	printf ("width:      %g\n",   g->width);
	printf ("opacity:    %d\n",   g->opacity);
	printf ("linecap:    %d\n",   g->linecap);
	printf ("join:       %d\n",   g->join);
	printf ("affine:     %g %g %g %g %g %g\n",
		g->affine [0],
		g->affine [1],
		g->affine [2],
		g->affine [3],
		g->affine [4],
		g->affine [5]);

	if (g->bpath){
		int i;
		
		for (i = 0; i < g->bpath->n_bpath; i++){

		}
	}
}

/*
 * This just enables us to avoid doing a control of the page/display
 * when the job is finished (for the sample applications).
 *
 * ie, we are delaying showpage's item_hide on the page until the first
 * output command for the next page.
 */
static void
lazy_showpage_check (GnomePrintPreviewPrivate *priv)
{
	if (priv->auto_hide_page_on_stroke){
		gnome_canvas_item_hide (priv->auto_hide_page_on_stroke);
		priv->auto_hide_page_on_stroke = NULL;
	}
}

static int
gpp_newpath (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	
	if (gc (priv)->bpath)
		gc_clear_bpath (gc (priv));

	gc (priv)->bpath = gnome_canvas_bpath_def_new ();

	return 1;
}

static void
map (GnomePrintPreviewPrivate *priv, double *x, double *y)
{
	ArtPoint p;

	p.x = *x;
	p.y = *y;

	/* Apply postscript affine transform */
	art_affine_point (&p, &p, gc (priv)->affine);

	/* Map PS to canvas coordinates */
	art_affine_point (&p, &p, gc (priv)->page_affine);

	*x = p.x;
	*y = p.y;
}

static int
gpp_moveto (GnomePrintContext *pc, double x, double y)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	
	if (gc (priv)->bpath == NULL)
		gc (priv)->bpath = gnome_canvas_bpath_def_new ();

	map (priv, &x, &y);
	gnome_canvas_bpath_def_moveto (gc (priv)->bpath, x, y);
	return 1;
}

static int
gpp_lineto (GnomePrintContext *pc, double x, double y)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	g_return_val_if_fail (gc (priv)->bpath != NULL, -1);
	
	map (priv, &x, &y);
	gnome_canvas_bpath_def_lineto (gc (priv)->bpath, x, y);

	return 1;
}

static int
gpp_curveto (GnomePrintContext *pc,
	     double x1, double y1,
	     double x2, double y2,
	     double x3, double y3)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	g_return_val_if_fail (gc (priv)->bpath != NULL, -1);
	
	map (priv, &x1, &y1);
	map (priv, &x2, &y2);
	map (priv, &x3, &y3);
	gnome_canvas_bpath_def_curveto (gc (priv)->bpath, x1, y1, x2, y2, x3, y3);
	return 1;
}

static int
gpp_closepath (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	g_return_val_if_fail (gc (priv)->bpath != NULL, -1);

	gnome_canvas_bpath_def_closepath (gc (priv)->bpath);

	return 1;
}

static int
gpp_setrgbcolor (GnomePrintContext *pc, double r, double g, double b)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	set_color (&gc (priv)->color, r, g, b, - 1);
	return 1;
}

static void
close_path (GnomeCanvasBpathDef *bpath)
{
	if (bpath->moveto_idx == -1)
		return;
	gnome_canvas_bpath_def_closepath (bpath);
}

static int
gpp_stroke (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasItem *item;
	GraphicContext *g = gc (priv);
	int close;
	
	if (g->bpath == NULL)
		return 0;

	/*
	 * Special case: libart does not like stroking of lines
	 */
	close = 1;
	if (g->bpath->n_bpath == 2){
		if ((g->bpath->bpath [0].code == ART_MOVETO ||
		     g->bpath->bpath [0].code == ART_MOVETO_OPEN) &&
		    g->bpath->bpath [1].code == ART_LINETO)
			close = 0;
	}

	if (close)
		close_path (g->bpath);
	lazy_showpage_check (priv);
	item = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (priv->page),
		gnome_canvas_bpath_get_type (),
		"bpath",       g->bpath,
		"width_units", g->width,
		"cap_style",   g->linecap,
		"join_style",  g->join,
		"outline_color_rgba", color_rgba (&g->color),
		NULL);

	gc_clear_bpath (g);
	return 1;
}

static int
gpp_strokepath (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasItem *item;
	
	g_return_val_if_fail (gc (priv)->bpath != NULL, -1);

	close_path (gc (priv)->bpath);
	g_warning ("Strokepath is not tested");
	lazy_showpage_check (priv);
	item = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (priv->page),
		gnome_canvas_bpath_get_type (),
		"bpath",       gc (priv)->bpath,
		"width_units", gc (priv)->width,
		"cap_style",   gc (priv)->linecap,
		"join_style",  gc (priv)->join,
		"outline_color_rgba", color_rgba (&gc (priv)->color),
		NULL);
	gc_clear_bpath (gc (priv));
	return 1;
}

static int
gpp_fill (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasItem *item;
	
	g_return_val_if_fail (gc (priv)->bpath != NULL, -1);

	close_path (gc (priv)->bpath);
	lazy_showpage_check (priv);
	item = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (priv->page),
		gnome_canvas_bpath_get_type (),
		"bpath",       gc (priv)->bpath,
		"width_units", gc (priv)->width,
		"cap_style",   gc (priv)->linecap,
		"join_style",  gc (priv)->join,
		"fill_color_rgba", color_rgba (&gc (priv)->color),
		NULL);
	gc_clear_bpath (gc (priv));
	return 1;
}

static int
gpp_eofill (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	g_warning ("eofill operator not implemented yet");
	gc_clear_bpath (gc (priv));
	return 1;
}

static int
gpp_setlinewidth (GnomePrintContext *pc, double width)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	gc (priv)->width = width;

	return 1;
}

static int
gpp_setmiterlimit (GnomePrintContext *pc, double limit)
{
	g_warning ("WOOOOO  What does SetMiterLimit do?");
	return -1;
}

static int
gpp_setlinejoin (GnomePrintContext *pc, int jointype)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	switch (jointype){
	default:
		gc (priv)->join = GDK_JOIN_MITER;
		g_warning ("Need to support all linejoin methods");
	}
	return 1;
}

static int
gpp_setlinecap (GnomePrintContext *pc, int captype)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	switch (captype){
	default:
		gc (priv)->linecap = GDK_CAP_BUTT;
		g_warning ("Need to support all linecap methods");
	}
	return 1;
}

static int
gpp_setdash (GnomePrintContext *pc, int n_values, double *values, double offset)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	if (gc (priv)->dash_values)
		g_free (gc (priv)->dash_values);

	gc_set_dash (gc (priv), n_values, values, offset);

	return 1;
}

static int
gpp_setfont (GnomePrintContext *pc, GnomeFont *font)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	if (gc (priv)->font)
		gtk_object_unref (GTK_OBJECT (gc (priv)->font));

	gc (priv)->font = font;
	gtk_object_ref (GTK_OBJECT (font));

	return 1;
}

static int
gpp_concat (GnomePrintContext *pc, double matrix [6])
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	art_affine_multiply (gc (priv)->affine,  matrix, gc (priv)->affine);

	return 1;
}

static int
gpp_setmatrix (GnomePrintContext *pc, double matrix[6])
{
	g_warning ("setmatrix is deprecated");

	return -1;
}

static GnomeCanvasBpathDef *
gnome_canvas_bpath_def_duplicate (GnomeCanvasBpathDef *def)
{
	GnomeCanvasBpathDef *bpd;

	g_return_val_if_fail (def != NULL, NULL);

	bpd = g_new (GnomeCanvasBpathDef, 1);
	
	bpd->n_bpath = def->n_bpath;
	bpd->n_bpath_max = def->n_bpath;
	bpd->moveto_idx = def->moveto_idx;
	bpd->ref_count = 1;
	bpd->bpath = g_new (ArtBpath, def->n_bpath);

	memcpy (bpd->bpath, def->bpath, def->n_bpath * sizeof (ArtBpath));
	return bpd;
}

static int
gpp_gsave (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GraphicContext *newctx;
	GraphicContext *ctx = gc (priv);

	newctx = g_new (GraphicContext, 1);
	memcpy (newctx, ctx, sizeof (GraphicContext));

	if (newctx->dash_values){
		
		gc_set_dash (newctx, ctx->dash_n_values, ctx->dash_values, ctx->dash_offset);
	}

	if (newctx->bpath){
		GnomeCanvasBpathDef *newpath;

		newpath = gnome_canvas_bpath_def_duplicate (ctx->bpath);

		newctx->bpath = newpath;
	}
	
	priv->context_list = g_slist_prepend (priv->context_list, newctx);

	return 1;
}

static void
pop_state (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	gc_free (gc (priv));
	priv->context_list = g_slist_remove (priv->context_list, gc (priv));
}

static int
gpp_grestore (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	if (priv->context_list->next == NULL){
		g_warning ("grestore invoked on emtpy stack");
		return -1;
	}

	pop_state (pc);
	return 1;
}

static int
gpp_clip (GnomePrintContext *pc)
{
	g_warning ("Implement clip");
	return 1;
}

static int
gpp_eoclip (GnomePrintContext *pc)
{
	g_warning ("Implement eoclip");
	return 1;
}

static int
gpp_image (GnomePrintContext *pc, char *data, int width, int height, int rowstride, int bytes_per_pixel)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GraphicContext *g = gc (priv);
	GnomeCanvasItem *item;
	ArtPixBuf *pixbuf;
	int size, bpp;
	void *dup;
	
	/*
	 * We do convert gray scale images to RGB
	 */
	if (bytes_per_pixel == 1)
		bpp = 3;
	else
		bpp = bytes_per_pixel;
	
	size = width * height * bpp;
	dup = g_malloc (size);
	if (!dup)
		return -1;

	if (bytes_per_pixel == 3){
		memcpy (dup, data, size);
		pixbuf = art_pixbuf_new_rgb (dup, width, height, rowstride);
	} else if (bytes_per_pixel == 4){
		memcpy (dup, data, size);
		pixbuf = art_pixbuf_new_rgba (dup, width, height, rowstride);
	} else if (bytes_per_pixel == 1){
		char *source, *target;
		int  ix, iy;

		source = data;
		target = dup;

		for (iy = 0; iy < height; iy++){
			for (ix = 0; ix < width; ix++){
				*target++ = *source;
				*target++ = *source;
				*target++ = *source;
				source++;
			}
		}
		pixbuf = art_pixbuf_new_rgb (dup, width, height, rowstride * 3);
	} else
		return -1;

	lazy_showpage_check (priv);
	item = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (priv->page),
		gnome_canvas_image_get_type (),
		"pixbuf", pixbuf,
		"x",      0.0,
		"y",      0.0,
		"width",  1.0,
		"height", 1.0,
		"anchor", GTK_ANCHOR_SW,
		NULL);


	/* Apply the transformation for the image */
	{
		double transform [6];
		double ident [6];
		double flip [6];
		
		art_affine_identity (ident);
		art_affine_flip (flip, ident, FALSE, TRUE);
		art_affine_multiply (transform, flip, g->affine);
		art_affine_multiply (transform, transform, g->page_affine);
		
		gnome_canvas_item_affine_relative (item, transform);
	}
	
	return 1;
}

static int
gpp_grayimage (GnomePrintContext *pc, char *data, int width, int height, int rowstride)
{
	return  gpp_image (pc, data, width, height, rowstride, 1);
}

static int
gpp_rgbimage (GnomePrintContext *pc, char *data, int width, int height, int rowstride)
{
	return gpp_image (pc, data, width, height, rowstride, 3);
}


static int
gpp_textline (GnomePrintContext *pc, GnomeTextLine *line)
{
	g_warning ("Implement textline");

	return -1;
}

static int
gpp_showpage (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;

	g_array_append_val (priv->pages, priv->page);

	priv->auto_hide_page_on_stroke = priv->page;
	
	priv->page = gnome_canvas_item_new (
		gnome_canvas_root (pp->canvas),
		gnome_canvas_group_get_type (),
		"x", 0.0,
		"y", 0.0,
		NULL);

	/*
	 * Reset affine and path
	 */
	art_affine_identity (gc (priv)->affine);

	/*
	 * Kill current path
	 */
	gc_clear_bpath (gc (priv));
	return 0;
}

static Gt1LoadedFont *
fetch_font (GnomePrintPreviewPrivate *priv, char *pfb_file, char *afm_file)
{
	Gt1LoadedFont *font;
	
	if (!priv->font_hash){
		priv->font_hash = g_hash_table_new (g_str_hash, g_str_equal);
	}

	font = g_hash_table_lookup (priv->font_hash, pfb_file);
	if (font == (void *) -1)
		return NULL;

	if (font)
		return font;

	if (font == NULL){
		font = gt1_load_font (pfb_file);
		if (font == NULL)
			return NULL;

		g_hash_table_insert (priv->font_hash, g_strdup (pfb_file), font);
	}
	return font;
}

static int
gpp_show (GnomePrintContext *pc, char const *text)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasItem *item;
	double x, y;
	Gt1LoadedFont *gt1_font;
	double transform [6], inverse[6];
	GraphicContext *g = gc (priv);


	if (g->font == NULL){
		g_warning ("Show invoked without prior setfont");
		return -1;
	}

	if (g->bpath == NULL){
		g_warning ("You need to moveto first");
		return -1;
	}

	gt1_font = fetch_font (priv,
			       g->font->fontmap_entry->pfb_fn,
			       g->font->fontmap_entry->afm_fn);

	if (!gt1_font){
		g_warning ("Could not use the specified font");
		return -1;
	}

	/*
	 * The X and Y positions were computed to be the base
	 * with the translation already done for the new
	 * Postscript->Canvas translation
	 */
	x = g->bpath->bpath [0].x3;
	y = g->bpath->bpath [0].y3;

	/*
	 * Flip the angle
	 */
	memcpy (inverse, g->affine, sizeof (g->affine));
       
	inverse [1] = -inverse [1];
        inverse [2] = -inverse [2];
       
	/*
	inverse [0] *= pp->canvas->pixels_per_unit;
        inverse [3] *= pp->canvas->pixels_per_unit;
	*/
	art_affine_invert (transform, inverse);
	x =  g->bpath->bpath [0].x3 * transform[0] + g->bpath->bpath [0].y3  * transform[2] + transform[4];
	y =  g->bpath->bpath [0].x3 * transform[1] + g->bpath->bpath [0].y3  * transform[3] + transform[5]; 

	lazy_showpage_check (priv);
	item = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (priv->page),
		gnome_canvas_hacktext_get_type (),
		"x",           x,
		"y",           y,
		"text",        text,
		"font",        gt1_font,
		"size",        g->font->size,
		"fill_color_rgba", color_rgba (&g->color),
		NULL);

	/*
	 * Flip the angle
	 */
	memcpy (transform, g->affine, sizeof (g->affine));
       
	transform [1] = -transform [1];
	transform [2] = -transform [2];
      
	gnome_canvas_item_affine_absolute (item, transform);
	priv->current_page++;
	
	gc_clear_bpath (gc (priv));
	return 0;
}

static int
gpp_close (GnomePrintContext *pc)
{
	return 0;
}

static gboolean
kill_font (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	gt1_unload_font (value);
	return TRUE;
}

static void
gpp_finalize (GtkObject *object)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (object);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GSList *l;
	int i;

	for (l = priv->context_list; l; l = l->next)
		gc_free (l->data);
	g_slist_free (priv->context_list);
	
	if (pp->canvas)
		gtk_object_unref (GTK_OBJECT (pp->canvas));

	if (priv->font_hash){
		g_hash_table_foreach_remove (priv->font_hash, kill_font, NULL);
		g_hash_table_destroy (priv->font_hash);
	}

	for (i = 0; i < priv->pages->len; i++){
		GnomeCanvasItem *page;
		
		page = g_array_index (priv->pages, GnomeCanvasItem *, i);
		gtk_object_destroy (GTK_OBJECT (page));
	}
	
	g_array_free (priv->pages, TRUE);

	g_free (priv);
	GTK_OBJECT_CLASS (print_preview_parent_class)->finalize (object);
}

static void
gpp_class_init (GnomePrintPreviewClass *class)
{
	GtkObjectClass *object_class = (GtkObjectClass *) class;
	GnomePrintContextClass *pc_class = (GnomePrintContextClass *) class;

	print_preview_parent_class = gtk_type_class (gnome_print_context_get_type ());

	object_class->finalize = gpp_finalize;
	
	pc_class->newpath = gpp_newpath;
	pc_class->moveto = gpp_moveto;
	pc_class->lineto = gpp_lineto;
	pc_class->curveto = gpp_curveto;
	pc_class->closepath = gpp_closepath;
	pc_class->setrgbcolor = gpp_setrgbcolor;
	pc_class->fill = gpp_fill;
	pc_class->eofill = gpp_eofill;
	pc_class->setlinewidth = gpp_setlinewidth;
	pc_class->setmiterlimit = gpp_setmiterlimit;
	pc_class->setlinejoin = gpp_setlinejoin;
	pc_class->setlinecap = gpp_setlinecap;
	pc_class->setdash = gpp_setdash;
	pc_class->strokepath = gpp_strokepath;
	pc_class->stroke = gpp_stroke;
	pc_class->setfont = gpp_setfont;
	pc_class->show = gpp_show;
	pc_class->concat = gpp_concat;
	pc_class->setmatrix = gpp_setmatrix;
	pc_class->gsave = gpp_gsave;
	pc_class->grestore = gpp_grestore;
	pc_class->clip = gpp_clip;
	pc_class->eoclip = gpp_eoclip;
	pc_class->grayimage = gpp_grayimage;
	pc_class->rgbimage = gpp_rgbimage;
	pc_class->textline = gpp_textline;
	pc_class->showpage = gpp_showpage;
	
	pc_class->close = gpp_close;
}

static void
gpp_init (GnomePrintPreview *preview)
{
	GnomePrintPreviewPrivate *priv;
	GraphicContext *gc;
	
	priv = preview->priv = g_new0 (GnomePrintPreviewPrivate, 1);
	gc = g_new0 (GraphicContext, 1);
	priv->context_list = g_slist_prepend (priv->context_list, gc);
	
	set_color (&gc (priv)->color, 0.0, 0.0, 0.0, 0xff);
	
	gc (priv)->width = 1.0;
	gc (priv)->linecap = GDK_CAP_BUTT;
	gc (priv)->join = GDK_JOIN_MITER;

	priv->pages = g_array_new (FALSE, FALSE, sizeof (GnomeCanvasItem *));
}

/**
 * gnome_print_preview_construct:
 * @preview: the #GnomePrintPreview object to construct
 * @canvas: Canvas on which the preview will render
 * @paper_info: a GnomePaper information
 *
 * Constructs the @preview object.
 */
void
gnome_print_preview_construct (GnomePrintPreview *preview,
			       GnomeCanvas *canvas,
			       const GnomePaper *paper_info)
{
	g_return_if_fail (preview != NULL);
	g_return_if_fail (GNOME_IS_PRINT_PREVIEW (preview));
	g_return_if_fail (canvas != NULL);
	g_return_if_fail (GNOME_IS_CANVAS (canvas));
	g_return_if_fail (paper_info != NULL);

	gtk_object_ref (GTK_OBJECT (canvas));
	preview->canvas = canvas;

	if (getenv ("GNOME_PRINT_DEBUG_WIDE"))
		gnome_canvas_set_scroll_region (
			canvas,	-900, -900, 900, 900);
	else
		gnome_canvas_set_scroll_region (
			canvas, 0, 0,
			gnome_paper_pswidth (paper_info),
			gnome_paper_psheight (paper_info));

	preview->priv->root = gnome_canvas_item_new (
		gnome_canvas_root (preview->canvas),
		gnome_canvas_group_get_type (),
		"x", 0.0,
		"y", 0.0,
		NULL);

	preview->priv->page = gnome_canvas_item_new (
		gnome_canvas_root (preview->canvas),
		gnome_canvas_group_get_type (),
		"x", 0.0,
		"y", 0.0,
		NULL);

	/*
	 * Setup the affines
	 */
	art_affine_identity (gc (preview->priv)->affine);
	art_affine_translate (gc (preview->priv)->base_affine, 0, -gnome_paper_psheight (paper_info));
	art_affine_flip (gc (preview->priv)->page_affine,
			 gc (preview->priv)->base_affine,
			 FALSE, TRUE);
}

/**
 * gnome_print_preview_new:
 * @canvas: Canvas on which we display the print preview
 * @paper_size: A valid name for a paper size
 *
 * Creates a new PrintPreview object that use the @canvas GnomeCanvas 
 * as its rendering backend.
 *
 * Returns: A GnomePrintContext suitable for using with the GNOME print API.
 */
GnomePrintContext *
gnome_print_preview_new (GnomeCanvas *canvas, const char *paper_size)
{
	GnomePrintPreview *preview;
	const GnomePaper *paper_info;
	
	g_return_val_if_fail (canvas != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_CANVAS (canvas), NULL);
	g_return_val_if_fail (paper_size != NULL, NULL);

	paper_info = gnome_paper_with_name (paper_size);
	if (paper_info == NULL)
		g_return_val_if_fail (FALSE, NULL);

	preview = gtk_type_new (gnome_print_preview_get_type ());
	if (preview == NULL)
		return NULL;

	gnome_print_preview_construct (preview, canvas, paper_info);
	return GNOME_PRINT_CONTEXT (preview);
}

/**
 * gnome_print_preview_get_type:
 *
 * GTK type identification routine for #GnomePrintPreview
 *
 * Returns: The Gtk type for the #GnomePrintPreview object
 */
GtkType
gnome_print_preview_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"GnomePrintPreview",
			sizeof (GnomePrintPreview),
			sizeof (GnomePrintPreviewClass),
			(GtkClassInitFunc) gpp_class_init,
			(GtkObjectInitFunc) gpp_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		type = gtk_type_unique (gnome_print_context_get_type (), &info);
	}
	return type;
}

/*
 *
 * GNOME Print Preview Object
 *
 */
static GtkObjectClass *pjob_parent_class;

struct _GnomePrintPreviewJobPrivate {
	GArray          *pages;
	int              last_page_shown;
};

static void
pjob_finalize (GtkObject *object)
{
	GnomePrintPreviewJob *pjob = (GnomePrintPreviewJob *)(object);

	gtk_object_unref (GTK_OBJECT (pjob->canvas));

	g_array_free (pjob->priv->pages, TRUE);
	g_free (pjob->priv);
	
	GTK_OBJECT_CLASS (pjob_parent_class)->finalize (object);
}

static void
pjob_init (GtkObject *object)
{
	GnomePrintPreviewJob *pjob = GNOME_PRINT_PREVIEW_JOB (object);
	
	pjob->priv = g_new0 (GnomePrintPreviewJobPrivate, 1);
	pjob->priv->last_page_shown = -1;
}

static void
pjob_class_init (GtkObjectClass *object_class)
{
	object_class->finalize = pjob_finalize;

	pjob_parent_class = gtk_type_class (gtk_object_get_type ());
}

/**
 * gnome_print_preview_job_get_type:
 *
 * GTK type identification routine for #GnomePrintPreviewJob
 *
 * Returns: The Gtk type for the #GnomePrintPreviewJob object
 */
GtkType
gnome_print_preview_job_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"GnomePrintPreviewJob",
			sizeof (GnomePrintPreviewJob),
			sizeof (GnomePrintPreviewJobClass),
			(GtkClassInitFunc) pjob_class_init,
			(GtkObjectInitFunc) pjob_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		type = gtk_type_unique (gtk_object_get_type (), &info);
	}
	return type;
}

/**
 * gnome_print_preview_job_construct:
 * @pjob: the #GnomePrintPreviewJob object to construct
 * @preview: The #GnomePrintPreview object that this job controls
 *
 * Constructs the @pjob object.
 */
void
gnome_print_preview_job_construct (GnomePrintPreviewJob *pjob, GnomePrintPreview *preview)
{
	int i, size;
	
	g_return_if_fail (preview != NULL);
	g_return_if_fail (GNOME_IS_PRINT_PREVIEW (preview));
	g_return_if_fail (pjob != NULL);
	g_return_if_fail (GNOME_IS_PRINT_PREVIEW_JOB (pjob));

	lazy_showpage_check (preview->priv);
	gtk_object_ref (GTK_OBJECT (preview->canvas));
	pjob->canvas = preview->canvas;
	size = preview->priv->pages->len;
	pjob->priv->pages = g_array_new (size, 0, sizeof (GnomeCanvasItem *));

	for (i = 0; i < size; i++){
		GnomeCanvasItem *item;
		
		item = g_array_index (preview->priv->pages, GnomeCanvasItem *, i);
		g_array_append_val (pjob->priv->pages, item);
	}

}

/**
 * gnome_print_preview_get_job:
 * @preview: A #GnomePrintPreview object.
 *
 * This routine returns a #GnomePrintPreviewJob object which enabled
 * you to control which page is being displayed in the @preview canvas.
 *
 * Returns: The #GnomePrintPreviewJob object associated with @preview.
 */
GnomePrintPreviewJob *
gnome_print_preview_get_job (GnomePrintPreview *preview)
{
	GnomePrintPreviewJob *pjob;
	
	g_return_val_if_fail (preview != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_PREVIEW (preview), NULL);

	pjob = gtk_type_new (gnome_print_preview_job_get_type ());

	gnome_print_preview_job_construct (pjob, preview);

	return pjob;
}

/**
 * gnome_print_preview_job_page_visible:
 * @pjob: #GnomePrintPreview object
 * @page: page to display, or -1 to not display any page.
 *
 * Displays the page @page.  
 */
void
gnome_print_preview_job_page_show (GnomePrintPreviewJob *pjob, int page)
{
	GnomeCanvasItem *item;
	
	g_return_if_fail (pjob != NULL);
	g_return_if_fail (GNOME_IS_PRINT_PREVIEW_JOB (pjob));
	g_return_if_fail (page >= 0);

	if (page < 0)
		return;

	if (page >= pjob->priv->pages->len)
		return;

	if (pjob->priv->last_page_shown != -1){
		GnomeCanvasItem *item;
		
		item = g_array_index (pjob->priv->pages, GnomeCanvasItem *, pjob->priv->last_page_shown);
		gnome_canvas_item_hide (item);
	}
	
	item = g_array_index (pjob->priv->pages, GnomeCanvasItem *, page);
	gnome_canvas_item_show (item);

	pjob->priv->last_page_shown = page;
}

int
gnome_print_preview_job_num_pages (GnomePrintPreviewJob *pjob)
{
	g_return_val_if_fail (pjob != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_PREVIEW_JOB (pjob), 0);

	return pjob->priv->pages->len;
}

int
gnome_print_preview_job_current_page (GnomePrintPreviewJob *pjob)
{
	g_return_val_if_fail (pjob != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_PREVIEW_JOB (pjob), 0);

	return pjob->priv->last_page_shown;
}

