/* Bpath item type for GnomeCanvas widget
 *
 * GnomeCanvas is basically a port of the Tk toolkit's most excellent canvas widget.  Tk is
 * copyrighted by the Regents of the University of California, Sun Microsystems, and other parties.
 *
 * Copyright (C) 1998,1999 The Free Software Foundation
 *
 * Authors: Federico Mena <federico@nuclecu.unam.mx>
 *          Raph Levien <raph@acm.org>
 *          Lauris Kaplinski <lauris@ariman.ee>
 *          Miguel de Icaza <miguel@kernel.org>
 */

/* These includes are set up for standalone compile. If/when this codebase
   is integrated into libgnomeui, the includes will need to change. */
#include <math.h>
#include <string.h>
#include <gnome.h>
#include "gnome-canvas-bpath.h"
#include "gnome-canvas-bpath-util.h"
#include <libart_lgpl/art_rect.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_bpath.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_vpath_bpath.h>
#include <libart_lgpl/art_svp.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_vpath_dash.h>
#include <libart_lgpl/art_svp_wind.h>
#include <libart_lgpl/art_svp_point.h>

#define NUM_STATIC_POINTS 256	/* Number of static points to use to avoid allocating arrays */


#define GROW_BOUNDS(bx1, by1, bx2, by2, x, y) {	\
	if (x < bx1)				\
		bx1 = x;			\
						\
	if (x > bx2)				\
		bx2 = x;			\
						\
	if (y < by1)				\
		by1 = y;			\
						\
	if (y > by2)				\
		by2 = y;			\
}


enum {
	ARG_0,
	ARG_BPATH,
	ARG_FILL_COLOR,
	ARG_FILL_COLOR_GDK,
	ARG_FILL_COLOR_RGBA,
	ARG_OUTLINE_COLOR,
	ARG_OUTLINE_COLOR_GDK,
	ARG_OUTLINE_COLOR_RGBA,
	ARG_FILL_STIPPLE,
	ARG_OUTLINE_STIPPLE,
	ARG_WIDTH_PIXELS,
	ARG_WIDTH_UNITS,
	ARG_CAP_STYLE,
	ARG_JOIN_STYLE,
	ARG_WIND
};


static void gnome_canvas_bpath_class_init (GnomeCanvasBpathClass *class);
static void gnome_canvas_bpath_init       (GnomeCanvasBpath      *bpath);
static void gnome_canvas_bpath_destroy    (GtkObject               *object);
static void gnome_canvas_bpath_set_arg    (GtkObject               *object,
					   GtkArg                  *arg,
					   guint                    arg_id);
static void gnome_canvas_bpath_get_arg    (GtkObject               *object,
					   GtkArg                  *arg,
					   guint                    arg_id);

static void   gnome_canvas_bpath_update      (GnomeCanvasItem *item, double *affine, ArtSVP *clip_path, int flags);
static void   gnome_canvas_bpath_realize     (GnomeCanvasItem *item);
static void   gnome_canvas_bpath_unrealize   (GnomeCanvasItem *item);
static void   gnome_canvas_bpath_draw        (GnomeCanvasItem *item, GdkDrawable *drawable,
						int x, int y, int width, int height);
static double gnome_canvas_bpath_point       (GnomeCanvasItem *item, double x, double y,
						int cx, int cy, GnomeCanvasItem **actual_item);
static void   gnome_canvas_bpath_bounds      (GnomeCanvasItem *item, double *x1, double *y1, double *x2, double *y2);
static void   gnome_canvas_bpath_render      (GnomeCanvasItem *item, GnomeCanvasBuf *buf);


static GnomeCanvasItemClass *parent_class;


struct _GnomeCanvasBpathPriv {
	ArtBpath *bpath;		/* The shape */
};

GtkType
gnome_canvas_bpath_get_type (void)
{
	static GtkType bpath_type = 0;

	if (!bpath_type) {
		GtkTypeInfo bpath_info = {
			"GnomeCanvasBpath",
			sizeof (GnomeCanvasBpath),
			sizeof (GnomeCanvasBpathClass),
			(GtkClassInitFunc) gnome_canvas_bpath_class_init,
			(GtkObjectInitFunc) gnome_canvas_bpath_init,
			NULL, /* reserved_1 */
			NULL, /* reserved_2 */
			(GtkClassInitFunc) NULL
		};

		bpath_type = gtk_type_unique (gnome_canvas_item_get_type (), &bpath_info);
	}

	return bpath_type;
}

static void
gnome_canvas_bpath_class_init (GnomeCanvasBpathClass *class)
{
	GtkObjectClass *object_class;
	GnomeCanvasItemClass *item_class;

	object_class = (GtkObjectClass *) class;
	item_class = (GnomeCanvasItemClass *) class;

	parent_class = gtk_type_class (gnome_canvas_item_get_type ());

	/* when this gets checked into libgnomeui, change the
           GTK_TYPE_POINTER to GTK_TYPE_GNOME_CANVAS_BPATH, and add an
           entry to gnome-boxed.defs */
	gtk_object_add_arg_type ("GnomeCanvasBpath::bpath", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_BPATH);
	gtk_object_add_arg_type ("GnomeCanvasBpath::fill_color", GTK_TYPE_STRING, GTK_ARG_WRITABLE, ARG_FILL_COLOR);
	gtk_object_add_arg_type ("GnomeCanvasBpath::fill_color_gdk", GTK_TYPE_GDK_COLOR, GTK_ARG_READWRITE, ARG_FILL_COLOR_GDK);
	gtk_object_add_arg_type ("GnomeCanvasBpath::fill_color_rgba", GTK_TYPE_UINT, GTK_ARG_READWRITE, ARG_FILL_COLOR_RGBA);
	gtk_object_add_arg_type ("GnomeCanvasBpath::outline_color", GTK_TYPE_STRING, GTK_ARG_WRITABLE, ARG_OUTLINE_COLOR);
	gtk_object_add_arg_type ("GnomeCanvasBpath::outline_color_gdk", GTK_TYPE_GDK_COLOR, GTK_ARG_READWRITE, ARG_OUTLINE_COLOR_GDK);
	gtk_object_add_arg_type ("GnomeCanvasBpath::outline_color_rgba", GTK_TYPE_UINT, GTK_ARG_READWRITE, ARG_OUTLINE_COLOR_RGBA);
	gtk_object_add_arg_type ("GnomeCanvasBpath::fill_stipple", GTK_TYPE_GDK_WINDOW, GTK_ARG_READWRITE, ARG_FILL_STIPPLE);
	gtk_object_add_arg_type ("GnomeCanvasBpath::outline_stipple", GTK_TYPE_GDK_WINDOW, GTK_ARG_READWRITE, ARG_OUTLINE_STIPPLE);
	gtk_object_add_arg_type ("GnomeCanvasBpath::width_pixels", GTK_TYPE_UINT, GTK_ARG_WRITABLE, ARG_WIDTH_PIXELS);
	gtk_object_add_arg_type ("GnomeCanvasBpath::width_units", GTK_TYPE_DOUBLE, GTK_ARG_WRITABLE, ARG_WIDTH_UNITS);
	gtk_object_add_arg_type ("GnomeCanvasBpath::cap_style", GTK_TYPE_GDK_CAP_STYLE, GTK_ARG_READWRITE, ARG_CAP_STYLE);
	gtk_object_add_arg_type ("GnomeCanvasBpath::join_style", GTK_TYPE_GDK_JOIN_STYLE, GTK_ARG_READWRITE, ARG_JOIN_STYLE);
	gtk_object_add_arg_type ("GnomeCanvasBpath::wind", GTK_TYPE_ENUM, GTK_ARG_READWRITE, ARG_WIND);

	object_class->destroy = gnome_canvas_bpath_destroy;
	object_class->set_arg = gnome_canvas_bpath_set_arg;
	object_class->get_arg = gnome_canvas_bpath_get_arg;

	item_class->update = gnome_canvas_bpath_update;
	item_class->realize = gnome_canvas_bpath_realize;
	item_class->unrealize = gnome_canvas_bpath_unrealize;
	item_class->draw = gnome_canvas_bpath_draw;
	item_class->point = gnome_canvas_bpath_point;
	item_class->bounds = gnome_canvas_bpath_bounds;
	item_class->render = gnome_canvas_bpath_render;
}

static void
gnome_canvas_bpath_init (GnomeCanvasBpath *bpath)
{
	bpath->width = 0.0;
	bpath->cap = GDK_CAP_BUTT;
	bpath->join = GDK_JOIN_MITER;
	bpath->wind = GNOME_CANVAS_BPATH_NONZERO; /* default winding rule */
	bpath->priv = g_new (GnomeCanvasBpathPriv, 1);
	bpath->priv->bpath = NULL;
}

static void
gnome_canvas_bpath_destroy (GtkObject *object)
{
	GnomeCanvasBpath *bpath;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_BPATH (object));

	bpath = GNOME_CANVAS_BPATH (object);

	if (bpath->priv->bpath)
		art_free (bpath->priv->bpath);
	g_free (bpath->priv);
	
	if (bpath->fill_stipple)
		gdk_bitmap_unref (bpath->fill_stipple);

	if (bpath->outline_stipple)
		gdk_bitmap_unref (bpath->outline_stipple);

	if (bpath->fill_svp)
		art_svp_free (bpath->fill_svp);

	if (bpath->outline_svp)
		art_svp_free (bpath->outline_svp);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/* Find the bounding box of a bezier path. */
void
art_drect_bpath (ArtDRect *bbox, const ArtBpath *bpath)
{
	/* This implementation is a hack. The real thing needs to go
	   into libart. */
	ArtVpath *vpath;

	vpath = art_bez_path_to_vec (bpath, 0.25);
	art_vpath_bbox_drect (vpath, bbox);
	art_free (vpath);
}

/* Computes the bounding box of the bpath.  Assumes that the number of points in the bpath is
 * not zero.
 */
static void
get_bounds (GnomeCanvasBpath *bpath, double *bx1, double *by1, double *bx2, double *by2)
{
	double width;
	ArtDRect bbox;

	/* Compute bounds of bpath */

	art_drect_bpath (&bbox, bpath->priv->bpath);

	/* Add outline width */

	if (bpath->width_pixels)
		width = bpath->width / bpath->item.canvas->pixels_per_unit;
	else
		width = bpath->width;

	width /= 2.0;

	bbox.x0 -= width;
	bbox.y0 -= width;
	bbox.x1 += width;
	bbox.y1 += width;

	/* Done */

	*bx1 = bbox.x0;
	*by1 = bbox.y0;
	*bx2 = bbox.x1;
	*by2 = bbox.y1;
}

/* Computes the bounding box of the bpath, in canvas coordinates.  Assumes that the number of points in the bpath is
 * not zero.
 */
static void
get_bounds_canvas (GnomeCanvasBpath *bpath, double *bx1, double *by1, double *bx2, double *by2, double affine[6])
{
	GnomeCanvasItem *item;
	ArtDRect bbox_world;
	ArtDRect bbox_canvas;

	item = GNOME_CANVAS_ITEM (bpath);

	get_bounds (bpath, &bbox_world.x0, &bbox_world.y0, &bbox_world.x1, &bbox_world.y1);

	art_drect_affine_transform (&bbox_canvas, &bbox_world, affine);
	/* include 1 pixel of fudge */
	*bx1 = bbox_canvas.x0 - 1;
	*by1 = bbox_canvas.y0 - 1;
	*bx2 = bbox_canvas.x1 + 1;
	*by2 = bbox_canvas.y1 + 1;
}

/* Recalculates the canvas bounds for the bpath */
static void
recalc_bounds (GnomeCanvasBpath *bpath)
{
	GnomeCanvasItem *item;
	double x1, y1, x2, y2;
	int cx1, cy1, cx2, cy2;
	double dx, dy;

	item = GNOME_CANVAS_ITEM (bpath);

	if (bpath->priv->bpath == NULL) {
		item->x1 = item->y1 = item->x2 = item->y2 = 0;
		return;
	}

	/* Get bounds in world coordinates */

	get_bounds (bpath, &x1, &y1, &x2, &y2);

	/* Convert to canvas pixel coords */

	dx = dy = 0.0;
	gnome_canvas_item_i2w (item, &dx, &dy);

	gnome_canvas_w2c (item->canvas, x1 + dx, y1 + dy, &cx1, &cy1);
	gnome_canvas_w2c (item->canvas, x2 + dx, y2 + dy, &cx2, &cy2);
	item->x1 = cx1;
	item->y1 = cy1;
	item->x2 = cx2;
	item->y2 = cy2;

	/* Some safety fudging */

	item->x1--;
	item->y1--;
	item->x2++;
	item->y2++;

	gnome_canvas_group_child_bounds (GNOME_CANVAS_GROUP (item->parent), item);
}

/* Sets the shape of the bpath item to the specified BpathDef.
 */
static void
set_bpath (GnomeCanvasBpath *bpath, GnomeCanvasBpathDef *bpd)
{
	ArtBpath *art_bpath;

	art_bpath = art_new (ArtBpath, bpd->n_bpath + 1);
	memcpy (art_bpath, bpd->bpath, bpd->n_bpath * sizeof (ArtBpath));
	art_bpath[bpd->n_bpath].code = ART_END;
	bpath->priv->bpath = art_bpath;
}

/* Convenience function to set a GC's foreground color to the specified pixel value */
static void
set_gc_foreground (GdkGC *gc, gulong pixel)
{
	GdkColor c;

	if (!gc)
		return;

	c.pixel = pixel;
	gdk_gc_set_foreground (gc, &c);
}

/* Sets the stipple pattern for the specified gc */
static void
set_stipple (GdkGC *gc, GdkBitmap **internal_stipple, GdkBitmap *stipple, int reconfigure)
{
	if (*internal_stipple && !reconfigure)
		gdk_bitmap_unref (*internal_stipple);

	*internal_stipple = stipple;
	if (stipple && !reconfigure)
		gdk_bitmap_ref (stipple);

	if (gc) {
		if (stipple) {
			gdk_gc_set_stipple (gc, stipple);
			gdk_gc_set_fill (gc, GDK_STIPPLED);
		} else
			gdk_gc_set_fill (gc, GDK_SOLID);
	}
}

/* Recalculate the outline width of the bpath and set it in its GC */
static void
set_outline_gc_width (GnomeCanvasBpath *bpath)
{
	int width;

	if (!bpath->outline_gc)
		return;

	if (bpath->width_pixels)
		width = (int) bpath->width;
	else
		width = (int) (bpath->width * bpath->item.canvas->pixels_per_unit + 0.5);

	gdk_gc_set_line_attributes (bpath->outline_gc, width,
				    GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);
}

static gulong
set_color_from_rgba (GnomeCanvasItem *item, GdkGC *gc, gint32 rgba_color)
{
	gulong pix;

	pix = gnome_canvas_get_color_pixel (item->canvas, rgba_color);
	set_gc_foreground (gc, pix);

	return pix;
}

static void
gnome_canvas_bpath_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeCanvasItem *item;
	GnomeCanvasBpath *bp;
	GnomeCanvasBpathDef *bpd;
	GdkColor color;

	item = GNOME_CANVAS_ITEM (object);
	bp = GNOME_CANVAS_BPATH (object);

	switch (arg_id) {
	case ARG_BPATH:
		bpd = GTK_VALUE_POINTER (*arg);

		if (bp->priv->bpath) {
			art_free (bp->priv->bpath);
			bp->priv->bpath = NULL;
		}

		if (bpd)
			set_bpath (bp, bpd);

		gnome_canvas_item_request_update (item);
		break;

	case ARG_FILL_COLOR:
		if (gnome_canvas_get_color (item->canvas, GTK_VALUE_STRING (*arg), &color)) {
			bp->fill_set = TRUE;
			bp->fill_pixel = color.pixel;
			if (item->canvas->aa)
				bp->fill_rgba =
					((color.red & 0xff00) << 16) |
					((color.green & 0xff00) << 8) |
					(color.blue & 0xff00) |
					0xff;
			else
				set_gc_foreground (bp->fill_gc, bp->fill_pixel);
		} else {
			bp->fill_set = FALSE;
			bp->fill_rgba = 0;
		}

		gnome_canvas_item_request_update (item);
		break;

	case ARG_FILL_COLOR_GDK:
		bp->fill_set = TRUE;
		bp->fill_pixel = ((GdkColor *) GTK_VALUE_BOXED (*arg))->pixel;
		set_gc_foreground (bp->fill_gc, bp->fill_pixel);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_FILL_COLOR_RGBA:
		bp->fill_set = TRUE;
		bp->fill_rgba = GTK_VALUE_UINT (*arg);
		if (!item->canvas->aa)
			bp->fill_pixel = set_color_from_rgba (item, bp->fill_gc, bp->fill_rgba);

		/* should probably request repaint on the fill_svp */
		gnome_canvas_item_request_update (item);

		break;

	case ARG_OUTLINE_COLOR:
		if (gnome_canvas_get_color (item->canvas, GTK_VALUE_STRING (*arg), &color)) {
			bp->outline_set = TRUE;
			bp->outline_pixel = color.pixel;
			if (item->canvas->aa)
				bp->outline_rgba =
					((color.red & 0xff00) << 16) |
					((color.green & 0xff00) << 8) |
					(color.blue & 0xff00) |
					0xff;
			else
				set_gc_foreground (bp->outline_gc, bp->outline_pixel);
		} else {
			bp->outline_set = FALSE;
			bp->outline_rgba = 0;
		}

		gnome_canvas_item_request_update (item);
		break;

	case ARG_OUTLINE_COLOR_GDK:
		bp->outline_set = TRUE;
		bp->outline_pixel = ((GdkColor *) GTK_VALUE_BOXED (*arg))->pixel;
		set_gc_foreground (bp->outline_gc, bp->outline_pixel);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_OUTLINE_COLOR_RGBA:
		bp->outline_set = TRUE;
		bp->outline_rgba = GTK_VALUE_UINT (*arg);
		if (!item->canvas->aa)
			bp->outline_pixel = set_color_from_rgba (item, bp->outline_gc, bp->outline_rgba);

		/* should probably request repaint on the outline_svp */
		gnome_canvas_item_request_update (item);

		break;

	case ARG_FILL_STIPPLE:
		set_stipple (bp->fill_gc, &bp->fill_stipple, GTK_VALUE_BOXED (*arg), FALSE);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_OUTLINE_STIPPLE:
		set_stipple (bp->outline_gc, &bp->outline_stipple, GTK_VALUE_BOXED (*arg), FALSE);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_WIDTH_PIXELS:
		bp->width = GTK_VALUE_UINT (*arg);
		bp->width_pixels = TRUE;
		set_outline_gc_width (bp);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_WIDTH_UNITS:
		bp->width = fabs (GTK_VALUE_DOUBLE (*arg));
		bp->width_pixels = FALSE;
		set_outline_gc_width (bp);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_WIND:
		bp->wind = GTK_VALUE_ENUM (*arg);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_CAP_STYLE:
		bp->cap = GTK_VALUE_ENUM (*arg);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_JOIN_STYLE:
		bp->join = GTK_VALUE_ENUM (*arg);
		gnome_canvas_item_request_update (item);
		break;
	
	default:
		break;
	}
}

/* Allocates a GdkColor structure filled with the specified pixel, and puts it into the specified
 * arg for returning it in the get_arg method.
 */
static void
get_color_arg (GnomeCanvasBpath *bpath, gulong pixel, GtkArg *arg)
{
	GdkColor *color;

	color = g_new (GdkColor, 1);
	color->pixel = pixel;
	gdk_color_context_query_color (bpath->item.canvas->cc, color);
	GTK_VALUE_BOXED (*arg) = color;
}

GnomeCanvasBpathDef *
gnome_canvas_bpath_get_def (GnomeCanvasBpath *object)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_CANVAS_BPATH (object), NULL);

	if (!object->priv->bpath)
		return NULL;
			
	return gnome_canvas_bpath_def_new_from (object->priv->bpath);
}

static void
gnome_canvas_bpath_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeCanvasBpath *bp;

	bp = GNOME_CANVAS_BPATH (object);

	switch (arg_id) {
	case ARG_BPATH:
		if (bp->priv->bpath) {
			GTK_VALUE_POINTER (*arg) =
				gnome_canvas_bpath_def_new_from (
					bp->priv->bpath);
		} else
			GTK_VALUE_POINTER (*arg) = NULL;
		break;

	case ARG_FILL_COLOR_GDK:
		get_color_arg (bp, bp->fill_pixel, arg);
		break;
		
	case ARG_OUTLINE_COLOR_GDK:
		get_color_arg (bp, bp->outline_pixel, arg);
		break;

	case ARG_FILL_COLOR_RGBA:
		GTK_VALUE_UINT (*arg) = bp->fill_color;
		break;

	case ARG_FILL_STIPPLE:
		GTK_VALUE_BOXED (*arg) = bp->fill_stipple;
		break;

	case ARG_OUTLINE_STIPPLE:
		GTK_VALUE_BOXED (*arg) = bp->outline_stipple;
		break;

	case ARG_WIND:
		GTK_VALUE_ENUM (*arg) = bp->wind;
		break;

	case ARG_CAP_STYLE:
		GTK_VALUE_ENUM (*arg) = bp->cap;
		break;

	case ARG_JOIN_STYLE:
		GTK_VALUE_ENUM (*arg) = bp->join;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
gnome_canvas_bpath_render (GnomeCanvasItem *item,
			     GnomeCanvasBuf *buf)
{
	GnomeCanvasBpath *bpath;

	bpath = GNOME_CANVAS_BPATH (item);

	if (bpath->fill_svp != NULL)
		gnome_canvas_render_svp (buf, bpath->fill_svp, bpath->fill_rgba);

	if (bpath->outline_svp != NULL)
		gnome_canvas_render_svp (buf, bpath->outline_svp, bpath->outline_rgba);
}

static void
gnome_canvas_bpath_update (GnomeCanvasItem *item, double *affine, ArtSVP *clip_path, int flags)
{
	GnomeCanvasBpath *bpath;
	ArtVpath *vpath;
	ArtSVP *svp;
	double width;
	double x1, y1, x2, y2;
	ArtBpath *affine_bpath;
	ArtVpathDash *dash = NULL;

#ifdef USE_DASH_HACK
	double dots[4] = {12, 2, 6, 2};
	dash = g_new0 (ArtVpath, 1);
	dash->offset = 0;
	dash->n_dash = 4;
	dash->dash = dots;
#endif

	bpath = GNOME_CANVAS_BPATH (item);

	if (parent_class->update)
		(* parent_class->update) (item, affine, clip_path, flags);

	gnome_canvas_item_reset_bounds (item);
	
	affine_bpath = art_bpath_affine_transform (bpath->priv->bpath,
						   affine);
	
	vpath = art_bez_path_to_vec (affine_bpath, 0.25);
	
	art_free (affine_bpath);
	
	if (bpath->fill_set) {
		if (bpath->wind == GNOME_CANVAS_BPATH_RAW) {
			svp = art_svp_from_vpath (vpath);
		} else {
			ArtVpath *perturbed_vpath;
			ArtSVP *tmp_svp;
			ArtWindRule art_wind;
			
			perturbed_vpath = art_vpath_perturb (vpath);
			svp = art_svp_from_vpath (perturbed_vpath);
			art_free (perturbed_vpath);
			tmp_svp = art_svp_uncross (svp);
			art_svp_free (svp);
			if (bpath->wind == GNOME_CANVAS_BPATH_NONZERO) {
				art_wind = ART_WIND_RULE_NONZERO;
			} else {
				art_wind = ART_WIND_RULE_ODDEVEN;
			}
			svp = art_svp_rewind_uncrossed (tmp_svp, art_wind);
			art_svp_free (tmp_svp);
		}
		gnome_canvas_item_update_svp_clip (item, &bpath->fill_svp, svp, clip_path);
	}
	
	if (bpath->outline_set) {
		if (bpath->width_pixels)
			width = bpath->width;
		else
			width = bpath->width * item->canvas->pixels_per_unit;
		
		if (width < 0.5)
			width = 0.5;
		
		if (dash)
		{
			int i;
			ArtVpath *old = vpath;
			
			for (i = 0; i < dash->n_dash; i++)
				dash->dash[i] *= item->canvas->pixels_per_unit;
			
			vpath = art_vpath_dash (old, dash);
			art_free (old);
		}
		
		svp = art_svp_vpath_stroke (vpath,
					    gnome_canvas_join_gdk_to_art (bpath->join),
					    gnome_canvas_cap_gdk_to_art (bpath->cap),
					    width,
					    4,
					    0.25);
		
		gnome_canvas_item_update_svp_clip (item, &bpath->outline_svp, svp, clip_path);
	}
	art_free (vpath);
	
	if (!item->canvas->aa){
		set_outline_gc_width (bpath);
		set_gc_foreground (bpath->fill_gc, bpath->fill_pixel);
		set_gc_foreground (bpath->outline_gc, bpath->outline_pixel);
		set_stipple (bpath->fill_gc, &bpath->fill_stipple, bpath->fill_stipple, TRUE);
		set_stipple (bpath->outline_gc, &bpath->outline_stipple, bpath->outline_stipple, TRUE);

		get_bounds_canvas (bpath, &x1, &y1, &x2, &y2, affine);
		gnome_canvas_update_bbox (item, x1, y1, x2, y2);		
	}
}

static void
gnome_canvas_bpath_realize (GnomeCanvasItem *item)
{
	GnomeCanvasBpath *bpath;

	bpath = GNOME_CANVAS_BPATH (item);

	if (parent_class->realize)
		(* parent_class->realize) (item);

	bpath->fill_gc = gdk_gc_new (item->canvas->layout.bin_window);
	bpath->outline_gc = gdk_gc_new (item->canvas->layout.bin_window);
}

static void
gnome_canvas_bpath_unrealize (GnomeCanvasItem *item)
{
	GnomeCanvasBpath *bpath;

	bpath = GNOME_CANVAS_BPATH (item);

	gdk_gc_unref (bpath->fill_gc);
	gdk_gc_unref (bpath->outline_gc);

	if (parent_class->unrealize)
		(* parent_class->unrealize) (item);
}

/*
 * Converts an array of world coordinates into an array of canvas pixel coordinates.  Takes in the
 * item->world deltas and the drawable deltas.
 */
static void
item_to_canvas (GnomeCanvas *canvas, double *item_coords, GdkPoint *canvas_coords, int num_points,
		double i2c[6])
{
	int i;
	ArtPoint pi, pc;

#ifdef VERBOSE
	{
		char str[128];
		art_affine_to_string (str, i2c);
		g_print ("bpath item_to_canvas %s\n", str);
	}
#endif

	for (i = 0; i < num_points; i++) {
		pi.x = item_coords[i * 2];
		pi.y = item_coords[i * 2 + 1];
		art_affine_point (&pc, &pi, i2c);
		canvas_coords->x = floor (pc.x + 0.5);
		canvas_coords->y = floor (pc.y + 0.5);
		canvas_coords++; 
	}
}

static GdkPoint *
vpath_to_points (ArtVpath *path, int *n)
{
	ArtVpath *p = path;
	GdkPoint *points;
	int i;
	
	*n = 0;
	for (p = path; p->code != ART_END; p++)
		(*n)++;

	points = g_new (GdkPoint, *n);

	for (i = 0, p = path; p->code != ART_END; p++, i++){
		points [i].x = p->x;
		points [i].y = p->y;
	}

	return points;
}

static void
gnome_canvas_bpath_draw (GnomeCanvasItem *item, GdkDrawable *drawable,
			   int x, int y, int width, int height)
{
	GnomeCanvasBpath *bpath, *affine_bpath;
	ArtVpath *vpath;
	double i2c [6];
	int n;
	GdkPoint *points;

	bpath = GNOME_CANVAS_BPATH (item);

	gnome_canvas_item_i2c_affine (item, i2c);
	i2c[4] -= x;
	i2c[5] -= y;
	affine_bpath = art_bpath_affine_transform (bpath->priv->bpath, i2c);
	vpath = art_bez_path_to_vec (affine_bpath, 0.25);
	art_free (affine_bpath);
	points = vpath_to_points (vpath, &n);

	if (bpath->fill_set) {
		if (bpath->fill_stipple)
			gnome_canvas_set_stipple_origin (item->canvas, bpath->fill_gc);

		gdk_draw_polygon (drawable, bpath->fill_gc, TRUE, points, n);
	}

	if (bpath->outline_set) {
		if (bpath->outline_stipple)
			gnome_canvas_set_stipple_origin (item->canvas, bpath->outline_gc);

		gdk_draw_polygon (drawable, bpath->outline_gc, FALSE, points, n);
	}

	g_free (points);
}

static double
gnome_canvas_bpath_point (GnomeCanvasItem *item, double x, double y,
			    int cx, int cy, GnomeCanvasItem **actual_item)
{
	GnomeCanvasBpath *bpath;
	double dist;
	int wind;

	bpath = GNOME_CANVAS_BPATH (item);

	/* todo: update? */
	if (bpath->fill_set) {
		wind = art_svp_point_wind (bpath->fill_svp, cx, cy);
		if (wind) {
			*actual_item = item;
			return 0.0;
		}
	}

	if (bpath->outline_set) {
		wind = art_svp_point_wind (bpath->outline_svp, cx, cy);
		if (wind) {
			*actual_item = item;
			return 0.0;
		}
	}

	if (bpath->outline_set) {
		dist = art_svp_point_dist (bpath->outline_svp, cx, cy);
	} else if (bpath->fill_set) {
		dist = art_svp_point_dist (bpath->fill_svp, cx, cy);
	} else {
		return 1e12;
	}

#ifdef VERBOSE
	g_print ("dist = %g\n", dist);
#endif
	/* should dist be scaled by zoom somehow? */
	*actual_item = item;
	return dist;
}

static void
gnome_canvas_bpath_bounds (GnomeCanvasItem *item, double *x1, double *y1, double *x2, double *y2)
{
	GnomeCanvasBpath *bpath;

	g_return_if_fail (item != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_BPATH (item));

	bpath = GNOME_CANVAS_BPATH (item);

	if (bpath->priv->bpath == NULL) {
		*x1 = *y1 = *x2 = *y2 = 0.0;
		return;
	}

	get_bounds (bpath, x1, y1, x2, y2);
}

