/* vi:set ts=8 sts=0 sw=8:
 * $Id: search.c,v 1.8 1999/01/18 20:07:12 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include "main.h"
#include "doc.h"
#include "dialog.h"
#include "misc.h"
#include "search.h"


/*** local function prototypes ***/
static void goto_line_dialog_create(win_t *w);
static void goto_line_by_point(doc_t *d, int point);
static void goto_line_execute(GtkWidget *wgt, gpointer cbdata);
static void replace_widgets_create(win_t *w);
static void search_dialog_create(win_t *w, bool_t do_replace);
static void search_replace_common(win_t *w, bool_t do_replace);
static void search_dialog_destroy(GtkWidget *wgt, gpointer cbdata);
static void search_execute(GtkWidget *wgt, gpointer cbdata);
static bool_t scrollbar_update(GtkText *txt, int tl, int ln);


/*** global function definitions ***/
/*
 * PUBLIC: search_cb
 *
 * main callback routine from pulldown menu
 */
void 
search_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_replace_common((win_t *)cbdata, FALSE);
} /* search_cb */


/*
 * PUBLIC: search_replace_cb
 *
 * menu callback
 */
void 
search_replace_cb(GtkWidget *wgt, gpointer cbdata)
{
	search_replace_common((win_t *)cbdata, TRUE);
} /* search_replace_cb */


/*
 * PUBLIC: search_again_cb
 *
 * menu callback
 */
void 
search_again_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (w->stxt == NULL || w->stxt[0] == '\0')
		return;

	search_execute(NULL, w);
} /* search_again_cb */


/*
 * PUBLIC: goto_line_cb
 *
 * public callback routine to popup dialog box asking user which line number
 * to go to.
 */
void 
goto_line_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (w->search)
		gdk_window_raise(w->search->window);
	else
		goto_line_dialog_create(w);
} /* goto_line_cb */



/*** local function definitions ***/
/*
 * PRIVATE: replace_widgets_create
 *
 * creates the widgets for 'replace text'
 */
static void
replace_widgets_create(win_t *w)
{
	GtkWidget *vbox, *hbox, *tmp;

	/* search & replace information (replace text) */
	tmp = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(tmp), 6);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(tmp), vbox);

	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w->search)->vbox), tmp,
		TRUE, TRUE, 0);
	hbox = gtk_hbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	tmp = gtk_label_new("Replace:");
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 2);
	w->rtxt_wgt = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), w->rtxt_wgt, TRUE, TRUE, 2);

	tmp = gtk_check_button_new_with_label("Prompt before replacing");
	gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);
} /* replace_widgets_create */


/*
 * PRIVATE: goto_line_dialog_create
 *
 * creates the search dialog box.
 */
static void
goto_line_dialog_create(win_t *w)
{
	GtkWidget *tmp, *vbox, *hbox;

	if (w->search != NULL)
		search_dialog_destroy(NULL, w);

	/* top level dialog window */
	w->search = gtk_dialog_new();
	gtk_window_set_policy(GTK_WINDOW(w->search), TRUE, TRUE, TRUE);
	gtk_window_set_title(GTK_WINDOW(w->search), "Goto Line #...");
	gtk_signal_connect(GTK_OBJECT(w->search), "destroy",
                GTK_SIGNAL_FUNC(search_dialog_destroy), w);

	vbox = GTK_DIALOG(w->search)->vbox;

	/* the frame that holds everything */
	tmp = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(tmp), 6);
	gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(tmp), vbox);

	/* create label and entry */
	hbox = gtk_hbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	tmp = gtk_label_new("Line #:");
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 2);
	w->stxt_wgt = gtk_entry_new_with_max_length(16);
	gtk_box_pack_start(GTK_BOX(hbox), w->stxt_wgt, TRUE, TRUE, 2);

	/* lastly, the buttons */
	tmp = gtk_button_new_with_label(BUTTON_OK);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w->search)->action_area),
		tmp, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(goto_line_execute), w);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(search_dialog_destroy), w);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	tmp = gtk_button_new_with_label(BUTTON_CANCEL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w->search)->action_area),
		tmp, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(search_dialog_destroy), w);

	gtk_widget_show_all(w->search);
} /* goto_line_dialog_create */


/*
 * PRIVATE: goto_line_execute
 *
 * goes to line by line number
 */
static void
goto_line_execute(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	char *buf, *haystack, *needle;
	doc_t *d;
	int numlines, linenum;
	int a, b, len;

	linenum = atoi(gtk_entry_get_text(GTK_ENTRY(w->stxt_wgt)));

	GNPDBG_SEARCH(("goto_line_execute: linenum = %d\n", linenum));
	numlines = 1;
	d = (doc_t *)(w->curdoc);

	len = gtk_text_get_length(GTK_TEXT(d->data));
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 1, len);

	a = 1;
	b = len;
	haystack = buf;
	do {
		needle = strchr(haystack, '\n');
		if (needle) {
			haystack = needle + 1;
			if (linenum == numlines)
				b = needle - buf + 1;
			numlines++;
			if (linenum == numlines)
				a = needle - buf + 1;
		}
	} while (needle != NULL);

	g_free(buf);
	GNPDBG_SEARCH(("goto_line_execute: numlines = %d\n", numlines));
	if (scrollbar_update(GTK_TEXT(d->data), numlines, linenum)) {
		gtk_text_set_point(GTK_TEXT(d->data), a + 1);
#ifdef GTK_HAVE_FEATURES_1_1_0
		gtk_editable_set_position(GTK_EDITABLE(d->data), a + 1);
#else
		GTK_EDITABLE(d->data)->current_pos = a + 1;
#endif
		gtk_editable_select_region(GTK_EDITABLE(d->data), a, b);
	}
} /* goto_line_execute */


/*
 * PRIVATE: goto_line_by_point
 *
 * goes to line by point/index into text.  called only from search_execute(),
 * after text has been found.
 */
static void
goto_line_by_point(doc_t *d, int point)
{
	char *buf, *haystack, *needle;
	int numlines, linenum;
	int len;
	bool_t found;

	GNPDBG_SEARCH(("goto_line_by_point: point = %d\n", point));
	len = gtk_text_get_length(GTK_TEXT(d->data));
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 1, len);

	found = FALSE;
	linenum = 1;
	numlines = 1;
	haystack = buf;
	do {
		needle = strchr(haystack, '\n');
		if (needle) {
			if (!found && needle - buf > point) {
				linenum = numlines;
				found = TRUE;
			}

			haystack = needle + 1;
			numlines++;
		}
	} while (needle != NULL);

	g_free(buf);
	GNPDBG_SEARCH(("goto_line_by_point: numlines = %d, linenum = %d\n",
			numlines, linenum));
	(void)scrollbar_update(GTK_TEXT(d->data), numlines, linenum);
} /* goto_line_by_point */


/*
 * PRIVATE: search_dialog_create
 *
 * creates the search dialog box.
 */
static void
search_dialog_create(win_t *w, bool_t do_replace)
{
	GtkWidget *tmp, *vbox, *hbox;

	if (w->search != NULL) {
		gdk_window_raise(w->search->window);
		return;
	}

	/* top level dialog window */
	w->soptions = 0;
	w->search = gtk_dialog_new();
	gtk_window_set_policy(GTK_WINDOW(w->search), TRUE, TRUE, TRUE);
	gtk_window_set_title(GTK_WINDOW(w->search),
		(do_replace) ? "Search and Replace Text" : "Search Text");
	gtk_signal_connect(GTK_OBJECT(w->search), "destroy",
                GTK_SIGNAL_FUNC(search_dialog_destroy), w);

	vbox = GTK_DIALOG(w->search)->vbox;

	/* the frame that holds everything */
	tmp = gtk_frame_new(NULL);
	gtk_container_border_width(GTK_CONTAINER(tmp), 6);
	gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(tmp), vbox);

	/* create search label and entry */
	hbox = gtk_hbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	tmp = gtk_label_new("Search:");
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 2);
	w->stxt_wgt = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), w->stxt_wgt, TRUE, TRUE, 2);

	/* create a couple radio buttons for starting search position */
	w->startcur = gtk_radio_button_new_with_label(NULL,
			"Start at cursor position");
	gtk_box_pack_start(GTK_BOX(vbox), w->startcur, TRUE, TRUE, 0);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(w->startcur), TRUE);
	w->startbeg = gtk_radio_button_new_with_label(
			gtk_radio_button_group(GTK_RADIO_BUTTON(w->startcur)),
			"Start at beginning of the document");
	gtk_box_pack_start(GTK_BOX(vbox), w->startbeg, TRUE, TRUE, 0);

	/* case sensitive or not */
	w->casesens = gtk_check_button_new_with_label("Case sensitive");
	gtk_box_pack_start(GTK_BOX(vbox), w->casesens, TRUE, TRUE, 2);

#ifdef NOT_YET
	/* regular expression */
	w->regexp = gtk_check_button_new_with_label("Regular Expression");
	gtk_box_pack_start(GTK_BOX(vbox), w->regexp, TRUE, TRUE, 2);
#endif

	/* create replace widgets if needed */
	if (do_replace)
		replace_widgets_create(w);

	/* lastly, the buttons */
	tmp = gtk_button_new_with_label(BUTTON_OK);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w->search)->action_area),
		tmp, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(search_execute), w);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	tmp = gtk_button_new_with_label(BUTTON_CANCEL);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(w->search)->action_area),
		tmp, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(search_dialog_destroy), w);

	gtk_widget_show_all(w->search);
} /* search_dialog_create */


/*
 * PRIVATE: search_dialog_destroy
 *
 * zap!
 */
static void
search_dialog_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (GTK_WIDGET_VISIBLE(w->search))
		gtk_widget_destroy(w->search);

	w->search = NULL;
} /* search_dialog_destroy */


/*
 * PRIVATE: search_replace_common
 *
 * common routine for search and search/replace callbacks.
 */
static void
search_replace_common(win_t *w, bool_t do_replace)
{
	g_assert(w != NULL);

	if (!w->search)
		search_dialog_create(w, do_replace);

	if (do_replace)
		SET_REPLACE(w);
	else
		CLEAR_REPLACE(w);

	search_execute(NULL, w);
} /* search_replace_common */


/*
 * PRIVATE: search_execute
 *
 * actually search for the text.
 */
static void
search_execute(GtkWidget *wgt, gpointer cbdata)
{
	bool_t oldchanged;
	int idx;
	int startpos;		/* starting position/index */
	int slen;		/* search text length */
	int sblen;		/* search buffer length */
	char *searchbuf;	/* text widget represented as one string */
	char *match;
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);

	/* get text to search for */
	w->stxt = gtk_entry_get_text(GTK_ENTRY(w->stxt_wgt));
	slen = strlen(w->stxt);
	if (slen < 1)
		return;	/* don't do anything if empty */

	/* get search start position */
	if (GTK_TOGGLE_BUTTON(w->startcur)->active) {
		/*
		startpos = GTK_EDITABLE(d->data)->current_pos;
		startpos = gtk_text_get_point(GTK_TEXT(d->data));
		*/
		startpos = GTK_EDITABLE(d->data)->selection_end_pos;
		GNPDBG_SEARCH(("search_execute: starting at cur pos = %d\n",
				startpos));
	} else {
		startpos = 0;
		GNPDBG_SEARCH(("search_execute: starting at beginning\n"));
	}

	/*
	 * get the text in the text widget...
	 * yikes, this will take a lot of memory.  perhaps in the future, GTK
	 * will have a better text widget.  it's either consume lots of memory,
	 * or have a search that's so slow and pathetic, it's not even useful.
	 */
	searchbuf = gtk_editable_get_chars(
			GTK_EDITABLE(d->data),
			startpos,
			gtk_text_get_length(GTK_TEXT(d->data)));

	sblen = strlen(searchbuf);
	if (sblen < slen) {	/* nothing to do if we don't have much text */
		g_free(searchbuf);
		return;
	}

	/* search for the text */
	GNPDBG_SEARCH(("search_execute: looking for '%s' at startpos = %d\n",
			w->stxt, startpos));
	if (GTK_TOGGLE_BUTTON(w->casesens)->active)
		match = strstr(searchbuf, w->stxt);
	else
		match = strcasestr(searchbuf, w->stxt);

	if (!match) {
		int len;
		char *msg;

		len = strlen(w->stxt) + 37;
		msg = (char *)g_malloc(len);

		sprintf(msg, " '%s' not found (reached end of file) ", w->stxt);
		(void)do_dialog_ok( "Not found", msg);
		g_free(msg);
		g_free(searchbuf);
		return;
	}

	/* found the text, now go to the location and highlight it */
	idx = (match - searchbuf) + startpos;
	GNPDBG_SEARCH(("search_execute: found match at idx = %d\n", idx));
	gtk_text_set_point(GTK_TEXT(d->data), idx);
	oldchanged = d->changed;
	gtk_text_freeze(GTK_TEXT(d->data));
	gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL, " ", 1);
	gtk_text_backward_delete(GTK_TEXT(d->data), 1);
	d->changed = oldchanged;
	gtk_editable_select_region(GTK_EDITABLE(d->data), idx, idx + slen);
#define TEXT_LENGTH(t)              ((t)->text_end - (t)->gap_size) 
	GNPDBG_SEARCH(("search_execute: text len = %d, idx + slen = %d\n",
			TEXT_LENGTH(GTK_TEXT(d->data)), idx+slen));
	gtk_text_set_point(GTK_TEXT(d->data), idx + slen);

	goto_line_by_point(d, idx + slen);
	gtk_text_thaw(GTK_TEXT(d->data));

	g_free(searchbuf);
#if NOTYET
	if (DO_REPLACE(w)) {
		w->rtxt = gtk_entry_get_text(GTK_ENTRY(w->rtxt_wgt));
	}
#endif
} /* search_execute */


/*
 * PRIVATE: scrollbar_update
 *
 * scrolls the text widget to the line number specified.
 *
 * tl: total lines
 * ln: line number to scroll/go to
 *
 * returns TRUE if updated, FALSE if not.
 */
static bool_t
scrollbar_update(GtkText *txt, int tl, int ln)
{
	float value;

	GNPDBG_SEARCH(("scrollbar_update: tl=%d, ln=%d\n", tl, ln));
	if (tl < 3)
		return FALSE;

	if (ln > tl) {
		char buf[32];

		g_snprintf(buf, 32, " File only has %d lines! ", tl);
		(void)do_dialog_ok("End of File!", buf);
		return FALSE;
	}

	value = (ln * GTK_ADJUSTMENT(txt->vadj)->upper) /
		tl - GTK_ADJUSTMENT(txt->vadj)->page_increment;

	gtk_adjustment_set_value(GTK_ADJUSTMENT(txt->vadj), value);
	return TRUE;
} /* scrollbar_update */


/* the end */
