/*
 * @(#)SimpleTimeZone.java	1.13 97/07/15
 *
 * (C) Copyright Taligent, Inc. 1996 - All Rights Reserved
 * (C) Copyright IBM Corp. 1996 - All Rights Reserved
 *
 * Portions copyright (c) 1996 Sun Microsystems, Inc. All Rights Reserved.
 *
 *   The original version of this source code and documentation is copyrighted
 * and owned by Taligent, Inc., a wholly-owned subsidiary of IBM. These
 * materials are provided under terms of a License Agreement between Taligent
 * and Sun. This technology is protected by multiple US and International
 * patents. This notice and attribution to Taligent may not be removed.
 *   Taligent is a registered trademark of Taligent, Inc.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

package java.util;

import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * <code>SimpleTimeZone</code> is a concrete subclass of <code>TimeZone</code>
 * that represents a time zone for use with a Gregorian
 * calendar. This simple class does not handle historical
 * changes, and has limited rules.
 *
 * <P>
 * Use a negative value for <code>dayOfWeekInMonth</code> to indicate that
 * <code>SimpleTimeZone</code> should count from the end of the month backwards.
 * For example, Daylight Savings Time ends at the last
 * (dayOfWeekInMonth = -1) Sunday in October, at 2 AM in standard time.
 *
 * @see          Calendar
 * @see          GregorianCalendar
 * @see          TimeZone
 * @version      1.13 07/15/97
 * @author       David Goldsmith, Mark Davis, Chen-Lieh Huang, Alan Liu
 */
public class SimpleTimeZone extends TimeZone {
    /**
     * Constructs a SimpleTimeZone with the given base time zone offset
     * from GMT and time zone ID. Timezone IDs can be obtained from
     * TimeZone.getAvailableIDs. Normally you should use TimeZone.getDefault
     * to construct a TimeZone.
     * @param rawOffset the given base time zone offset to GMT.
     * @param ID the time zone ID which is obtained from
     * TimeZone.getAvailableIDs.
     */
    public SimpleTimeZone(int rawOffset, String ID)
    {
        this.rawOffset = rawOffset;
        setID (ID);
    }

    /**
     * Constructs a SimpleTimeZone with the given base time zone offset
     * from GMT, time zone ID, time to start and end the daylight time.
     * Timezone IDs can be obtained from TimeZone.getAvailableIDs.
     * Normally you should use TimeZone.getDefault to create a TimeZone.
     * For a time zone that does not use daylight saving time, do not
     * use this constructor; instead you should use
     * SimpleTimeZone(rawOffset, ID).
     * @param rawOffset the given base time zone offset to GMT.
     * @param ID the time zone ID which is obtained from
     * TimeZone.getAvailableIDs.
     * @param startMonth the daylight savings starting month. Month is 0-based.
     * eg, 0 for January.
     * @param startDayOfWeekInMonth the daylight savings starting
     * day-of-week-in-month. Please see the member description for an example.
     * @param startDayOfWeek the daylight savings starting day-of-week.
     * Please see the member description for an example.
     * @param startTime the daylight savings starting time. Please see the
     * member description for an example.
     * @param endMonth the daylight savings ending month. Month is 0-based.
     * eg, 0 for January.
     * @param endDayOfWeekInMonth the daylight savings ending
     * day-of-week-in-month. Please see the member description for an example.
     * @param endDayOfWeek the daylight savings ending day-of-week. Please see
     * the member description for an example.
     * @param endTime the daylight savings ending time. Please see the member
     * description for an example.
     */
    public SimpleTimeZone(int rawOffset, String ID, int startMonth,
              int startDayOfWeekInMonth, int startDayOfWeek, int startTime,
              int endMonth, int endDayOfWeekInMonth, int endDayOfWeek, int endTime)
    {
        setID (ID);
        this.rawOffset = rawOffset;
        this.startMonth = startMonth;
        this.startDay = startDayOfWeekInMonth;
        this.startDayOfWeek = startDayOfWeek;
        this.startTime = startTime;
        this.endMonth = endMonth;
        this.endDay = endDayOfWeekInMonth;
        this.endDayOfWeek = endDayOfWeek;
        this.endTime = endTime;
        this.useDaylight = true;
    }

    /**
     * Sets the daylight savings starting year.
     * @param year the daylight savings starting year.
     */
    public void setStartYear(int year)
    {
        startYear = year;
        useDaylight = true;
    }

    /**
     * Sets the daylight savings starting rule. For example, Daylight Savings
     * Time starts at the first Sunday in April, at 2 AM in standard time.
     * Therefore, you can set the start rule by calling:
     * setStartRule(TimeFields.APRIL, 1, TimeFields.SUNDAY, 2*60*60*1000);
     * @param month the daylight savings starting month. Month is 0-based.
     * eg, 0 for January.
     * @param dayOfWeekInMonth the daylight savings starting
     * day-of-week-in-month. Please see the member description for an example.
     * @param dayOfWeek the daylight savings starting day-of-week. Please see
     * the member description for an example.
     * @param time the daylight savings starting time. Please see the member
     * description for an example.
     */
    public void setStartRule(int month, int dayOfWeekInMonth, int dayOfWeek,
                             int time)
    {
        startMonth = month;
        startDay = dayOfWeekInMonth;
        startDayOfWeek = dayOfWeek;
        startTime = time;
        useDaylight = true;
    }

    /**
     * Sets the daylight savings ending rule. For example, Daylight Savings
     * Time ends at the last (-1) Sunday in October, at 2 AM in standard time.
     * Therefore, you can set the end rule by calling:
     * setEndRule(TimeFields.OCTOBER, -1, TimeFields.SUNDAY, 2*60*60*1000);
     * @param month the daylight savings ending month. Month is 0-based.
     * eg, 0 for January.
     * @param dayOfWeekInMonth the daylight savings ending
     * day-of-week-in-month. Please see the member description for an example.
     * @param dayOfWeek the daylight savings ending day-of-week. Please see
     * the member description for an example.
     * @param time the daylight savings ending time. Please see the member
     * description for an example.
     */
    public void setEndRule(int month, int dayOfWeekInMonth, int dayOfWeek,
                           int time)
    {
        endMonth = month;
        endDay = dayOfWeekInMonth;
        endDayOfWeek = dayOfWeek;
        endTime = time;
        useDaylight = true;
    }

    /**
     * Overrides TimeZone
     * Gets the time zone offset, for current date, modified in case of
     * daylight savings. This is the offset to add *to* UTC to get local time.
     *
     * Please see TimeZone.getOffset for descriptions of parameters.
     * @see TimeZone#getOffset
     */
    public int getOffset(int era, int year, int month, int day, int dayOfWeek,
                         int millis)
    {
        int result = rawOffset;

        // Bail out if we are before the onset of daylight savings time
        if (year < startYear || era != GregorianCalendar.AD)
            return result;

        // Check for southern hemisphere.  We assume that the start and end
        // month are different.
        boolean southern = (startMonth > endMonth);

        // Compare the date to the starting and ending rules.  For the ending
        // rule comparison, we add an hour to the millis passed in to convert
        // them from standard to daylight.  +1 = date>rule, -1 = date<rule, 0 =
        // date==rule.
        int startCompare = compareToRule(month, day, dayOfWeek, millis,
                         startMonth, startDayOfWeek, startDay, startTime);
        int endCompare = compareToRule(month, day, dayOfWeek, millis + millisPerHour,
                           endMonth, endDayOfWeek, endDay, endTime);

        // Check for both the northern and southern hemisphere cases.  We
        // assume that in the northern hemisphere, the start rule is before the
        // end rule within the calendar year, and vice versa for the southern
        // hemisphere.
        if ((!southern && (startCompare >= 0 && endCompare < 0)) ||
                (southern && (startCompare >= 0 || endCompare < 0)))
            result += millisPerHour;

        return result;
    }

    /**
     * Compare a given date in the year to a rule.  Return 1, 0, or -1, depending
     * on whether the date is after, equal to, or before the rule date.  The millis
     * are compared directly against the ruleMillis, so any standard-daylight adjustments
     * must be handled by the caller.  Assume that no rule references the end of
     * February (e.g., last Sunday in February).
     * @return 1 if the date is after the rule date, -1 if the date is before the
     * rule date, or 0 if the date is equal to the rule date.
     */
    private static int compareToRule(int month, int dayOfMonth, int dayOfWeek, int millis,
                     int ruleMonth, int ruleDayOfWeek, int ruleDOWinMonth, int ruleMillis)
    {
        if (month < ruleMonth)
            return -1;
        else if (month > ruleMonth)
            return 1;

        int monthLen = staticMonthLength[month];

        int ruleDayOfMonth;
        if (ruleDOWinMonth > 0)
            ruleDayOfMonth = 1 + (ruleDOWinMonth - 1) * 7 +
                    (7 + ruleDayOfWeek - (dayOfWeek - dayOfMonth + 1)) % 7;
        else // Assume ruleDOWinMonth < 0 here
            ruleDayOfMonth = monthLen + (ruleDOWinMonth + 1) * 7 -
                    (7 + (dayOfWeek + monthLen - dayOfMonth) - ruleDayOfWeek) % 7;

        if (dayOfMonth < ruleDayOfMonth)
            return -1;
        else if (dayOfMonth > ruleDayOfMonth)
            return 1;

        if (millis < ruleMillis)
            return -1;
        else if (millis > ruleMillis)
            return 1;
        else
            return 0;
    }

    /**
     * Overrides TimeZone
     * Gets the GMT offset for this time zone.
     */
    public int getRawOffset()
    {
        // The given date will be taken into account while
        // we have the historical time zone data in place.
        return rawOffset;
    }

    /**
     * Overrides TimeZone
     * Sets the base time zone offset to GMT.
     * This is the offset to add *to* UTC to get local time.
     * Please see TimeZone.setRawOffset for descriptions on the parameter.
     */
    public void setRawOffset(int offsetMillis)
    {
        this.rawOffset = offsetMillis;
    }

    /**
     * Overrides TimeZone
     * Queries if this time zone uses Daylight Savings Time.
     */
    public boolean useDaylightTime()
    {
        return useDaylight;
    }

    /**
     * Overrides TimeZone
     * Queries if the given date is in Daylight Savings Time.
     */
    public boolean inDaylightTime(Date date)
    {
        GregorianCalendar gc = new GregorianCalendar(this);
        gc.setTime(date);
        return gc.inDaylightTime();
   }

    /**
     * Overrides Cloneable
     */
    public Object clone()
    {
        return super.clone();
        // other fields are bit-copied
    }

    /**
     * Override hashCode.
     * Generates the hash code for the SimpleDateFormat object
     */
    public synchronized int hashCode()
    {
        return startMonth ^ startDay ^ startDayOfWeek ^ startTime ^
               endMonth ^ endDay ^ endDayOfWeek ^ endTime ^ rawOffset;
    }

    /**
     * Compares the equality of two SimpleTimeZone objects.
     * @param obj the SimpleTimeZone object to be compared with.
     * @return true if the given obj is the same as this SimpleTimeZone
     * object; false otherwise.
     */
    public boolean equals(Object obj)
    {
        if (this == obj)
            return true;
        if (!(obj instanceof SimpleTimeZone))
            return false;

        SimpleTimeZone that = (SimpleTimeZone) obj;

        if (!this.getID().equals(that.getID()))
            return false;
        if (this.startMonth != that.startMonth)
            return false;
        if (this.startDay != that.startDay)
            return false;
        if (this.startDayOfWeek != that.startDayOfWeek)
            return false;
        if (this.startTime != that.startTime)
            return false;
        if (this.endMonth != that.endMonth)
            return false;
        if (this.endDay != that.endDay)
            return false;
        if (this.endDayOfWeek != that.endDayOfWeek)
            return false;
        if (this.endTime != that.endTime)
            return false;
        if (this.startYear != that.startYear)
            return false;
        if (this.rawOffset != that.rawOffset)
            return false;
        if (this.useDaylight != that.useDaylight)
            return false;
        return true;
    }

    // =======================privates===============================

    private int startMonth, startDay, startDayOfWeek, startTime;
    private int endMonth, endDay, endDayOfWeek, endTime;
    private int startYear;
    private int rawOffset;
    private boolean useDaylight=false; // indicate if this time zone uses DST
    private static final int millisPerHour = 60*60*1000;
    // WARNING: assumes that no rule is measured from the end of February,
    // since we don't handle leap years. Could handle assuming always
    // Gregorian, since we know they didn't have daylight time when
    // Gregorian calendar started.
    // monthLength was non-static in JDK 1.1, so we have to keep it that way
    // to maintain serialization compatibility. However, there's no need to
    // recreate the array each time we create a new time zone.
    private final byte monthLength[] = staticMonthLength;
    private final static byte staticMonthLength[] = {31,28,31,30,31,30,31,31,30,31,30,31};

    /**
     * Read this object back from a serialization stream.
     */
    private void readObject(ObjectInputStream stream)
        throws IOException, ClassNotFoundException
    {
        stream.defaultReadObject();

        // Now we fix up a bug in the 1.1 SimpleTimeZone code -- namely,
        // startDayOfWeek and endDayOfWeek were usually uninitialized.  We can't do
        // too much, so we assume SUNDAY, which works most of the time.
        if (startDayOfWeek == 0)
            startDayOfWeek = Calendar.SUNDAY;
        if (endDayOfWeek == 0)
            endDayOfWeek = Calendar.SUNDAY;
    }

    // Proclaim compatibility with 1.1
    static final long serialVersionUID = -403250971215465050L;
}
