/* ============================================================================
 *
 * File:	misc.c
 * Project:	TkDesk
 * Started:	05.01.94
 * Changed:	31.03.96
 *
 * Description:	Implements several utility Tcl commands for TkDesk.
 *
 * Copyright (C) 1996  Christian Bolik
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * See the file "COPYING" in the base directory of this distribution
 * for more.
 *
 * ----------------------------------------------------------------------------
 *
 * Functions:
 *
 *
 * ========================================================================= */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#if defined(HAVE_STATFS) || defined(HAVE_STATVFS)
#include <sys/types.h> /* comment this out if it causes problems */
#ifndef NETBSD
#include <sys/vfs.h>  /* for statfs() */
#else
#include <sys/param.h>
#include <sys/mount.h>
#endif /* NETBSD */
#endif /* HAVE_STATFS */
#include "libdesk.h"


/* ============================================================================
 * Name   : dsk_stripc_Cmd
 * In     : ...
 * Out    : ...
 * Desc   : Syntax: dsk_stripc ?-keep? string
 *          This strips the file type character (one of "/@*+-_") from the
 *          string string. If the -keep option is given, only one of "+-_"
 *          will be removed.
 * Side-FX: none
 * ------------------------------------------------------------------------- */

int dsk_striptc_Cmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char *argv[];
{
    int keep = 0, i;
    char tc;
    char instr[TCL_RESULT_SIZE], outstr[TCL_RESULT_SIZE];
    
    if (argc < 2 || argc > 3) {
	sprintf (interp->result, "usage: dsk_striptc ?-keep? string");
	return (TCL_ERROR);
    }
    if (argc == 3) {
	if (strcmp ("-keep", argv[1]) == 0) {
	    keep = 1;
	    strcpy(instr, argv[2]);
	} else {
	    sprintf (interp->result, "usage: dsk_striptc ?-keep? string");
	    return (TCL_ERROR);
	}
    } else {
	strcpy(instr, argv[1]);
    }

    if (instr[0] == 0) {
	*interp->result = 0;
	return (TCL_OK);
    }

    strcpy (outstr, instr);
    i = 0;
    while (outstr[i+1] != '\\' && outstr[i+1]) {
	i++;
    }
    /* outstr[i] now contains the type character */
    tc = outstr[i];
    if (!keep) {
	if (tc == '@' || tc == '*' || tc == '/' || tc == '+' \
	   || tc == '-' || tc == '_') {
	    if (outstr[i+1])
		outstr[i] = ' ';
	    else 
		outstr[i] = '\0';
	}
    } else {
	if (tc == '_') {
	    if (outstr[i+1])
		outstr[i] = ' ';
	    else 
		outstr[i] = '\0';
	}
    }

    sprintf (interp->result, outstr);
    return (TCL_OK);
} /* dsk_striptc_Cmd */

/* ============================================================================
 * Name   : dsk_statfs_Cmd
 * In     : ...
 * Out    : ...
 * Desc   : Syntax: dsk_statfs file
 *          Returns a tcl list that consists of three elements, namely
 *          capacity, usage and available. All number are in KB and are
 *          given for the file-system that $file resides on.
 * Side-FX: none
 * ------------------------------------------------------------------------- */

int dsk_statfs_Cmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char *argv[];
{
#if defined(HAVE_STATFS) || defined(HAVE_STATVFS)
#ifdef HAVE_STATFS
    struct statfs fs;
#endif
#ifdef HAVE_STATVFS
    struct statvfs fs;
#endif
    long cap, used, avail;
    float factor;
    
    if (argc != 2) {
	sprintf(interp->result, "usage: dsk_statfs file");
	return(TCL_ERROR);
    }
    
#ifdef HAVE_STATFS
    if (statfs(argv[1], &fs) != 0) {
#endif
#ifdef HAVE_STATVFS
    if (statvfs(argv[1], &fs) != 0) {
#endif
	sprintf(interp->result, "%s: %s", argv[1], strerror(errno));
	return(TCL_ERROR);
    }

    factor = (float) fs.f_bsize / 1024.;
    cap = (long) ((float) fs.f_blocks * factor);
    used = cap - (long) ((float) fs.f_bfree * factor);
    avail = (long) ((float) fs.f_bavail * factor);

    sprintf (interp->result, "%ld %ld %ld", cap, used, avail);
    return (TCL_OK);
#else
    sprintf (interp->result, "0 0 0");
    return (TCL_OK);
#endif    
}
