/*  LAST EDIT: Mon Oct  2 20:59:38 1995 by Thorsten Kukuk (kukuk)  */
/*
    YPS-0.2, NIS-Server for Linux
    Copyright (C) 1994  Tobias Reber

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Modified for use with ypserv by Thorsten Kukuk <kukuk@uni-paderborn.de>
*/

#include "system.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <rpc/rpc.h>
#include "yp.h"
#include <rpcsvc/ypclnt.h>
#include <sys/param.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <ctype.h>
#include <netdb.h>
#include <errno.h>
#include <stdarg.h>
#include <memory.h>
#include <gdbm.h>
#include <string.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#elif !defined(HAVE_GETOPT) || defined(sun)
#include <compat/getopt.h>
#endif

#ifndef HAVE_GETOPT
#include <compat/getopt.c>
#endif

#ifndef HAVE_STRERROR
#include <compat/strerror.c>
#endif

static void Perror(char *fmt, ...)
{
   va_list ap;
   va_start(ap, fmt);
   vfprintf(stderr, fmt, ap);
   va_end(ap);
}

#ifndef YPMAPDIR
#define YPMAPDIR "/var/yp"
#endif

#undef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))

static char *DomainName=NULL;
static bool_t Verbose=FALSE;

static char *ThisHost=NULL;

static char *MapName;
static u_int MapOrderNum;

static CLIENT *PushClient=NULL;

static u_int CallbackTransid;
static u_int CallbackProg=0;
static SVCXPRT *CallbackXprt;

#if HAVE__RPC_DTABLESIZE
extern int _rpc_dtablesize(void);
#elif HAVE_GETDTABLESIZE
int _rpc_dtablesize()
{
        static int size;
        
        if (size == 0) {
                size = getdtablesize();
        }
        return (size);
}
#else

#include <sys/resource.h>

int _rpc_dtablesize()
{
    static int size = 0;
    struct rlimit rlb;


    if (size == 0)
    {
	if (getrlimit(RLIMIT_NOFILE, &rlb) >= 0)
	    size = rlb.rlim_cur;
    }

    return size;
}
#endif

/*
 * This is the rpc server side idle loop
 * Wait for input, call server program.
 */

static inline void
my_svc_run( void)
{
#ifdef FD_SETSIZE
  fd_set readfds;
#else
  int readfds;
#endif /* def FD_SETSIZE */
  struct timeval t;
  
  t.tv_sec=60; t.tv_usec=0;

  for (;;) 
    {
#ifdef FD_SETSIZE
      readfds = svc_fdset;
#else
      readfds = svc_fds;
#endif /* def FD_SETSIZE */
      switch (select(_rpc_dtablesize(), &readfds, (void *)0, (void *)0, &t)) 
	{
	case -1:
	  if (errno == EINTR) 
	    {
	      continue;
	    }
	  Perror("svc_run: - select failed (%s)",strerror(errno));
	  return;
	case 0:
	  Perror( "YPPUSH: Callback timed out\n");
	  exit(0);
	default:
	  svc_getreqset(&readfds);
	}
    }
}


static inline void Usage(void)
{
  Perror("Usage: yppush [ -d domain ] [ -v ] mapname ...\n");
  exit(1);
}

char *yppush_err_string(enum yppush_status y) 
{
  switch(y) 
    {
    case YPPUSH_SUCC:      return "Success";
    case YPPUSH_AGE:       return "Master's version not newer";
    case YPPUSH_NOMAP:	   return "Can't find server for map";
    case YPPUSH_NODOM:	   return "Domain not supported";
    case YPPUSH_RSRC:	   return "Local resource alloc failure";
    case YPPUSH_RPC:	   return "RPC failure talking to server";
    case YPPUSH_MADDR:	   return "Can't get master address";
    case YPPUSH_YPERR:	   return "YP server/map db error";
    case YPPUSH_BADARGS:   return "Request arguments bad";
    case YPPUSH_DBM:	   return "Local dbm operation failed";
    case YPPUSH_FILE:	   return "Local file I/O operation failed";
    case YPPUSH_SKEW:	   return "Map version skew during transfer";
    case YPPUSH_CLEAR:	   return "Can't send \"Clear\" req to local ypserv";
    case YPPUSH_FORCE:	   return "No local order number in map  use -f flag.";
    case YPPUSH_XFRERR:	   return "ypxfr error";
    case YPPUSH_REFUSED:   return "Transfer request refused by ypserv";
    }
  return "YPPUSH: Unknown Error, this should not happen!";
}

void *yppushproc_null_1(void * req, CLIENT * rqstp) 
{
  static void *resp;
  return &resp;
}

yppushresp_xfr *yppushproc_xfrresp_1_svc(void *req, struct svc_req * rqstp)
{
  static yppushresp_xfr resp;
  
  memcpy(&resp, (yppushresp_xfr*)req, sizeof(yppushresp_xfr));
  if (resp.status!=YPPUSH_SUCC)
    Perror("YPPUSH: %s", yppush_err_string(resp.status));
  return &resp;
}

void yppush_xfrrespprog_1( struct svc_req *rqstp, SVCXPRT *transp)
{
  union {
    int fill;
  } argument;
  static char *result;
  static bool_t (*xdr_argument)(XDR *, void *), (*xdr_result)(XDR *, void *);
  static char *(*local)( void *, struct svc_req *);
  
  switch (rqstp->rq_proc) 
    {
    case YPPUSHPROC_NULL:
      xdr_argument = (bool_t (*)(XDR *, void *))xdr_void;
      xdr_result = (bool_t (*)(XDR *, void *))xdr_void;
      local = (char *(*)( void *, struct svc_req *)) yppushproc_null_1;
      break;
      
    case YPPUSHPROC_XFRRESP:
      xdr_argument = (bool_t (*) (XDR *, void *))xdr_yppushresp_xfr;
      xdr_result = (bool_t (*)(XDR *, void *))xdr_void;
      local = (char *(*)( void *, struct svc_req *)) yppushproc_xfrresp_1_svc;
      break;
      
    default:
      svcerr_noproc(transp);
      exit(1);
    }
  memset(&argument, '\0',  sizeof(argument));
  if (!svc_getargs(transp, xdr_argument, (void *) &argument)) 
    {
      svcerr_decode(transp);
      exit(1);
    }
  result = (*local)(&argument, rqstp);
  if (result != NULL && !svc_sendreply(transp, xdr_result, result)) 
    {
      svcerr_systemerr(transp);
    }
  if (!svc_freeargs(transp, xdr_argument, (void *) &argument)) 
    {
      Perror("unable to free arguments\n");
      exit(1);
      if (rqstp->rq_proc!=YPPUSHPROC_NULL)
	exit(0);
    }
  exit(0);
}

static  char *getHostName( void)
{
  static char hostname[MAXHOSTNAMELEN+1];
  struct hostent *h;
  if (gethostname(hostname, sizeof hostname)!=0) 
    {
      perror("YPPUSH: gethostname");
      return NULL;
    }
  h=gethostbyname(hostname);
  if (h!=NULL) 
    {
      strncpy(hostname, h->h_name, sizeof (hostname)-1);
      hostname[sizeof (hostname)-1]='\0';
    }
  return hostname;
}

static  u_int getOrderNum( void)
{
  static char mapPath[MAXPATHLEN];
  datum o;
  int i;
  GDBM_FILE dbm;

  sprintf(mapPath, "%s/%s/%s", YPMAPDIR, DomainName, MapName);
  if ((dbm=gdbm_open(mapPath, 0, GDBM_READER, 0600, NULL))==NULL) 
    {
      Perror("YPPUSH: Cannot open %s\n", mapPath);
      return -1;
    }
	
  o.dptr="YP_LAST_MODIFIED"; o.dsize=strlen(o.dptr);
  o=gdbm_fetch(dbm,o);
  gdbm_close(dbm);
  if (o.dptr==NULL) 
    {
      Perror("YPPUSH: %s: Cannot determine order number\n", MapName);
      return -1;
    }
  
  for (i=0; i<o.dsize; i++) 
    {
      if (!isdigit(o.dptr[i])) 
	{
	  Perror("YPPUSH: %s: Invalid order number '%s'\n",
		 MapName, o.dptr);
	  return -1;
	}
    }
  
  return atoi(o.dptr);
}

static void doPushClient( const char *targetHost)
{
  static struct ypreq_xfr req;
  static struct timeval tv={0,0};
  
  req.map_parms.domain=DomainName;
  req.map_parms.map=(char *)MapName;
  req.map_parms.peer=ThisHost;
  req.map_parms.ordernum=MapOrderNum;
  req.transid=CallbackTransid;
  req.prog=CallbackProg;
  req.port=CallbackXprt->xp_port;
  
  if (Verbose)
    printf("%d: %s(%d@%s) -> %s@%s\n", req.transid,
	   req.map_parms.map, req.map_parms.ordernum,
	   req.map_parms.peer, targetHost,
	   req.map_parms.domain);
  switch (clnt_call(PushClient, YPPROC_XFR, xdr_ypreq_xfr, (void *) &req,
		    xdr_void, NULL, tv)) 
    {
    case RPC_SUCCESS:
    case RPC_TIMEDOUT:
      break;
    default:
      clnt_perror(PushClient, "YPPUSH: Cannot call YPPROC_XFR");
      kill(CallbackTransid, SIGTERM);
    }
  return;
}

static  bool_t registerServer(void)
{
  int s;
  s=RPC_ANYSOCK;
  CallbackXprt=svcudp_create(s);
  if (CallbackXprt==NULL) 
    {
      Perror("YPPUSH: Cannot create callback transport.\n");
      return FALSE;
    }
  for (CallbackProg=0x40000000; CallbackProg<0x5fffffff; CallbackProg++) 
    {
      if (svc_register(CallbackXprt, CallbackProg, 1,
		       yppush_xfrrespprog_1, IPPROTO_UDP))
	return TRUE;
    }
  return FALSE;
}

static bool_t createClient(const char *targetHost)
{
  PushClient=clnt_create((char *)targetHost, YPPROG, YPVERS, "tcp");
  if (PushClient==NULL) 
    {
      clnt_pcreateerror("YPPUSH: Cannot create client");
      return FALSE;
    }
  return TRUE;
}

static void doPush( const char *targetHost)
{
  int s;
#if HAVE_WAIT3
  struct rusage r;
#endif

  if (!createClient(targetHost))
    return;
  if (!registerServer())
    return;
  
  switch (CallbackTransid=fork()) 
    {
    case -1:
      perror("YPPUSH: Cannot fork");
      exit(1);
    case 0:
      my_svc_run();
      exit(0);
    default:
      close(CallbackXprt->xp_sock);
      doPushClient(targetHost);
#if HAVE_WAIT3
      wait4(CallbackTransid, &s, 0, &r);
#else
      waitpid(CallbackTransid,&s, 0);
#endif
      svc_unregister(CallbackProg, 1);
      CallbackProg=0;
      if (PushClient!=NULL) 
	{
	  clnt_destroy(PushClient);
	  PushClient=NULL;
	}
    }
}

static int yppushForeach(int status, const char *inKey, int inKeylen,
			 const char *inVal, int inVallen, const char *data)
{
  char targetHost[YPMAXPEER+1];

  if (status!=YP_TRUE) return 0;
  memcpy(targetHost, inKey, MIN(sizeof (targetHost)-1, inKeylen));
  targetHost[MIN(sizeof (targetHost)-1, inKeylen)]='\0';
  doPush(targetHost);
  return 0;
}
	
static void intrHandler(int sig)
{
  if (CallbackProg!=0)
    svc_unregister(CallbackProg, 1);
  exit(1);
}

int main(int argc, char **argv)
{
  char c;
  struct ypall_callback f;
  enum ypstat y;
  struct sigaction a;
  
  a.sa_handler=intrHandler;
  sigemptyset(&a.sa_mask);
#if defined(linux) || (defined(sun) && defined(__srv4__))
  a.sa_flags=SA_NOMASK; 
  /* Do  not  prevent  the  signal   from   being
     received from within its own signal handler. */
#endif
  sigaction(SIGINT, &a, NULL);
  
  while((c=getopt(argc, argv, "d:v"))!=EOF) 
    {
      switch(c) 
	{
	case 'd':
	  DomainName=optarg;
	  break;
	case 'v':
	  Verbose=TRUE;
	  break;
	default:
	  Usage();
	}
    }
  argc-=optind;
  argv+=optind;
  
  if (argc<1) Usage();
  
  if (DomainName==NULL) 
    {
      if (yp_get_default_domain(&DomainName)!=0) 
	{
	  Perror("YPPUSH: Cannot get default domain\n");
	  return 1;
	}
    }
  
  ThisHost=getHostName();
  if (ThisHost==NULL) 
    {
      Perror("YPPUSH: Cannot determine local hostname\n");
      return 1;
    }
  
  while (*argv) 
    {
      MapName=*argv++;
      MapOrderNum=getOrderNum();
      if (MapOrderNum==0xffffffff)
	continue;
      f.foreach=yppushForeach;
      y=yp_all(DomainName, "ypservers", &f);
      if (y && y!=YP_NOMORE) 
	{
	  Perror("YPPUSH: Could not read ypservers: %d %s\n",
		 y, yperr_string(y));
	  return 1;
	}
    }
  return 0;
}
