/*
 *      Copyright (C) 1993-1995 Bas Laarhoven.
 *
 * Considerable changes by Claus-Justus Heine, 1994, 1995
 * implemented compression and random block access
 *
 * Lot of changes to ftape_flush_buffers and
 * _ftape_write. New function ftape_simple_write()
 *
 *
 

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 /home/cvs/zftape/ftape-write.c,v
 claus
 *
 1.5
 1995/11/16 22:33:33
 Exp
 *
 *      This file contains the writing code
 *      for the QIC-117 floppy-tape driver for Linux.
 */

static char RCSid[] =
"ftape-write.c,v 1.5 1995/11/16 22:33:33 claus Exp";

  
#include "ftape.h"
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <asm/segment.h>

#include "ftape-write.h"
#include "ftape-read.h"
#include "qic117.h"
#include "ftape-io.h"
#include "ftape-ctl.h"
#include "ftape-rw.h"
#include "ftape-eof.h"
#include "ecc.h"
#include "ftape-bsm.h"
#include "qic80-compress.h"
#include "qic80-vtbl.h"
 


/*      Global vars.
 */

/*      Local vars.
 */
static int last_write_failed = 0;
static int buf_pos_start = 0;
static int need_flush = 0;

void
ftape_zap_write_buffers( void)
{
  int i;

  for (i = 0; i < ftape_num_buffers; ++i) {
    buffer[ i].status = done;
  }
  need_flush = 0;
}

static int
copy_and_gen_ecc( char* destination, byte* source,
                 unsigned int bad_sector_map)
{
  TRACE_FUN( 8, "copy_and_gen_ecc");
  int result;
  struct memory_segment mseg;
  int bads = count_ones( bad_sector_map);

  if (bads > 0) {
    TRACEi( 4, "bad sectors in map:", bads);
  }
  if (bads + 3 >= SECTORS_PER_SEGMENT) {
    TRACE( 4, "empty segment");
    mseg.blocks = 0;            /* skip entire segment */
    result = 0;                 /* nothing written */
  } else {
    mseg.blocks = SECTORS_PER_SEGMENT - bads;
    mseg.data = destination;
    memcpy( mseg.data, source, (mseg.blocks - 3) * SECTOR_SIZE);
    result = ecc_set_segment_parity( &mseg);
    if (result < 0) {
      TRACE( 1, "ecc_set_segment_parity failed");
    } else {
      result = (mseg.blocks - 3) * SECTOR_SIZE;
    }
  }
  TRACE_EXIT;
  return result;
}

void
prevent_flush( void)
{
  need_flush = 0;
  ftape_state = idle;
}

int
start_writing( int mode)
{
  TRACE_FUN( 5, "start_writing");
  int result = 0;
  buffer_struct* buff = &buffer[ head];
  int segment_id = buff->segment_id;

  if (ftape_state == writing && buff->status == waiting) {
    setup_new_segment( buff, segment_id, 1);
    if (mode == WRITE_SINGLE) {
      buffer[ head].next_segment = 0; /* stop tape instead of pause */
    }
    calc_next_cluster( buff);     /* prepare */
    buff->status = writing;
    if (runner_status == idle) {
      TRACEi( 5, "starting runner for segment", segment_id);
      result = ftape_start_tape( segment_id, buff->sector_offset);
      if (result >= 0) {
        runner_status = running;
      }
    }
    if (result >= 0) {
      result = setup_fdc_and_dma( buff, FDC_WRITE); /* go */
    }
    ftape_state = writing;
  }
  TRACE_EXIT;
  return result;
}

int
loop_until_writes_done( void)
{
  TRACE_FUN( 5, "loop_until_writes_done");
  int i;
  int result = 0;

  /*
   *  Wait until all data is actually written to tape.
   */
  while (ftape_state == writing && buffer[ head].status != done) {
    TRACEx2( 7, "tail: %d, head: %d", tail, head);
    for (i = 0; i < ftape_num_buffers; ++i) {
      TRACEx3( 8, "buffer[ %d] segment_id: %d, status: %d",
              i, buffer[ i].segment_id, buffer[ i].status);
    }
    result = fdc_interrupt_wait( 5 * SECOND);
    if (result < 0) {
      TRACE( 1, "fdc_interrupt_wait failed");
      last_write_failed = 1;
      break;
    }
    if (buffer[ head].status == error) {
      /* Allow escape from loop when signaled !
       */
      if (current->signal & _DONT_BLOCK) {
        TRACE( 2, "interrupted by signal");
        TRACE_EXIT;
        result = -EINTR;          /* is this the right return value ? */
        break;
      }
      if (buffer[ head].hard_error_map != 0) {
        /*  Implement hard write error recovery here
         */
      }
      buffer[ head].status = waiting; /* retry this one */
      if (runner_status == aborting) {
        ftape_dumb_stop();
        runner_status = idle;
      }
      if (runner_status != idle) {
        TRACE( 1, "unexpected state: runner_status != idle");
        result = -EIO;
        break;
      }
      start_writing( WRITE_MULTI);
    }
    TRACE( 5, "looping until writes done");
    result = 0;                 /* normal exit status */
  }
  TRACE_EXIT;
  return result;
}

/*      Write given segment from buffer at address onto tape.
 */
int
write_segment( unsigned segment_id, byte* address, int flushing)
{
  TRACE_FUN( 5, "write_segment");
  int result = 0;
  int bytes_written = 0;
  
  TRACEi( 5, "segment_id =", segment_id);
  if (ftape_state != writing) {
    if (ftape_state == reading) {
      TRACE( 5, "calling ftape_abort_operation");
      result = ftape_abort_operation();
      if (result < 0) {
        TRACE( 1, "ftape_abort_operation failed");
      }
    }
    ftape_zap_read_buffers();
    ftape_zap_write_buffers();
    ftape_state = writing;
  }
  /*    if all buffers full we'll have to wait...
   */
  wait_segment( writing);
  if (buffer[ tail].status == error) {
    /*  setup for a retry
     */
    buffer[ tail].status = waiting;
    bytes_written = -EAGAIN;    /* force retry */
    if (buffer[ tail].hard_error_map != 0) {
      TRACEx1( 1, "warning: %d hard error(s) in written segment",
              count_ones( buffer[ tail].hard_error_map));
      TRACEx1( 4, "hard_error_map = 0x%08lx", buffer[ tail].hard_error_map);
      /*  Implement hard write error recovery here
       */
    }
  } else if (buffer[ tail].status == done) {
    history.defects += count_ones( buffer[ tail].hard_error_map);
  } else {
    TRACE( 1, "wait for empty segment failed");
    result = -EIO;
  }
  /*    If just passed last segment on tape: wait for BOT or EOT mark.
   */
  if (result >= 0 && runner_status == logical_eot) {
    int status;

    result = ftape_ready_wait( timeout.seek, &status);
    if (result < 0 || status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT) == 0) {
      TRACE( 1, "eot/bot not reached");
    } else {
      runner_status = end_of_tape;
    }
  }
  /*    should runner stop ?
   */
  if (result >= 0 &&
      (runner_status == aborting || runner_status == buffer_underrun ||
       runner_status == end_of_tape)) {
    if (runner_status != end_of_tape) {
      result = ftape_dumb_stop();
    }
    if (result >= 0) {
      if (runner_status == aborting) {
        if (buffer[ head].status == writing) {
          buffer[ head].status = done; /* ????? */
        }
      }
      runner_status = idle;       /* aborted ? */
    }
  }
  /*  Don't start tape if runner idle and segment empty.
   */
  if (result >= 0 && !(runner_status == idle &&
                       get_bad_sector_entry( segment_id) == EMPTY_SEGMENT)) {
    if (buffer[ tail].status == done) {
      /*    now at least one buffer is empty, fill it with our data.
       *    skip bad sectors and generate ecc.
       *    copy_and_gen_ecc return nr of bytes written,
       *    range 0..29 Kb inclusive !
       */
      result = copy_and_gen_ecc( buffer[ tail].address, address,
                                get_bad_sector_entry( segment_id));
      if (result >= 0) {
        bytes_written = result;
        buffer[ tail].segment_id = segment_id;
        buffer[ tail].status = waiting;
        next_buffer( &tail);
      }
    }
    /*    Start tape only if all buffers full or flush mode.
     *    This will give higher probability of streaming.
     */
    if (result >= 0 && runner_status != running &&
        ((head == tail && buffer[ tail].status == waiting) || flushing)) {
      result = start_writing( WRITE_MULTI);
    }
  }
  TRACE_EXIT;
  return (result < 0) ? result : bytes_written;
}

/*  Write as much as fits from buffer to the given segment on tape
 *  and handle retries.
 *  Return the number of bytes written (>= 0), or:
 *      -EIO          write failed
 *      -EINTR        interrupted by signal
 *      -ENOSPC       device full
 */
int
_write_segment( unsigned int segment_id, byte* buffer, int flush)
{
  TRACE_FUN( 5, "_write_segment");
  int retry = 0;
  int result;

  history.used |= 2;
  for (;;) {
    if (segment_id > ftape_last_data_segment && !flush) {
      result = -ENOSPC;         /* tape full */
      break;
    }
    result = write_segment( segment_id, buffer, flush);
    if (result < 0) {
      if (result == -EAGAIN) {
        if (++retry > 100) {
          TRACE( 1, "write failed, >100 retries in segment");
          result = -EIO;        /* give up */
          break;
        } else {
          TRACEx1( 2, "write error, retry %d", retry);
        }
      } else {
        TRACEi( 1, "write_segment failed, error:", -result);
        break;
      }
    } else {                    /* success */
      if (result == 0) {        /* empty segment */
        TRACE( 4, "empty segment, nothing written");
      }
      break;
    }
    /* Allow escape from loop when signaled !
     */
    if (current->signal & _DONT_BLOCK) {
      TRACE( 2, "interrupted by signal");
      TRACE_EXIT;
      result = -EINTR;          /* is this the right return value ? */
      break;
    }
  }
  TRACE_EXIT;
  return result;
}


static int
update_header_segment( unsigned segment, byte* buffer)
{
  TRACE_FUN( 5, "update_header_segment");
  int result = 0;

  if (buffer == NULL) {
    TRACE( 5, "no input buffer specified");
    buffer = deblock_buffer;
    result = read_segment( used_header_segment, buffer, 0);
    if (bad_sector_map_changed) {
      store_bad_sector_map( buffer);
    }
    if (failed_sector_log_changed) {
      update_failed_sector_log( buffer);
    }
  }
  if (result >= 0 && GET4( buffer, 0) != 0xaa55aa55) {
    TRACE( 1, "wrong header signature found, aborting");
    result = -EIO;
  }
  if (result >= 0) {
    result = ftape_verify_write_segment( segment, buffer );
  }
  TRACE_EXIT;
  return result;
}

int
ftape_write_header_segments( byte* buffer)
{
  TRACE_FUN( 5, "ftape_write_header_segments");
  int result = 0;
  int retry = 0;
  int header_1_ok = 0;
  int header_2_ok = 0;

  do {
    if (!header_1_ok) {
      result = update_header_segment( header_segment_1, buffer);
      if (result < 0) {
        continue;
      }
      header_1_ok = 1;
    }
    if (!header_2_ok) {
      result = update_header_segment( header_segment_2, buffer);
      if (result < 0) {
        continue;
      }
      header_2_ok = 1;
    }
  } while (result < 0 && retry++ < 3);
  if (result < 0) {
    if (!header_1_ok) {
      TRACE( 1, "update of first header segment failed");
    }
    if (!header_2_ok) {
      TRACE( 1, "update of second header segment failed");
    }
    result = -EIO;
  }
  TRACE_EXIT;
  return result;
}

int
ftape_update_header_segments( byte* buffer, int update)
{
  TRACE_FUN( 5, "ftape_update_header_segments");
  int result = 0;
  int header_changed = 1;


  if (ftape_state == writing) {
    result = loop_until_writes_done();
  }
  if (read_only) {
    result = 0;                 /* exit and fake success */
    TRACE( 4, "Tape set read-only: no update");
  } else if (result >= 0) {
    result = ftape_abort_operation();
    if (result >= 0) {
      if (buffer == NULL) {
        if ( bad_sector_map_changed    ||
             failed_sector_log_changed ||
             ftape_compression_map_changed   ||
             ftape_volume_table_changed      ||
             ftape_label_changed
          )
        {
          TRACE(5, "Reading header segments ... ");
          ftape_seek_to_bot();  /* prevents extra rewind */
          buffer = deblock_buffer;
          ftape_deblock_segment = -1;
          result = read_segment( used_header_segment, buffer, 0);
          if (result < 0) {
            TRACE_EXIT;
            return result;
          }
        }
        header_changed = 0;
      }
      if (update) {
        if (bad_sector_map_changed) {
          store_bad_sector_map( buffer);
          header_changed = 1;
        }
        if (failed_sector_log_changed) {
          update_failed_sector_log( buffer);
          header_changed = 1;
        }
        if ( ftape_label_changed ) {
          memcpy( buffer + 30, ftape_label, 44 );
          PUT4( buffer, 74, 0);
          TRACEx1( 5, "new label: `%s'", (char*)(buffer + 30));
          header_changed= 1;
        }
        if ( ftape_compression_map_changed ) {
          PUT2( buffer, 78, ftape_compression_map_location );
          PUT2( buffer, 10, first_data_segment );
          header_changed= 1;
        }
        if ( ftape_volume_table_changed ) {
           header_changed = 1;
         }
      }
      if (header_changed) {
        ftape_seek_to_bot();    /* prevents extra rewind */
        result = ftape_write_header_segments( buffer);
        if (result >= 0) {
          if ( ftape_volume_table_changed && ftape_qic80_mode != 0 ) {
            result = qic80_update_volume_table( first_data_segment, buffer );
          }
          if ( result >= 0                         &&
               ftape_compression_map_location != 0 &&
               ftape_compression_map_changed       &&
               ftape_qic80_mode != 0                   )
          {
            result = ftape_update_compression_segments( ftape_compression_map_location, buffer );
          }
          if (result >= 0) {
            result = loop_until_writes_done();
            if ( update ) {
              bad_sector_map_changed    =
              failed_sector_log_changed =
              ftape_compression_map_changed   =
              ftape_volume_table_changed      =
              ftape_label_changed             = 0;
            }
          }
        }
      }
    }
  }
  TRACE_EXIT;
  return result;
}

int
ftape_flush_buffers( void)
{
  TRACE_FUN( 5, "ftape_flush_buffers");
  int result;
  int data_remaining;
  int total_remaining;
  int this_segs_size;
  static int active = 0;
#ifndef NO_TRACE_AT_ALL
  int old_tracing;     
#endif

  if (active) {
    TRACE( 5, "nested call, abort");
    TRACE_EXIT;
    return 0;
  }
  active = 1;
  TRACEi( 5, "entered, ftape_state =", ftape_state);
  if (ftape_state != writing && !need_flush) {
    active = 0;       
    TRACE( 5, "no need for flush");
    TRACE_EXIT;
    return 0;
  }
  if (last_write_failed) {
    ftape_zap_write_buffers();
    active = 0;
    TRACE_EXIT;
    return write_protected ? -EROFS : -EIO;
  }
  TRACE( 5, "flushing write buffers");
  total_remaining =
  data_remaining  = ftape_seg_data_pos;
  /*
   *    If there is any data not written to tape yet, append zero's
   *    up to the end of the sector (if using compression) or merge
   *    it with the data existing on the tape
   *    Then write the segment(s) to tape.
   *
   */
  TRACEi( 5, "flush, remainder in buffer:", total_remaining);
  /* 
   *  now we have the dirty task to merge the new data with the old data
   */
  if ( data_remaining > 0 ) 
  {
    do 
    {
      this_segs_size = ftape_get_seg_size( ftape_seg_pos );
      if (    (this_segs_size > data_remaining)
           && (ftape_deblock_segment != ftape_seg_pos) )
      {
        if ( !ftape_qic80_mode )
        {
          TRACE( 5, "waiting");
          start_writing( WRITE_MULTI);
          result = loop_until_writes_done();
          if (result < 0) {
            TRACE( 1, "flush buffers failed");
            need_flush = 0;
            active = 0;
            TRACE_EXIT;
            return result;
          }      
#ifndef NO_TRACE_AT_ALL
          old_tracing= tracing;
          tracing= -2;   
#endif
          result= read_segment_fraction( ftape_seg_pos,
                                               deblock_buffer,
                                               0, data_remaining,
                                               this_segs_size-data_remaining);
#ifndef NO_TRACE_AT_ALL
          tracing= old_tracing;
#endif
          if ( result != (this_segs_size - data_remaining ) ) {
            TRACEx1( 7, "read_segment result: %d", result);
            TRACE(4,"Impossible to read old data, cannot merge it with new data");
            TRACE(4,"This is normal when using the enhanced driver with file-mark support");
            TRACEx2(4,"Overwriting this segment (%d) with new data,"
                     "padded with %d zero bytes",
                     ftape_seg_pos, 
                     this_segs_size - data_remaining );
            if ( result < 0 )
              memset( deblock_buffer+data_remaining, 0,
                              this_segs_size - data_remaining );
            else
              memset( deblock_buffer+data_remaining, 0,
                     this_segs_size - data_remaining - result );
          }
        } else {
          /*
           *   it's impossible to insert a compressed block in the middle of
           *   the tape. We can't do anything but append blocks at the end
           *   of the already compressed area.
           */
          memset( deblock_buffer+data_remaining, 0,
                  this_segs_size - data_remaining );
        }
        ftape_deblock_segment= ftape_seg_pos;
      }
      result = _write_segment( ftape_seg_pos, deblock_buffer, 1);
      if (result != this_segs_size ) {
        if (result != -ENOSPC) {
          last_write_failed = 1;
        }
        ftape_data_pos    -= ftape_seg_data_pos;
        ftape_seg_data_pos = 0;
        active = 0;
        TRACE_EXIT;
        return result;
      }
      TRACEi( 5, "flush, moved out buffer:", result);
      /*
       *  the following shouldn't happen anymore as we never put more data
       *  in the deblock_buffer than fit in the actual segment
       *  but I'm not quite sure what happens with empty segments.
       *  The following shouldn't be neccessary anyway
       */
      if ( result < data_remaining ) { /* need next segment for more data */
        total_remaining = DEBLOCK_SIZE - result;
        if (result > 0) {       /* move remainder to buffer beginning */
          memmove( deblock_buffer, deblock_buffer + result,
                                                        total_remaining);
        }
      } else {
        total_remaining = 0;        /* no need to write more, reset buffer */
      }
      data_remaining -= result;
      ++ftape_seg_pos;
    } while (data_remaining > 0);
    TRACEi( 7, "result:", result);
    if (data_remaining == 0) {  /* first byte next segment */
      ftape_seg_data_pos = 0;
      qic80_maybe_write_eof( ftape_seg_pos - 1, ftape_uncmpr_pos );
    } else {                    /* after data previous segment */
      ftape_seg_pos -= 1;
      ftape_seg_data_pos = data_remaining + result;
      qic80_maybe_write_eof( ftape_seg_pos, ftape_uncmpr_pos );
    }
  } else {
    TRACE( 4, "deblock_buffer empty");
    qic80_maybe_write_eof( ftape_seg_pos - 1 , ftape_uncmpr_pos );
    start_writing( WRITE_MULTI);
  }
  /*
   *  why set just_before_eof ?
   *
   */
  just_before_eof = ftape_qic80_mode;
  TRACEx2(4,"ftape_seg_pos: %d, ftape_seg_data_pos: %d",
             ftape_seg_pos, ftape_seg_data_pos);
  TRACE( 7, "waiting");
  result = loop_until_writes_done();
  if (result < 0) {
    /*
     *  thast's really bad. We don't know what to to with ftape_data_pos
     */
    TRACE( 1, "flush buffers failed");
  }
  ftape_state= idle;
  last_write_failed = 0;
  need_flush = 0;
  active = 0;
  TRACE_EXIT;
  return result;
}

/*
 * return-value: the number of bytes removed from the user-buffer
 * out: 
 *      int *write_cnt: how much actually has been moved to the deblock_buffer
 *      int *req_len  : MUST NOT BE CHANGED, except at EOT, in which case it
 *                      may be adjusted
 * in : 
 *      char *buff          : the user buffer
 *      int buf_pos_write   : copy of  buf_len_wr
 *      int this_segs_size  : the size in bytes of the actual segment
 *      char *deblock_buffer: deblock_buffer
 *
 */
static int
ftape_simple_write ( int *cnt, int *req_len, const char *buff, 
                     int buf_pos_write, int this_seg_size, 
		     char *deblock_buffer )
{
  TRACE_FUN( 5, "ftape_simple_write");
  int remaining;
  int space_left;

  /*
   *  remaining space in this segment, NOT deblock_buffer
   */
  space_left = this_seg_size - buf_pos_write;
  /*
   *  what reamains on this tape
   */
  remaining = ftape_total_data_amount - (ftape_data_pos + *req_len);
  if ( remaining < 0 ) {
    TRACEi( 4, "clipping req_len from:", *req_len );
    *req_len += remaining;
    TRACEi( 4, "to:", *req_len );
  }                           
  *cnt = *req_len < space_left ? *req_len : space_left;
  /*
   *  already verified in ftape_write()
   */
  memcpy_fromfs( deblock_buffer + buf_pos_write, buff, *cnt);
  ftape_uncmpr_pos += *cnt;
  TRACE_EXIT;
  return *cnt;
}

/*
 *  called by the kernel-interface routine "ftape_write()"
 */

int
_ftape_write( const char* buff, int req_len)
{
  TRACE_FUN( 5, "_ftape_write");
  int result = 0;
  int written = 0;
  int cnt;
  int this_seg_size;

  ftape_resid = req_len;

  if ( ftape_offline || !formatted || no_tape) {
    TRACEx1( 1,"Error: tape is %s", ftape_offline ?
                              "offline" 
                            : (!formatted ?
                                "not formatted"
                              : "not present"   ));
    result = -EIO;
  } else if ( write_protected ) {
    TRACE( 1, "Error: cartridge write protected");
    last_write_failed = 1;
    result = -EROFS;
  } else if ( first_data_segment == -1) {
  /*
   *    If we haven't read the header segment yet, do it now.
   *    This will verify the configuration, get the eof markers
   *    and the bad sector table and the compression-map.
   *    We'll use the deblock buffer for scratch.
   */
    result = read_header_segment( deblock_buffer );
  }
  if ( result >= 0 ) {
    if ( ftape_old_ftape ) {
      TRACE( 1, "Error: cartridge soft write protected, don't write old ftapes");
      last_write_failed = 1;
      result = -EROFS;
    } else if ( (req_len % ftape_block_size) != 0 ) {
      TRACEx2( 2,"Error: write-count %d must be multiple of block-size %d",
                       req_len, ftape_block_size );
      result= -EINVAL;
    } else if( ftape_use_compression ) {
      if ( ftape_data_pos > (ftape_total_data_amount - ftape_block_size - 8)) {
        /*
         * compressed segment:
         *
         *   Offset 0   : first-block-offset (2 bytes)
         *   Offset 2   : remainder from previous segment (n bytes)
         *   Offset n+2 : file-set-offset of first cluster (4 bytes)
         *   Offset n+6 : length of cluster (2 bytes)
         *   Offset n+8 : cluster (at most ftape_block_size bytes)
         *
         * Thus a cluster consumes in the worst case ftape_block_size + 8 bytes 
         * provided that the compressed data takes at most the size of the 
         * uncompressed data (which is assured). Note that ftape_data_pos 
         * already covers the remainder from the previous segment
         *
         */
        if ( just_before_eof ) {
          TRACEx2( 3,"data_pos: %d, total_data_amount: %d",
                       ftape_data_pos, ftape_total_data_amount );
          TRACE( 1,"Error: tape is full");
          result = -ENOSPC;
        } else {
          just_before_eof = 1;
        }
      } 
    } else if ( ftape_data_pos >= ftape_total_data_amount ) {
      TRACE(-1,"Error: tape is full");
      result = -ENOSPC;         /* full is full */
    }
  }                
  if ( result >= 0  && ftape_io_state != io_writing ) {
    TRACEi( 4, "ftape_io_state = ", ftape_io_state);
    /*
     * we need this for our single-sector-access. 
     * ftape_state doesn't suffice as it give the "low-level" state
     * of the function ftape_read/write_segment()
     */
    ftape_io_state= io_writing;
    /*
     * provide for single sector access.
     */
    TRACE( 9, "first call");
    if ( ftape_qic80_mode ) {
      /*
       *  qic80-mode creates a simple volume table to fake eof-marks
       */
      result = qic80_maybe_write_beof( ftape_seg_pos, just_before_eof != 0 );
      if ( result < 0 ) {
        TRACE( 1,"Error in qic80_maybe_write_beof()");
        TRACE_EXIT;
        return result;
      }
      if ( result == 0 && 
           ftape_block_size != qic80_get_volume_block_size( ftape_seg_pos ) ) {
        /*
         *  might be the case if the user does a MTFSFM or MTBSFM to append
         *  data to an existing file. If the user forget about the blocksize
         *  he used when creating the file, he looses.
	 *
	 *  This caveat is the reason for creating the MTIOC_ZFTAPE_BLKSZ 
	 *  ioctl
         */
        TRACE( 1, "Error: Can't append to a file with different blocksize");
        TRACE_EXIT;
        return -EINVAL;
      }
      TRACEi( 7, "just_before_eof = ", just_before_eof != 0 );
      just_before_eof= 0;
      /*
       *  tell update_header_segments & co. to write it to tape
       */
      ftape_compression_map_changed = ftape_use_compression;
    }
    buf_pos_start = ftape_seg_data_pos;
    /*
     *  now merge with old data if neccessary
     */
    if( ftape_seg_pos != ftape_deblock_segment && ftape_seg_data_pos != 0 ) {
      result = read_segment_fraction( ftape_seg_pos,
                                            deblock_buffer, 0,
                                            0, ftape_seg_data_pos );
      ftape_deblock_segment= ftape_seg_pos;
      if (result < 0 && (result != -EINTR) ) {
	/*
	 *  Well, this is not really an error. It may occur when
	 *  writing to newly formatted tapes. Just remember that
	 *  we need correct ecc information for reading a segment
	 *  which cannot be there on a never before accessed segment
	 */
        TRACEx1( 2, "Warning: read_segment result: %d", result);
        memset( deblock_buffer, '\0', ftape_seg_data_pos );
        result= 0;
      }
    }
  }
  if (result < 0) {               
    TRACE_EXIT; 
    return result;
  } else {
   /*
    * do it once for all the data to be written. I really don't know, why
    * this was done inside the write-loop before. If the user wants
    * to write to a device, he has to provide a valid buffer for all
    * the data he wants to write, and not only for the data actually
    * written. This also spares us some stuff with ftape_simple_write()
    * and ftape_compress_write()
    *
    * if there is anybody out there who reads this, please fix me if 
    * I am wrong: Heine@axpmgr.physik.rwth-aachen.de
    */
    result = verify_area( VERIFY_READ, buff, req_len );
    if (result) {
      TRACE( 1, "verify_area failed");
      last_write_failed = 1;
      TRACE_EXIT;
      return result;
    }
  }
  /*
   *    This part writes data blocks to tape until the
   *    requested amount is written.
   *    The data will go in a buffer until it's enough
   *    for a segment. Then we write the segment to tape.
   *    The size of the segment is calculated before we
   *    do this. Important routines are ftape_simple_write() and
   *    ftape_compress_write(), which compresses
   *    the data. That is the reason for pre-computing the
   *    segment size; it's easier for the compression-routine
   */
  while ( req_len > 0 ) {
    TRACEi( 4, "remaining req_len:", req_len);
    TRACEi( 4, "          buf_pos:", ftape_seg_data_pos );

    this_seg_size = ftape_get_seg_size( ftape_seg_pos );
    if ( ftape_use_compression ) {
      result= ftape_compress_write( &cnt,
                                    &req_len,
                                    buff,
                                    ftape_seg_data_pos,
                                    this_seg_size,
                                    deblock_buffer );
    } else {
      result= ftape_simple_write( &cnt,
                                  &req_len, 
                                  buff, 
                                  ftape_seg_data_pos,
                                  this_seg_size, 
                                  deblock_buffer );
    }
    /*
     *   deblock_buffer does not contain a valid segment any longer
     */
    ftape_deblock_segment = -1;

    if (result < 0 ) {
      last_write_failed = 1;
      TRACEx2( 1, "Error: ftape_%s_write failed with result %d.",
                ftape_use_compression ? "compress" : "simple", result );
      /*
       *  we can't flush the buffer.
       */
      ftape_data_pos -= ftape_seg_data_pos;
      ftape_seg_data_pos = 0;
      ftape_resid -= written;
      TRACE_EXIT;
      return result;
    }
    TRACEx1( 4, "removed from user-buffer: %05d bytes.", result );
    TRACEx1( 4, "copied to deblock_buffer: %05d bytes.",   cnt );
    written          += result;
    buff             += result;
    req_len          -= result;

    ftape_seg_data_pos += cnt;
    ftape_data_pos     += cnt;
    TRACEx1( 4, "ftape_data_pos: %d bytes.",  ftape_data_pos );
    /*
     * Got one full buffer, write it to disk
     * the "while"-loop handles retries
     */
    while ( ftape_seg_data_pos == this_seg_size ) {
#if 0
      /*
       * hack for debugging compression info, really a lot of noise
       */
      qic80_decode_compression_header( deblock_buffer, ftape_seg_pos );
#endif
      result = _write_segment( ftape_seg_pos, deblock_buffer, 0);
      TRACEi( 5, "_write_segment result =", result);
      if ( result != this_seg_size ) {
        if (result == -EAGAIN) {
          TRACE( 5, "retry...");
          continue;             /* failed, retry same segment */
        }
	TRACE( 1, "Error: Uncontinuable error in _write_segment");
        /*
         *  we can't flush the buffer.
         */
        ftape_data_pos    -= ftape_seg_data_pos;
        ftape_seg_data_pos = 0;
        last_write_failed  = 1;
        ftape_resid -= written;
        if (result >= 0) {
          TRACEi( 1, "Error: Couldn't write entire segment? Partial write count ",
                                                                    result);
	  result = -EIO;
        }
        TRACE_EXIT;
        return result;
      } else {
        TRACEi( 7, "moved out of blocking buffer:", result);
        /*
         *  finish this loop
         */
        this_seg_size = -1;
        /*
         *  only full segments are written.
         *  the other more complicated stuff is done
         *  in flush_buffers()                       
         */
        ftape_seg_data_pos =  0;
        buf_pos_start      = -1;
        ftape_deblock_segment = ftape_seg_pos;
        ++ftape_seg_pos;
      }
      /* Allow us to escape from this loop with a signal !
       */
      if (current->signal & _DONT_BLOCK) {
        TRACE( 2, "interrupted by signal");
        /*
         *  no need to bother for ftape_seg_data_pos and the like,
         *  we only reach this point after writing data to tape
         */
        ftape_data_pos    -= ftape_seg_data_pos;
        ftape_seg_data_pos = 0;
        last_write_failed = 1;
        ftape_resid -= written;
        TRACE_EXIT;
        return -EINTR;          /* is this the right return value ? */
      }
    }
  } 
  TRACEi( 4, "remaining in blocking buffer:", ftape_seg_data_pos);
  TRACEi( 4, "just written bytes:", written );
  last_write_failed = 0;
  ftape_resid -= written;
  if ( !need_flush && written > 0) {
    need_flush = 1;
  }
  TRACE_EXIT;         
  return written;               /* bytes written */
}

int
ftape_fix( void)
{
  TRACE_FUN( 5, "ftape_fix");
  int result = 0;
  int status;

  if (write_protected) {
    result = -EROFS;
  } else {
    /*  This will copy header segment 2 to header segment 1
     *  Spares us a tape format operation if header 2 is still good.
     */
    if (    first_data_segment == -1
         || header_segment_1 == header_segment_2 )
    {
      header_segment_1 = 0;
      header_segment_2 = 1;
      first_data_segment = 2;
    }
    /*************************************************************************/
    /* the following lines caused the scratch_buffer to be compared with     */
    /* itself if compiled with VERIFY_HEADERS :):(                           */
    /*  result =                                                             */
    /*       read_segment( header_segment_2,scratch_buffer, &dummy,0);       */
    /*  result = ftape_ready_wait( 5 * SECOND, &status);                     */
    /*  result = ftape_write_header_segments( scratch_buffer);               */
    /*************************************************************************/
    result = read_segment( header_segment_2,
                                 deblock_buffer, 0);
    if( result >= 0 ) {
      (void)ftape_ready_wait( 5 * SECOND, &status);
      result = ftape_write_header_segments( deblock_buffer );
      first_data_segment =
      ftape_deblock_segment    = -1;
    }
  }
  TRACE_EXIT;
  return result;
}









