/*
 * mice.c - mouse definitions for gpm-Linux
 *
 * Copyright 1993   ajh@gec-mrc.co.uk (Andrew Haylett)
 * Copyright 1994   rubini@ipvvis.unipv.it (Alessandro Rubini)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

/*
 * This file is part of the mouse server. The information herein
 * is kept aside from the rest of the server to ease fixing mouse-type
 * issues. Each mouse type is expected to fill the `buttons', `dx' and `dy'
 * fields of the Gpm_Event structure and nothing more.
 *
 * The `data' parameter points to a byte-array with event data, as read
 * by the mouse device. The mouse device should return a fixed number of
 * bytes to signal an event, and that exact number is read by the server
 * before calling one of these functions.
 *
 * The conversion function defined here should return 0 on success and -1
 * on failure.
 *
 * Refer to the definition of Gpm_Type to probe further.
 */

#include <termios.h>
#include <fcntl.h>
#include <termios.h>

#include <sys/types.h>
#include <sys/stat.h> /* stat() */
#include <sys/time.h> /* select() */
#include <unistd.h>
#include <linux/fs.h> /* MAJOR */
#include <errno.h>

#include "gpmInt.h"

/*========================================================================*/
/*
 * Ok, here we are: first, provide the functions
 * The return value is the number of unprocessed bytes
 */

static int M_ms(Gpm_Event *state,  unsigned char *data)
{
  /*
   * some devices report a change of middle-button state by
   * repeating the current button state  (patch by Mark Lord)
   */
  static unsigned char prev=0;

#if 0 /* original patch */

  if (data[0] == 0x40 && !(prev|data[1]|data[2]))
    state->buttons = 2;           /* third button on MS compatible mouse */
  else
    state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  prev = state->buttons;
  state->dx=      (char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));

#else /* my own trial to keep track of all the buttons simultaneously */

  state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  state->dx=      (char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));

  if (state->dx || state->dy) /* motion: preserve buttons */
	state->buttons=prev;
  else
	{
	if (state->buttons == (prev&~GPM_B_MIDDLE))
	  state->buttons = prev^GPM_B_MIDDLE;     /* no change: toggle middle */
    else
	  state->buttons |= prev&GPM_B_MIDDLE;    /* change: preserve middle */
	
	}
  prev=state->buttons;

#endif

  return 0;
}

static int M_bare(Gpm_Event *state,  unsigned char *data)
{
  /* a bare ms protocol */
  state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  state->dx=      (char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
  return 0;
}

static int M_sun(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (char)(data[1]);
  state->dy=     -(char)(data[2]);
  return 0;
}

static int M_msc(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (char)(data[1]) + (char)(data[3]);
  state->dy=     -((char)(data[2]) + (char)(data[4]));
  return 0;
}

static int M_mm(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= data[0] & 0x07;
  state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
  state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
  return 0;
}

static int M_logi(Gpm_Event *state,  unsigned char *data) /* equal to mm */
{
  state->buttons= data[0] & 0x07;
  state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
  state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
  return 0;
}

static int M_bm(Gpm_Event *state,  unsigned char *data) /* equal to sun */
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (char)data[1];
  state->dy=     -(char)data[2];
  return 0;
}

static int M_ps2(Gpm_Event *state,  unsigned char *data)
{
  state->buttons=
    !!(data[0]&1) * GPM_B_LEFT +
    !!(data[0]&2) * GPM_B_RIGHT +
    !!(data[0]&4) * GPM_B_MIDDLE;

/* ps2 makes me crazy.... */
/* fprintf(stderr,"%02X %02X %02X\n",data[0],data[1],data[2]); */

  state->dx=   (data[0] & 0x10) ? data[1]-256 : data[1];
  state->dy= -((data[0] & 0x20) ? data[2]-256 : data[2]);
  return 0;
}

static int M_mman(Gpm_Event *state,  unsigned char *data)
{
  /*
   * the damned MouseMan has 3/4 bytes packets. The extra byte 
   * is only there if the middle button is active.
   * I get the extra byte as a packet with magic numebers in it.
   * and then switch to 4-byte mode.
   */
  static unsigned char prev=0;
  static Gpm_Type *mytype=mice; /* it is the first */

  if (data[1]==GPM_EXTRA_MAGIC_1 && data[2]==GPM_EXTRA_MAGIC_2)
	{
	/* got unexpected fourth byte */
	LOG(("Extra byte = %02x",*data));
	if (*data != 0x20) return -1; /* must be 0x20 */
	state->buttons= (prev |= GPM_B_MIDDLE);
	state->dx=state->dy=0;

	mytype->packetlen=4;
	mytype->getextra=0;
	}
  else
	{
	/* got 3/4, as expected */
	state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
	if (mytype->packetlen==4)
	  {
	  if (data[3]) state->buttons |= GPM_B_MIDDLE;
	  else {mytype->packetlen=3; mytype->getextra=1;}
	  }
	}

  /* motion is independent of packetlen... */
  state->dx= (char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy= (char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
  prev=state->buttons;
  
  return 0;
}

/*========================================================================*/
/* Then, mice should be initialized */

static int setspeed(int fd, int old, int new, unsigned short flags)
{
struct termios tty;
char *c;

  tcgetattr(fd, &tty);
    
  tty.c_iflag = IGNBRK | IGNPAR;
  tty.c_oflag = 0;
  tty.c_lflag = 0;
  tty.c_line = 0;
  tty.c_cc[VTIME] = 0;
  tty.c_cc[VMIN] = 1;

  switch (old)
    {
    case 9600:	tty.c_cflag = flags | B9600; break;
    case 4800:	tty.c_cflag = flags | B4800; break;
    case 2400:	tty.c_cflag = flags | B2400; break;
    case 1200:
    default:	tty.c_cflag = flags | B1200; break;
    }

  tcsetattr(fd, TCSAFLUSH, &tty);

  switch (new)
    {
    case 9600:	c = "*q";  tty.c_cflag = flags | B9600; break;
    case 4800:	c = "*p";  tty.c_cflag = flags | B4800; break;
    case 2400:	c = "*o";  tty.c_cflag = flags | B2400; break;
    case 1200:
    default:	c = "*n";  tty.c_cflag = flags | B1200; break;
    }

  write(fd, c, 2);
  usleep(100000);
  tcsetattr(fd, TCSAFLUSH, &tty);
  return 0;
}



static struct {
    int sample; char code[2];
    }
  sampletab[]={
    {  0,"O"},
    { 15,"J"},
    { 27,"K"},
    { 42,"L"},
    { 60,"R"},
    { 85,"M"},
    {125,"Q"},
    {1E9,"N"},
};

static Gpm_Type *I_serial(int fd, unsigned short flags, struct Gpm_Type *type)
{
int i; unsigned char c;
fd_set set; struct timeval timeout={0,0}; /* used when not debugging */


  /* change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
    setspeed(fd, i, opt_baud, flags);

#if 0 /* actually, I never understood this... */
  /* configure the sample rate */
  for (i=0;opt_sample<=sampletab[i].sample;i++)
    ;
  write(fd,sampletab[i].code,1);
#endif

#ifndef DEBUGGING
  /*
   * flush any pending input (thanks, Miguel)
   * moreover, damned mouseman's can be detected here (though it isn't)
   */
  FD_ZERO(&set);
  for(i=0; /* always */ ; i++)
	{
	FD_SET(fd,&set);
	switch(select(fd+1,&set,(fd_set *)NULL,(fd_set *)NULL,&timeout/* zero */))
	  {
	  case 1:  if (read(fd,&c,1)==0) break;
	  case -1: continue;
	  }
	break;
	}
#endif

  if (type->fun==M_ms && i==2 && c==0x33) /* Aha.. a mouseman... */
	{
	LOG(("MouseMan detected"));
	return mice; /* it is the first */
	}
  return type;
}

static Gpm_Type *I_logi(int fd, unsigned short flags, struct Gpm_Type *type)
{
int i;
struct stat buf;
int busmouse;

  /* is this a serial- or a bus- mouse? */
  if (fstat(fd,&buf)==-1) oops("fstat()");
  i=MAJOR(buf.st_rdev);
  if (stat("/dev/ttyS0",&buf)==-1) oops("stat()");
  busmouse=(i != MAJOR(buf.st_rdev));

  /* fix the howmany field, so that serial mice have 1, while busmice have 3 */
  type->howmany = busmouse ? 3 : 1;

  /* change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
    setspeed(fd, i, opt_baud, flags);

  /* this stuff is peculiar of logitech mice, also for the serial ones */
  write(fd, "S", 1);
  setspeed(fd, opt_baud, opt_baud,CS8 |PARENB |PARODD |CREAD |CLOCAL |HUPCL);

  /* configure the sample rate */
  for (i=0;opt_sample<=sampletab[i].sample;i++)
    ;
  write(fd,sampletab[i].code,1);
  return type;
}


/*========================================================================*/
/* Finally, the table */
#define STD_FLG (CREAD|CLOCAL|HUPCL)

Gpm_Type mice[]={
  {"mman", M_mman, I_serial, CS7 | STD_FLG, /* MUST BE FIRST */
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 1},
  {"ms",   M_ms, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0},
  {"bare", M_bare, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00}, 3, 1, 0},
  {"sun",  M_sun, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00}, 3, 1, 0},
  {"msc",  M_msc, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00}, 5, 1, 0},
  {"mm",   M_mm, I_serial, CS8 | PARENB|PARODD | STD_FLG,
                                {0xe0, 0x80, 0x80, 0x00}, 3, 1, 0},
  {"logi", M_logi, I_logi, CS8 | CSTOPB | STD_FLG,
                                {0xe0, 0x80, 0x80, 0x00}, 3, 3, 0},
  {"bm",   M_bm, NULL,  STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00}, 3, 3, 0},
  {"ps2",  M_ps2, NULL,  STD_FLG,
                                {0xc0, 0x00, 0x00, 0x00}, 3, 1, 0},

  {"",     NULL, NULL, 0,       {0x00, 0x00, 0x00, 0x00}, 0, 0, 0}
};




