/* #Specification: /etc/conf.modules / format
	Modules may be located at different place in the filesystem.
	We expect some standard to emerge. We expect that the
	FSSTND will address this in the future.

	There will always be some need to override this, especially for
	modules developpers.

	The file /etc/conf.modules will contain different definition to
	control the manipulation of the module.

	The format will be fairly simple:

	parameter=value
	.
	parameter=value

	Standard Unix style comments and continuation line are supported.
	Comments begin with a # and continue until the end of the line.
	A line continue on the next one if the last non-white character
	is a \.
*/
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include "link.h"

struct PATH_TYPE{
	char *type;
	char *path;
};

static PATH_TYPE tb[100];
static int nb=0;
static char *depfile=NULL;

/*
	Do a popen with error message reporting.
	If quit != 0, the function does not return if an error occur.

	Return the FILE *. Must be close with pclose().
*/
FILE *popen_err (
	const char *cmd,
	const char *mode,
	int quit)
{
	FILE *ret = popen (cmd,mode);
	if (ret == NULL){
		fprintf (stderr,"Can't execute: %s\n",cmd);
		if (quit) exit(-1);
	}
	return ret;
}

/*
	Read the configuration file.
	Return -1 if any error. Error messages a generated on stderr.
*/
int config_read ()
{
	int ret = 0;
	FILE *fin = fopen (ETC_CONF_MODULES,"r");
	char depfile_tmp[PATH_MAX];
	depfile_tmp[0] = '\0';
	if (fin == NULL){
		/* #Specification: /etc/conf.modules / missing
			This file is optionnal. No error is printed if it
			is missing. If it is missing the following content
			is assumed.

			path[boot]=/lib/modules/boot
			path[misc]=/lib/modules/`uname -r`/misc
			path[fs]=/lib/modules/`uname -r`/fs
			path[net]=/lib/modules/`uname -r`/net

			path[misc]=/lib/modules/default/misc
			path[fs]=/lib/modules/default/fs
			path[net]=/lib/modules/default/net
		
			path[misc]=/lib/modules/misc
			path[fs]=/lib/modules/fs
			path[net]=/lib/modules/net
			
			The idea is that modprobe will look first it the
			modules compiled for the current release of the kernel.
			If not found, it will look into the default release.
			And if not found, it will look in the other directory.
			
			The strategy should be like this. When you install a
			new linux, the modules should go in a directory
			related to the release of the kernel you are installing.
			Then you do a symlink default to this directory.
			
			Each time you compile a new kernel, the make modules_install
			will set new directory, but won't change de default.

			When you get a module unrelated to the kernel distribution
			you place it in one of the last three directory.
			
			This is the default strategy. Off course you can overide
			this in /etc/conf.modules.
		*/
		tb[0].type = strdup_err("boot");
		tb[0].path = strdup_err("/lib/modules/boot");

		tb[1].type = strdup_err("fs");
		tb[1].path = strdup_err("/lib/modules/`uname -r`/fs");
		tb[2].type = strdup_err("misc");
		tb[2].path = strdup_err("/lib/modules/`uname -r`/misc");
		tb[3].type = strdup_err("net");
		tb[3].path = strdup_err("/lib/modules/`uname -r`/net");

		tb[4].type = strdup_err("fs");
		tb[4].path = strdup_err("/lib/modules/default/fs");
		tb[5].type = strdup_err("misc");
		tb[5].path = strdup_err("/lib/modules/default/misc");
		tb[6].type = strdup_err("net");
		tb[6].path = strdup_err("/lib/modules/default/net");

		tb[7].type = strdup_err("fs");
		tb[7].path = strdup_err("/lib/modules/fs");
		tb[8].type = strdup_err("misc");
		tb[8].path = strdup_err("/lib/modules/misc");
		tb[9].type = strdup_err("net");
		tb[9].path = strdup_err("/lib/modules/net");
		nb = 10;
	}else{
		char buf[1000];
		int noline = 0;
		while (fgets_strip(buf,sizeof(buf)-1,fin,&noline)!=NULL){
			char *parm = str_skip (buf);
			if (*parm != '\0'){
				int one_err = 1;
				char *pt = parm;
				while (*pt > ' ' && *pt != '=') pt++;
				if (pt > parm){
					if (*pt != '='){
						*pt++ = '\0';
						pt = str_skip (pt);
					}
					if (*pt == '='){
						*pt++ = '\0';
						pt = str_skip (pt);
						if (strcmp(parm,"depfile")==0){
							strcpy (depfile_tmp,pt);
							one_err = 0;
						}else if (strncmp(parm,"path",4)==0){
							/* #Specification: config file / path parameter
								The path parameter specify a directory to
								search for module. This parameter may
								be repeated multiple time.

								Optionnally the path parameter carries
								a tag. This tells us a little more about
								the purpose of this directory and
								allows some automated operations.
								The tag is sticked to the path word
								enclose in square braket.
								#
								path[boot]=/lib/modules/boot
								#

								This identify the path a of directory
								holdding modules loadable a boot time.

								Hopefully, insmod will have an option
								to load a list of module using
								such a tag.

								If the tag is missing, the word "misc"
								is assumed.
							*/
							assert (nb<sizeof(tb)/sizeof(tb[0]));	
							if (parm[4] == '\0'){
								tb[nb].type = strdup_err ("misc");
								tb[nb].path = strdup_err (pt);
								nb++;
								one_err = 0;
							}else if (parm[4] == '['){
								char *pt_type = parm+5;
								while (*pt_type != '\0' && *pt_type != ']'){
									pt_type++;
								}
								if (*pt_type == ']' && pt_type[1] == '\0'){
									*pt_type = '\0';		
									tb[nb].type = strdup_err (parm+5);
									tb[nb].path = strdup_err (pt);
									nb++;
									one_err = 0;
								}
							}
						}
					}
				}
				if (one_err){
					fprintf (stderr,"Invalid line %d in " ETC_CONF_MODULES "\n\t%s\n"
						,noline,buf);
					ret = -1;
				}
			}
		}
		fclose (fin);
	}
	if (ret != -1){
		if (depfile_tmp[0] == '\0'){
			/* #Specification: config file / depfile parameter
				The default value for the depfile parameter is:

				depfile=/lib/modules/`uname -r`/modules.dep

				If the config file exist but lack a depfile
				specification, it is used also since the system
				can't work without one.

				Once we have the depfile value, we pass to the shell
				with a popen() to resolve whatever shell construct in
				its value. We execute a echo command.

				echo read_value
			*/
			strcpy (depfile_tmp,"/lib/modules/`uname -r`/modules.dep");
		}
		char cmd[PATH_MAX+10];
		sprintf (cmd,"echo %s",depfile_tmp);
		FILE *fin = popen_err (cmd,"r",0);
		if (fin != NULL){
			if (fgets(depfile_tmp,sizeof(depfile_tmp)-1,fin)!=NULL){
				strip_end (depfile_tmp);
				depfile = strdup_err (depfile_tmp);
			}else{
				ret = -1;
			}
			pclose (fin);
		}
	}
	return ret;
}

/*
	Find out all modules matching the name "match" in directory of type "type"

	Return the number of module located and placed in lst[]. The caller
	must free all entry in lst[] itself by doing a tbstr_free(lst,nb);
*/
int config_lstmod (
	const char *match,
	const char *type,		// Type of directory (path[type]), or NULL
							// to match all
	char *lst[])
{
	/* #Specification: reading directory / ls command
		We are using the command ls to locate entries in directories.
		This has the advantage of allowing complex wildcard specification
		in /etc/conf.modules. For example.

		path[misc]=/lib/modules/1.1.5?/misc
	*/
	char cmd[1000];
	int find_one = 0;
	{
		char *pt = stpcpy (cmd,"/bin/ls ");
		for (int i=0; i<nb; i++){
			if (type == NULL || strcmp(tb[i].type,type)==0){
				pt = stpcpy (pt,tb[i].path);
				*pt++ = '/';
				pt = stpcpy (pt,match);
				*pt++ = ' ';
				find_one = 1;
			}
		}
		*pt = '\0';
	}
	int ret = 0;
	if (find_one){
		int fd2 = dup(2);
		if (fd2 == -1){
			fprintf (stderr,"Out of file handle\n");
		}else{
			close (2);
			FILE *fin = popen_err (cmd,"r",1);
			if (fin != NULL){
				while (fgets(cmd,sizeof(cmd)-1,fin)!=NULL){
					strip_end (cmd);
					lst[ret++] = strdup_err (cmd);
				}
				pclose (fin);
			}
			dup2(fd2,2);
			close (fd2);
		}
	}
	return ret;
}

/*
	Return the path of the depandancy file to produce
*/
const char *config_getdepfile()
{
	return depfile;
}

#ifdef TEST

int main (int argc, char *argv[])
{
	if (argc != 3){
		fprintf (stderr,"type and match\n");
	}else if (config_read() == -1){
		fprintf (stderr,"Can't read configuration file\n");
	}else{
		char *lst[1000];
		int nb = config_lstmod (argv[1],argv[2],lst);
		for (int i=0; i<nb; i++){
			printf ("\t%s\n",lst[i]);
		}
		tbstr_free (lst,nb);
	}
	return 0;
}

#endif

