------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                U I N T P                                 --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.23 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  Support for universal integer arithmetic

--  WARNING: There is a C version of this package. Any changes to this
--  source file must be properly reflected in the C header file sinfo.h

with System; use System;
with Ttypes; use Ttypes;
with Types;  use Types;
with Table;

package Uintp is

   --------------------
   -- Uint Constants --
   --------------------

   --  Note: these are all in the directly represented range, so it is
   --  legitimate to use the = and /= operators directly on these values,
   --  rather than using the UI_Eq and UI_Ne functions

   Uint_0  : constant Uint := Uint_Direct_Bias;
   Uint_1  : constant Uint := Uint_Direct_Bias + 1;
   Uint_2  : constant Uint := Uint_Direct_Bias + 2;
   Uint_3  : constant Uint := Uint_Direct_Bias + 3;
   Uint_4  : constant Uint := Uint_Direct_Bias + 4;
   Uint_5  : constant Uint := Uint_Direct_Bias + 5;
   Uint_6  : constant Uint := Uint_Direct_Bias + 6;
   Uint_7  : constant Uint := Uint_Direct_Bias + 7;
   Uint_8  : constant Uint := Uint_Direct_Bias + 8;
   Uint_9  : constant Uint := Uint_Direct_Bias + 9;
   Uint_10 : constant Uint := Uint_Direct_Bias + 10;
   Uint_16 : constant Uint := Uint_Direct_Bias + 16;
   Uint_32 : constant Uint := Uint_Direct_Bias + 32;
   Uint_64 : constant Uint := Uint_Direct_Bias + 64;

   Uint_Minus_1 : constant Uint := Uint_Direct_Bias - 1;
   Uint_Minus_2 : constant Uint := Uint_Direct_Bias - 2;
   Uint_Minus_3 : constant Uint := Uint_Direct_Bias - 3;
   Uint_Minus_4 : constant Uint := Uint_Direct_Bias - 4;
   Uint_Minus_5 : constant Uint := Uint_Direct_Bias - 5;
   Uint_Minus_6 : constant Uint := Uint_Direct_Bias - 6;
   Uint_Minus_7 : constant Uint := Uint_Direct_Bias - 7;
   Uint_Minus_8 : constant Uint := Uint_Direct_Bias - 8;
   Uint_Minus_9 : constant Uint := Uint_Direct_Bias - 9;

   -----------------
   -- Subprograms --
   -----------------

   procedure Initialize;
   --  Initialize Uint tables

   function Udigits_Address return System.Address;
   --  Return address of Udigits table (used for Back_End call to Gigi)

   function Uints_Address return System.Address;
   --  Return address of Uints table (used for Back_End call to Gigi)

   function UI_Abs (Right : Uint) return Uint;
   pragma Inline (UI_Abs);
   --  Returns abs function of universal integer.

   function UI_Difference (Left, Right : Uint) return Uint;
   pragma Inline (UI_Difference);
   --  Returns difference of two universal integers.

   function UI_Eq (Left, Right : Uint) return Boolean;
   pragma Inline (UI_Eq);
   --  Compares universal integers for equality.

   function UI_Exponentiate (Left, Right : Uint) return Uint;
   --  Returns result of exponentiating two universal integers.
   --  Fatal error if Right is negative.

   function UI_From_Int (Input : Int) return Uint;
   --  Converts Int value to universal integer form.

   function UI_Ge (Left, Right : Uint) return Boolean;
   pragma Inline (UI_Ge);
   --  Compares universal integers for greater than or equal.

   function UI_Gt (Left, Right : Uint) return Boolean;
   pragma Inline (UI_Gt);
   --  Compares universal integers for greater than.

   UI_Image_Buffer : String (1 .. Positive (Standard_Long_Long_Integer_Width));
   UI_Image_Length : Natural;

   procedure UI_Image (Input : Uint);
   --  Places a representation of Uint, consisting of a possible minus sign,
   --  followed by a string of decimal digits, in UI_Image_Buffer, setting
   --  UI_Image_Length to the number of stored characters. The value must be
   --  in the range of the target type Long_Long_Integer. See also UI_Write
   --  which does not have this range limitation.

   function UI_Is_In_Int_Range (Input : Uint) return Boolean;
   pragma Inline (UI_Is_In_Int_Range);
   --  Determines if universal integer is in Int range.

   function UI_Is_Negative (Input : Uint) return Boolean;
   pragma Inline (UI_Is_Negative);
   --  Determines if universal integer value is less than zero.

   function UI_Is_Positive (Input : Uint) return Boolean;
   pragma Inline (UI_Is_Positive);
   --  Determines if universal integer value is greater than zero.

   function UI_Is_Zero (Input : Uint) return Boolean;
   pragma Inline (UI_Is_Zero);
   --  Determines if universal integer value is zero.

   function UI_Le (Left, Right : Uint) return Boolean;
   pragma Inline (UI_Le);
   --  Compares universal integers for less than or equal.

   function UI_Lt (Left, Right : Uint) return Boolean;
   --  Compares universal integers for less than.

   function UI_Max (Left, Right : Uint) return Uint;
   --  Returns maximum of two universal integers

   function UI_Min (Left, Right : Uint) return Uint;
   --  Returns minimum of two universal integers

   function UI_Mod (Left, Right : Uint) return Uint;
   pragma Inline (UI_Mod);
   --  Returns mod function of two universal integers.

   function UI_Ne (Left, Right : Uint) return Boolean;
   pragma Inline (UI_Ne);
   --  Compares universal integers for inequality.

   function UI_Negate (Right : Uint) return Uint;
   pragma Inline (UI_Negate);
   --  Returns negative of universal integer.

   function UI_Product (Left, Right : Uint) return Uint;
   --  Returns product of two universal integers.

   function UI_Quotient (Left, Right : Uint) return Uint;
   --  Returns quotient of two universal integers. Fatal error if
   --  Right is zero.

   function UI_Rem (Left, Right : Uint) return Uint;
   pragma Inline (UI_Rem);
   --  Returns rem function of two universal integers.

   function UI_Sum (Left, Right : Uint) return Uint;
   --  Returns sum of two universal integers.

   function UI_To_Int (Input : Uint) return Int;
   --  Converts universal integer value to Int. Fatal error
   --  if value is not in appropriate range.

   procedure UI_Write (Input : Uint);
   --  Writes value of Uint as string of decimal digits, preceded by a
   --  possible minus sign, to the output file.

end Uintp;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.21
--  date: Sun Apr  3 14:35:13 1994;  author: dewar
--  (UI_Image): New calling sequence
--  (UI_Write): New procedure
--  ----------------------------
--  revision 1.22
--  date: Sat May  7 18:38:16 1994;  author: dewar
--  (UI_Max, UI_Min): New functinos
--  ----------------------------
--  revision 1.23
--  date: Tue May 10 15:57:03 1994;  author: porter
--  Convert Standard_Long_Long_Integer_Width to Positive in String
--   buound now that ttypes defines it as Pos
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
