/*
 * mice.c - mouse definitions for gpm-Linux
 *
 * Copyright 1993   ajh@gec-mrc.co.uk (Andrew Haylett)
 * Copyright 1994   rubini@ipvvis.unipv.it (Alessandro Rubini)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

/*
 * This file is part of the mouse server. The information herein
 * is kept aside from the rest of the server to ease fixing mouse-type
 * issues. Each mouse type is expected to fill the `buttons', `dx' and `dy'
 * fields of the Gpm_Event structure and nothing more.
 *
 * The `data' parameter points to a byte-array with event data, as read
 * by the mouse device. The mouse device should return a fixed number of
 * bytes to signal an event, and that exact number is read by the server
 * before calling one of these functions.
 *
 * The conversion function defined here should return 0 on success and -1
 * on failure.
 *
 * Refer to the definition of Gpm_Type to probe further.
 */

#include <termios.h>
#include <fcntl.h>
#include "gpmInt.h"
#include <termios.h>


/*========================================================================*/
/* Ok, here we are: first, provide the functions */

static int M_ms(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
  state->dx=      (char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
  state->dy=      (char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
  return 0;
}

static int M_sun(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (char)(data[1]);
  state->dy=     -(char)(data[2]);
  return 0;
}

static int M_msc(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (char)(data[1]) + (char)(data[3]);
  state->dy=     -((char)(data[2]) + (char)(data[4]));
  return 0;
}

static int M_mm(Gpm_Event *state,  unsigned char *data)
{
  state->buttons= data[0] & 0x07;
  state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
  state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
  return 0;
}

static int M_logi(Gpm_Event *state,  unsigned char *data) /* equal to mm */
{
  state->buttons= data[0] & 0x07;
  state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
  state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
  return 0;
}

static int M_bm(Gpm_Event *state,  unsigned char *data) /* equal to sun */
{
  state->buttons= (~data[0]) & 0x07;
  state->dx=      (char)data[1];
  state->dy=     -(char)data[2];
  return 0;
}

static int M_ps2(Gpm_Event *state,  unsigned char *data)
{
  state->buttons=
    !!(data[0]&1) * GPM_B_LEFT +
    !!(data[0]&2) * GPM_B_RIGHT +
    !!(data[0]&4) * GPM_B_MIDDLE;

/* ps2 makes me crazy.... */
/* fprintf(stderr,"%02X %02X %02X\n",data[0],data[1],data[2]); */

  state->dx=   (data[0] & 0x10) ? data[1]-256 : data[1];
  state->dy= -((data[0] & 0x20) ? data[2]-256 : data[2]);
  return 0;
}

#ifndef linux /* this is used to slowly read data from a file (serial mouse) */

static int M_test(Gpm_Event *state,  unsigned char *data)
{sleep(1); return M_ms(state, data);}

#endif

/*========================================================================*/
/* Then, mice should be initialized */

static int setspeed(int fd, int old, int new, unsigned short flags)
{
struct termios tty;
char *c;

  tcgetattr(fd, &tty);
    
  tty.c_iflag = IGNBRK | IGNPAR;
  tty.c_oflag = 0;
  tty.c_lflag = 0;
  tty.c_line = 0;
  tty.c_cc[VTIME] = 0;
  tty.c_cc[VMIN] = 1;

  switch (old)
    {
    case 9600:	tty.c_cflag = flags | B9600; break;
    case 4800:	tty.c_cflag = flags | B4800; break;
    case 2400:	tty.c_cflag = flags | B2400; break;
    case 1200:
    default:	tty.c_cflag = flags | B1200; break;
    }

  tcsetattr(fd, TCSAFLUSH, &tty);

  switch (new)
    {
    case 9600:	c = "*q";  tty.c_cflag = flags | B9600; break;
    case 4800:	c = "*p";  tty.c_cflag = flags | B4800; break;
    case 2400:	c = "*o";  tty.c_cflag = flags | B2400; break;
    case 1200:
    default:	c = "*n";  tty.c_cflag = flags | B1200; break;
    }

  write(fd, c, 2);
  usleep(100000);
  tcsetattr(fd, TCSAFLUSH, &tty);
  return 0;
}



static struct {
    int sample; char code[2];
    }
  sampletab[]={
    {  0,"O"},
    { 15,"J"},
    { 27,"K"},
    { 42,"L"},
    { 60,"R"},
    { 85,"M"},
    {125,"Q"},
    {1E9,"N"},
};

static int I_serial(int fd, unsigned short flags)
{
int i;

/* change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
    setspeed(fd, i, opt_baud, flags);

/* configure the sample rate */
  for (i=0;opt_sample<=sampletab[i].sample;i++)
    ;
  write(fd,sampletab[i].code,1);
  return 0;
}

static int I_logi(int fd, unsigned short flags)
{
int i;

/* change from any available speed to the chosen one */
  for (i=9600; i>=1200; i/=2)
    setspeed(fd, i, opt_baud, flags);

/* this stuff is peculiar of logitech mice */
    write(fd, "S", 1);
    setspeed(fd, opt_baud, opt_baud,CS8 |PARENB |PARODD |CREAD |CLOCAL |HUPCL);

/* configure the sample rate */
  for (i=0;opt_sample<=sampletab[i].sample;i++)
    ;
  write(fd,sampletab[i].code,1);
  return 0;
}


/*========================================================================*/
/* Finally, the table */
#define STD_FLG (CREAD|CLOCAL|HUPCL)

Gpm_Type mice[]={
  {"ms",   "/dev/ttyS0",        M_ms, I_serial, CS7 | STD_FLG,
                                {0x40, 0x40, 0x40, 0x00, 3}},
  {"sun",  "",                  M_sun, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00, 3}},
  {"msc",  "/dev/ttyS0",        M_msc, I_serial, CS8 | CSTOPB | STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00, 5}},
  {"mm",   "",                  M_mm, I_serial, CS8 | PARENB|PARODD | STD_FLG,
                                {0xe0, 0x80, 0x80, 0x00, 3}},
  {"logi", "/dev/bmouselogitec",M_logi, I_logi, CS8 | CSTOPB | STD_FLG,
                                {0xe0, 0x80, 0x80, 0x00, 3}},
  {"bm",   "/dev/bmousems",     M_bm, NULL,  STD_FLG,
                                {0xf8, 0x80, 0x00, 0x00, 5}},
  {"ps2",  "/dev/bmouseps2",    M_ps2, NULL,  STD_FLG,
                                {0xc0, 0x00, 0x00, 0x00, 3}},

#ifndef linux
  {"test", "./data",            M_test, NULL, 0,
     {0x40, 0x40, 0x40, 0x00, 3}},
#endif

  {"",     "",                  NULL, NULL, 0,   {0x00, 0x00, 0x00, 0x00, 0}}
};




