/*
 *      Copyright (C) 1993 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 $Source: /usr/src/distr/ftape-0.9.9d/RCS/fdc-io.c,v $
 $Author: bas $
 *
 $Revision: 1.19 $
 $Date: 1994/01/16 15:44:54 $
 $State: ALPHA $
 *
 *      This file contains the low-level floppy disk interface code
 *      for the QIC-40/80 tape streamer device driver.
 */

static char RCSid[] = "$Id: fdc-io.c,v 1.19 1994/01/16 15:44:54 bas ALPHA $";

#include <linux/errno.h>
#include <linux/sched.h>
#include <asm/system.h>
#include <asm/io.h>
#include <asm/dma.h>

#include "fdc-io.h"
#include "ftape-io.h"
#include "ftape-rw.h"
#include "calibr.h"

/*      Global vars.
 */
int ftape_unit = -1;
int ftape_motor = 0;
int current_cylinder = -1;
int expect_stray_interrupt = 0;
int seek_completed = 0;
int interrupt_seen = 0;
volatile unsigned char fdc_head; /* FDC head */
volatile unsigned char fdc_cyl; /* FDC track */
volatile unsigned char fdc_sect; /* FDC sector */

/*      Local vars.
 */
static int fdc_calibr_count;
static int fdc_calibr_time;
static int fdc_confused = 0;
static int fdc_status;

#define RQM_DELAY (12)

#undef CHECK_TIMING

#ifdef CHECK_TIMING
static struct timeval t1 = { 0, 0 }, t2 = { 0, 0 };
static int delta_t;
static int new_command;
#endif

#define abs(a) ((a) < 0 ? -(a) : (a))

/*  Wait during a timeout period for a given FDC status.
 *  If usecs == 0 then just test status, else wait at least for usecs.
 *  Returns -ETIME on timeout. Function must be calibrated first !
 */
int
fdc_wait( int usecs, unsigned char mask, unsigned char state)
{
  int count_1 = (fdc_calibr_count * usecs - 1) / fdc_calibr_time;

  do {
    fdc_status = inb_p( FDC_STATUS);
    if ((fdc_status & mask) == state) {
      return 0;
    }
  } while (count_1-- >= 0);
  return -ETIME;
}

int
fdc_ready_wait( int usecs)
{
  return fdc_wait( usecs, FDC_DATA_READY, FDC_DATA_READY);
}

static void
fdc_usec_wait( int usecs)
{
  fdc_wait( usecs, 0, 1);       /* will always timeout ! */
}

int
fdc_ready_out_wait( int usecs)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  return fdc_wait( usecs, FDC_DATA_OUT_READY, FDC_DATA_OUT_READY);
}

int
fdc_ready_in_wait( int usecs)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  return fdc_wait( usecs, FDC_DATA_OUT_READY, FDC_DATA_IN_READY);
}

int
fdc_wait_calibrate( void)
{
  return calibrate( "fdc_wait",
                   fdc_usec_wait, &fdc_calibr_count, &fdc_calibr_time);
}

/*  Wait for a (short) while for the FDC to become ready
 *  and transfer the next command byte.
 *  Return -ETIME on timeout on getting ready (depends on hardware!).
 */
int
fdc_write( unsigned char data)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  if (fdc_wait( 150, FDC_DATA_READY_MASK, FDC_DATA_IN_READY) < 0) {
    return -ETIME;
  } else {
    outb( data, FDC_DATA);
    return 0;
  }
}

/*  Wait for a (short) while for the FDC to become ready
 *  and transfer the next result byte.
 *  Return -ETIME if timeout on getting ready (depends on hardware!).
 */
int
fdc_read( unsigned char* data)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  if (fdc_wait( 150, FDC_DATA_READY_MASK, FDC_DATA_OUT_READY) < 0) {
    return -ETIME;
  } else {
    *data = inb( FDC_DATA);
    return 0;
  }
}

/*  Output a cmd_len long command string to the FDC.
 *  The FDC should be ready to receive a new command or
 *  an error (EBUSY) will occur.
 */
int
fdc_command( unsigned char* cmd_data, int cmd_len)
{
  int result = 0;
  unsigned long flags;
  int count = cmd_len;

  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  save_flags( flags);
  cli();
  fdc_status = inb( FDC_STATUS);
  if ((fdc_status & FDC_DATA_READY_MASK) == FDC_DATA_IN_READY) {
    int retry = 0;
    fdc_mode = *cmd_data;       /* used by isr */
    interrupt_seen = 0;
    while (count) {
      result = fdc_write( *cmd_data);
      if (result < 0) {
        TRACEx3( 6, "fdc_command",
                "fdc_mode = %02x, status = %02x at index %d",
                (int) fdc_mode, (int) fdc_status, cmd_len - count);
        if (++retry <= 3) {
          TRACE( 2, "fdc_command", "fdc_write timeout, retry");
        } else {
          TRACE( 1, "fdc_command", "fdc_write timeout, fatal");
          fdc_confused = 1;
          /* recover ??? */
          break;
        }
      } else {
        --count;
        ++cmd_data;
      }
    }
  } else {
    TRACE( 1, "fdc_command", "fdc not ready");
    result = -EBUSY;
  }
  restore_flags( flags);
  return result;
}

/*  Input a res_len long result string from the FDC.
 *  The FDC should be ready to send the result or an error
 *  (EBUSY) will occur.
 */
int
fdc_result( unsigned char* res_data, int res_len)
{
  int result = 0;
  unsigned long flags;
  int count = res_len;

  save_flags( flags);
  cli();
  fdc_status = inb( FDC_STATUS);
  if ((fdc_status & FDC_DATA_READY_MASK) == FDC_DATA_OUT_READY) {
    int retry = 0;
    while (count) {
      if (!(fdc_status & FDC_BUSY)) {
        TRACE( 1, "fdc_result", "premature end of result phase");
      }
      result = fdc_read( res_data);
      if (result < 0) {
        TRACEx3( 6, "fdc_result",
                "fdc_mode = %02x, status = %02x at index %d",
                (int) fdc_mode, (int) fdc_status, res_len - count);
        if (++retry <= 3) {
          TRACE( 2, "fdc_result", "fdc_read timeout, retry");
        } else {
          TRACE( 1, "fdc_result", "fdc_read timeout, fatal");
          fdc_confused = 1;
          /* recover ??? */
          break;
        }
      } else {
        --count;
        ++res_data;
      }
    }
  } else {
    TRACE( 1, "fdc_result", "fdc not ready");
    result = -EBUSY;
  }
  restore_flags( flags);
  fdc_usec_wait( RQM_DELAY);    /* allow FDC to negate BSY */
  return result;
}

/*      Handle command and result phases for
 *      commands without data phase.
 */
int
fdc_issue_command( unsigned char *out_data, int out_count,
                  unsigned char *in_data, int in_count)
{
  int result;
  int t0, t1;

  if (out_count > 0) {
    result = fdc_command( out_data, out_count);
    if (result < 0) {
      TRACE( 1, "fdc_issue_command", "fdc_command failed");
      return result;
    }
  }
  /* will take 24 - 30 usec for fdc_sense_drive_status and
   * fdc_sense_interrupt_status commands.
   *    35 fails sometimes (5/9/93 SJL)
   * On a loaded system it incidentally takes longer than
   * this for the fdc to get ready ! ?????? WHY ??????
   * So until we know what's going on use a very long timeout.
   */
  t0 = timestamp();
  result = fdc_ready_out_wait( 500 /* usec */);
  t1 = timestamp();
  if (result < 0) {
    TRACEi( 1, "fdc_issue_command",
           "fdc_ready_out_wait failed after:", t0 - t1);
    return result;
  }
  if (in_count > 0) {
    result = fdc_result( in_data, in_count);
    if (result < 0) {
      TRACE( 1, "fdc_issue_command", "result phase aborted");
      return result;
    }
  }
  return 0;
}

/*      FDC interrupt service routine.
 */
void
fdc_isr( void)
{
  int result;
  int status;
  static unsigned int last_error_id = (unsigned) -1;
  static int soft_error_retry_count = 0;
  enum { no_error = 0, id_am_error = 0x01, id_crc_error = 0x02,
         data_am_error = 0x04, data_crc_error = 0x08,
         no_data_error =0x10, overrun_error = 0x20,
       } cause = no_error;
  char* fdc_mode_text[] = {
    "fdc_idle", "fdc_reading", "fdc_seeking",
    "fdc_writing_data", "fdc_reading_id", "fdc_recalibrating",
  };
  unsigned char in[ 7];
  static int retrying = 0;
  static int ketchup = 0;
#ifdef RETRY_HACK
  static int retry_overrule = 0;
#endif

  do_floppy = fdc_isr;          /* hook our handler into the fdc code */

#ifdef CHECK_TIMING
  do_gettimeofday( &t2);
  /* hack to get around un-initialized t1 on unexpected interrupt */
  if ((t1.tv_usec | t1.tv_sec) == 0) {
    t1 = t2;                    /* will give warning! */
  }
  /* end of hack */
  delta_t = (t2.tv_sec - t1.tv_sec) * 1000000 + (t2.tv_usec - t1.tv_usec);
  if (new_command) {
    if ((delta_t < 2000 * new_command && tracing > 1) || delta_t < 0) {
      TRACEi( 2, "fdc_isr", "Fast interrupt for command:", new_command);
      TRACEi( 2, "fdc_isr",
             "Lapsed time command to interrupt (usec):", delta_t);
    }
    new_command = 0;
  } else {
    if (tracing > 4 || delta_t < 0) {
      TRACEi( 5, "fdc_isr", "Time to next interrupt (usec):", delta_t);
    }
  }
  t1 = t2;
#endif

  status = inb_p( FDC_STATUS);
  
  if (status & FDC_BUSY) {
    /*  Entering Result Phase.
     */
    result = fdc_result( in, 7); /* better get it fast ! */
    if (result < 0) {
      /*  Entered unknown state...
       */
      TRACE( 1, "fdc_isr", "probably fatal error during FDC Result Phase");
      TRACE( 1, "fdc_isr", "drive may hang until (power) reset :-(");
      /*  what to do next ????
       */
    } else {
      int i;
      char* fdc_mode_txt;
      /*  Valid in, decode cause of interrupt.
       */
      switch (fdc_mode) {
      case fdc_idle:
        fdc_mode_txt = fdc_mode_text[ 0];
        break;
      case fdc_reading_data:
        fdc_mode_txt = fdc_mode_text[ 1];
        break;
      case fdc_seeking:
        fdc_mode_txt = fdc_mode_text[ 2];
        break;
      case fdc_writing_data:
        fdc_mode_txt = fdc_mode_text[ 3];
        break;
      case fdc_reading_id:
        fdc_mode_txt = fdc_mode_text[ 4];
        break;
      case fdc_recalibrating:
        fdc_mode_txt = fdc_mode_text[ 5];
        break;
      default:
        fdc_mode_txt = "unknown";
        break;
      }
      switch (in[ 0] & ST0_INT_MASK) {
      case FDC_INT_NORMAL:
        TRACEx1( 5, "fdc_isr", "normal command completion: %s",
                fdc_mode_txt);
        break;
      case FDC_INT_ABNORMAL:
        TRACEx1( 5, "fdc_isr", "abnormal command completion during %s",
                fdc_mode_txt);
        TRACEx3( 6, "fdc_isr", "sr0: 0x%02x, sr1: 0x%02x, sr2: 0x%02x",
                in[ 0], in[ 1], in[ 2]);
        TRACEx4( 6, "fdc_isr", "cyl: 0x%02x, head: 0x%02x, "
                "sect: 0x%02x, size: 0x%02x", in[ 3], in[ 4], in[ 5], in[ 6]);
        if (in[ 1] & 0x01) {
          if (in[ 2] & 0x01) {
            cause = data_am_error;
          } else {
            cause = id_am_error;
          }
        } else if (in[ 1] & 0x20) {
          if (in[ 2] & 0x20) {
            cause = data_crc_error;
          } else {
            cause = id_crc_error;
          }
        } else if (in[ 1] & 0x04) {
          cause = no_data_error;
        } else if (in[ 1] & 0x10) {
          cause = overrun_error;
        }
        break;
      case FDC_INT_INVALID:
        TRACEx1( 5, "fdc_isr", "invalid command completion during %s",
                fdc_mode_txt);
        break;
      case FDC_INT_READYCH:
        break;
        TRACEx1( 5, "fdc_isr", "ready change command completion during %s",
                fdc_mode_txt);
      default:
      }
      
      /*      expected command completion, check status.
       */
      for (i = 0; i < NR_ITEMS( buffer); ++i) {
        TRACEx3( 8, "fdc_isr", "buffer[%d] status: %d, segment_id: %d",
                i, buffer[i].status, buffer[i].segment_id);
      }
      switch (fdc_mode) {
        
      case fdc_reading_data: {
        
        int must_retry = 0;
        int have_error = 0;
        
        TRACEi( 5, "fdc_isr", "fdc irq reading segment",
               buffer[ head].segment_id);
        
        if (runner_status == aborting || runner_status == do_abort) {
          TRACEx1( 5, "fdc_isr", "aborting %s", fdc_mode_txt);
          break;
        }
        if (buffer[ head].segment_id != last_error_id) {
          last_error_id = buffer[ head].segment_id;
          retrying = 0;
          soft_error_retry_count = 0; 
        } else if (ketchup) {
          ketchup = 0;
        } else {
          ++soft_error_retry_count;
          TRACE( 5, "fdc_isr", "this is a retry");
        }
        if ((cause & (id_am_error | id_crc_error | data_am_error |
                      data_crc_error | no_data_error)) != 0 &&
            bad_sector_map == ((unsigned) EMPTY_SEGMENT) >> 1) {
          /* This condition occurs when reading a `fake' sector that's
           * not accessible. Doesn't really matter as we would have
           * ignored it anyway ! Might be located at wrong segment,
           * then we'll fail on the next segment.
           */
          TRACE( 5, "fdc_isr", "skipping empty segment (read)");
#ifdef RETRY_HACK
          retry_overrule = 0;
#endif
          /* continue reading...
           */
        } else if (cause != no_error) {
          /* Handle all errors (missing AM, CRC, NO_DATA,...)
           */
          int dma_residue = get_dma_residue( FLOPPY_DMA);
          disable_dma( FLOPPY_DMA);
          TRACEx1( 6, "fdc_isr", "read DMA residue = 0x%04x", dma_residue);
          if (cause == no_data_error) {
            if (ftape_fast_start.active &&
                dma_residue == sectors_per_segment * SECTOR_SIZE) {
              ftape_fast_start.error = 1;
              TRACE( 5, "fdc_isr", "fast start failed (read)");
            }
            /*  Data not found: probably reading wrong segment.
             *  Can't rely on ecc code now, so abort and retry.
             */
            if (!retrying &&
                soft_error_retry_count >= RETRIES_ON_SOFT_ERROR) {
              TRACE( 5, "fdc_isr", "should retry and skip failing sector");
              TRACE( 5, "fdc_isr", "this isn't implemented yet !");
#ifdef RETRY_HACK
              TRACE( 5, "fdc_isr", "for now we'll let the ecc handle it");
              retry_overrule = 1; /* this is the hack: next pass we'll
                                   * let the ecc correct both the bad
                                   * sector and the next one */
#endif
            }
            must_retry = 1;
          } else if ((cause & (id_am_error | id_crc_error | data_am_error |
                               data_crc_error)) == 0) {
            /*  Don't know why this could happen but find out.
             */
            TRACE( 1, "fdc_isr", "unexpected error, see state above");
            must_retry = 1;
          } else {
            /*  Handle all normal errors.
             */
            if (cause & id_am_error) {
              history.id_am_errors++;
            } else if (cause & id_crc_error) {
              history.id_crc_errors++;
            } else if (cause & data_am_error) {
              history.data_am_errors++;
            } else if (cause & data_crc_error) {
              history.data_crc_errors++;
            } else if (cause & overrun_error) {
              history.overrun_errors++;
            }
            if (buffer[ head].crc_error_map == 0) {
              /*  Let ecc code correct this single error
               */
#ifdef RETRY_HACK
              if (cause & data_crc_error) {
                if (!retry_overrule) {
                  TRACE( 4, "fdc_isr", "1 crc error, let ecc handle it");
                } else {
                  TRACE( 4, "fdc_isr", "crc hack: let ecc handle it");
                }
              } else if (cause & overrun_error) {
                if (!retry_overrule) {
                  TRACE( 4, "fdc_isr", "1 overrun error, let ecc handle it");
                } else {
                  TRACE( 4, "fdc_isr", "overrun hack: let ecc handle it");
                }
              } else {
                if (!retry_overrule) {
                  TRACE( 4, "fdc_isr", "1 am/crc error, let ecc handle it");
                } else {
                  TRACE( 4, "fdc_isr", "am/crc hack: let ecc handle it");
                }
              }
#else
              if (cause & data_crc_error) {
                TRACE( 4, "fdc_isr", "1 crc error, let ecc handle it");
              } else if (cause & overrun_error) {
                TRACE( 4, "fdc_isr", "1 overrun error, let ecc handle it");
              } else {
                TRACE( 4, "fdc_isr", "1 am/crc error, let ecc handle it");
              }
#endif
              ketchup = 1;
            } else {
              /*  Retry to minimize risk of failing ecc code
               */
              if (soft_error_retry_count < RETRIES_ON_SOFT_ERROR) {
                TRACE( 4, "fdc_isr", "too many errors, retry");
                must_retry = 1;
              } else {
                /*  We're doomed, ecc is our last hope
                 */
                ketchup = 1;
                TRACE( 2, "fdc_isr", "too many retries, must rely on ecc");
              }
            }
            if (!must_retry) {
              int nr_not_read = ((dma_residue + (SECTOR_SIZE - 1)) /
                                 SECTOR_SIZE);
              TRACE( 5, "fdc_isr", "no retry, continue");
              /*  Try to catch-up on next sector. If we're too slow
               *  we'll get a no_data error on the next interrupt.
               */
              if (cause & data_crc_error) {
                sector_count-= nr_not_read;
                bad_sector_map <<= nr_not_read; /* re-adjust */
              } else {
                sector_count-= nr_not_read - 1;
                bad_sector_map <<= nr_not_read - 1; /* re-adjust */
              }
            }
            have_error = 1;
          }
        } else { /* cause == no_error */
#ifdef RETRY_HACK
          retry_overrule = 0;
#endif
        }
        retrying = must_retry;
        if (must_retry) {
          TRACE( 5, "fdc_isr", "read error, setup for retry");
          runner_status = aborting;
          fdc_mode = fdc_idle;
          buffer[ head].status = error;
          ++history.retries;
          break;
        }
        if (runner_status != running) {
          /*  We've got a problem: what's happening ?
           */
          TRACEi( 1, "fdc_isr", "fdc_reading_data in illegal runner state",
                 runner_status);
        }
        if (runner_status == running && buffer[ head].status == reading) {
          /*  Update var's influenced by previous DMA operation.
           */
          sector_offset += sector_count;
          buffer_ptr += sector_count * SECTOR_SIZE;
          remaining -= sector_count;
          if (have_error) {
            /* merge position of bad sector in bitmap
             */
            buffer[ head].crc_error_map |=  (1 << (sector_offset - 1));
#ifdef RETRY_HACK
            if (retry_overrule && remaining > 0) {
              /*  We're not able to catch up with the next sector after
               *  this crc error, so mark it bad and skip it too.
               */
              ++sector_offset;
              buffer_ptr += SECTOR_SIZE;
              --remaining;
              buffer[ head].crc_error_map |=  (1 << (sector_offset - 1));
            }
            retry_overrule = 0;
#endif
          }
          if (remaining > 0 && calc_next_cluster() > 0) {
            /*        Read remaining sectors in segment
             */
            if (setup_fdc_and_dma( FDC_READ) != 0) {
              /*      On error abort read operation but do updata buffer
               *      information.
               */
              buffer[ head].bytes = ((char*) buffer_ptr -
                                     (char*) buffer[ head].address);
              buffer[ head].status = error;
              next_buffer( &head);
              runner_status = aborting;
            } else {
              ketchup = 1;
              break;          /* keep fdc_mode set to fdc_reading_data */
            }
          } else {
            /*        Read next segment after updating buffer information.
             */
            buffer[ head].bytes = ((char*) buffer_ptr -
                                   (char*) buffer[ head].address);
            buffer[ head].status = full; /* segment done */
            next_buffer( &head);
            /*        Read ahead: read next segment if possible
             */
            if ((next_segment % segments_per_track) != 0) {
              if (buffer[ head].status == empty) {
                setup_new_segment( head, next_segment, 0,
                                  sectors_per_segment);
                if (calc_next_cluster() == 0 ||
                    setup_fdc_and_dma( FDC_READ) != 0) {
                  TRACE( 1, "fdc_isr", "couldn't start read-ahead");
                  runner_status = do_abort;
                } else {
                  buffer[ head].status = reading;
                  break;      /* keep fdc_mode set to fdc_reading_data */
                }
              } else {
                TRACE( 5, "fdc_isr", "all input buffers full");
                runner_status = buffer_overrun;
              }
            } else {
              /*      Just got last segment on track
               */
              runner_status = logical_eot;
            }
          }
        }
        fdc_mode = fdc_idle;
        break;
      }
        
      case fdc_reading_id: {
        
        if (cause == no_error) {
          fdc_cyl = in[ 3];
          fdc_head = in[ 4];
          fdc_sect =in[ 5];
        } else {              /* no valid information, use invalid sector */
          fdc_cyl =
          fdc_head =
          fdc_sect = 0;
        }
        TRACEx3( 6, "fdc_isr", "id read: c: 0x%02x, h: 0x%02x, s: 0x%02x",
                fdc_cyl, fdc_head, fdc_sect);
        fdc_mode = fdc_idle;
        break;
      }
        
      case fdc_writing_data: {
        
        TRACEi( 5, "fdc_isr", "fdc irq writing segment",
               buffer[ head].segment_id);
        if (runner_status == aborting) {
          TRACEx1( 5, "fdc_isr", "aborting %s", fdc_mode_txt);
          break;
        }
        
        if ((cause & (id_am_error | id_crc_error | data_am_error |
                      data_crc_error | no_data_error)) != 0 &&
            bad_sector_map == ((unsigned) EMPTY_SEGMENT) >> 1) {
          /* This condition occurs when trying to write to a `fake'
           * sector that's not accessible. Doesn't really matter as
           * it isn't used anyway ! Might be located at wrong segment,
           * then we'll fail on the next segment.
           */
          TRACE( 5, "fdc_isr", "skipping empty segment (write)");
          /* continue reading...
           */
        } else if (cause != no_error) {
          /* Handle all errors (missing AM, CRC, NO_DATA,...)
           */
          int dma_residue = get_dma_residue( FLOPPY_DMA);
          disable_dma( FLOPPY_DMA);
          TRACEx1( 6, "fdc_isr", "write DMA residue = 0x%04x", dma_residue);
          if (cause & (id_am_error | id_crc_error | data_am_error |
                       no_data_error | overrun_error)) {
            int nr_not_written = ((dma_residue + (SECTOR_SIZE - 1)) /
                                  SECTOR_SIZE);
            /*  Adjust for incomplete dma operation, retry failing sector
             */
            sector_count-= nr_not_written;
            bad_sector_map <<= nr_not_written; /* re-adjust */
            buffer[ head].status = error;
            if (cause & id_am_error) {
              history.id_am_errors++;
            } else if (cause & id_crc_error) {
              history.id_crc_errors++;
            } else if (cause & data_am_error) {
              history.data_am_errors++;
            } else if (cause & overrun_error) {
              history.overrun_errors++;
            }
            if (cause & no_data_error && ftape_fast_start.active &&
                dma_residue == sectors_per_segment * SECTOR_SIZE) {
              ftape_fast_start.error = 1;
              TRACE( 5, "fdc_isr", "fast start failed (write)");
            }
          } else {
            if (in[ 1] & 0x02) {
              TRACE( 1, "fdc_isr", "media not writable");
            } else {
              TRACE( -1, "fdc_isr", "unforseen write error");
            }
          }
          runner_status = aborting;
          fdc_mode = fdc_idle;
          break;
        }
        if (runner_status != running) {
          /*  We've got a problem: what's happening ?
           */
          TRACEi( 1, "fdc_isr", "fdc_writing_data in illegal runner state",
                 runner_status);
        }
        if (runner_status == running && buffer[ head].status == writing) {
          /*  Update var's influenced by previous DMA operation.
           */
          sector_offset += sector_count;
          buffer_ptr += sector_count * SECTOR_SIZE;
          remaining -= sector_count;
          
          if (remaining > 0 && calc_next_cluster() > 0) {
            /*        Write remaining sectors in segment
             */
            if (setup_fdc_and_dma( FDC_WRITE) != 0) {
              /*  On error abort write operation but do updata
               *  buffer information.
               */
              buffer[ head].status = error;
              next_buffer( &head);
              runner_status = aborting;
            } else {
              break;          /* keep fdc_mode set to fdc_writing_data */
            }
          } else {
            /*        Write next segment after updating buffer information.
             */
            buffer[ head].bytes = ((char*) buffer_ptr -
                                   (char*) buffer[ head].address);
            buffer[ head].status = empty; /* segment done */
            next_buffer( &head);
            
            if ((next_segment % segments_per_track) != 0) {
              if (buffer[ head].status == full) {
                setup_new_segment( head, next_segment, 0,
                                  sectors_per_segment);
                if (calc_next_cluster() == 0 ||
                    setup_fdc_and_dma( FDC_WRITE) != 0) {
                  TRACE( 1, "fdc_isr", "couldn't start next write");
                  runner_status = do_abort;
                } else {
                  buffer[ head].status = writing;
                  break;      /* keep fdc_mode set to fdc_writing_data */
                }
              } else {
                TRACE( 5, "fdc_isr", "no full output buffer found");
                runner_status = buffer_underrun;
              }
            } else {
              /*      Just got last segment on track
               */
              runner_status = logical_eot;
            }
          }
        } else {
          if (runner_status != running) {
            TRACE( 1, "fdc_isr", "runner-status != running");
          }
          if (buffer[ head].status != writing) {
            TRACE( 1, "fdc_isr", "buffer-status != writing");
          }
        }
        fdc_mode = fdc_idle;
        break;
      }
        
      default:
        
        TRACEx1( 1, "fdc_isr", "Warning: unexpected irq during: %s",
                fdc_mode_txt);
        fdc_mode = fdc_idle;
        break;
      }
    }
    ftape_fast_start.active = 0;
    if (runner_status == do_abort) {
      /*      cease operation, remember tape position
       */
      TRACE( 5, "fdc_isr", "runner aborting");
      runner_status = aborting;
      expect_stray_interrupt = 1;
    }
  } else { /* !FDC_BUSY  */
    /*  clear interrupt, cause should be gotten by issueing
     *  a Sense Interrupt Status command.
     */
    if (fdc_mode == fdc_recalibrating || fdc_mode == fdc_seeking) {
      seek_completed = 1;
      fdc_mode = fdc_idle;
    } else if (!wait_intr) {
      if (!expect_stray_interrupt) {
        TRACE( 2, "fdc_isr", "unexpected stray interrupt");
      } else {
        TRACE( 5, "fdc_isr", "expected stray interrupt");
      }
    }
  }
  
  /*    Handle sleep code.
   */
  ++interrupt_seen;
  if (wait_intr) {
    wake_up_interruptible( &wait_intr);
  }
}

/*      Wait for FDC interrupt with timeout.
 */
int
fdc_interrupt_wait( int time) {
  
  unsigned long flags;
  struct wait_queue wait = { current, NULL };
  int result = -ETIME;
  
  if (wait_intr) {
    TRACE( 1, "fdc_interrupt_wait", "error: nested call");
    return -EIO;                /* return error... */
  }
  save_flags( flags);
  cli();
  if (interrupt_seen == 0) {
    int signal_pending = 0;
   /*   timeout time will be between 0 and MSPT milliseconds too long ! */
    current->timeout = jiffies + 1 + (time + MSPT - 1) / MSPT;
    current->state = TASK_INTERRUPTIBLE;
  
    add_wait_queue( &wait_intr, &wait);
  
    while (1) {
      sti();                    /* allow interrupt_seen */
      while (!interrupt_seen && current->state != TASK_RUNNING) {
        schedule();             /* sets TASK_RUNNING on timeout */
      }
      cli();                    /* prevents late interrupt_seen */
      if (current->signal & ~current->blocked) {
        /* probably awakened by non-blocked signal, fdc probably still busy.
         * will create a mess if we don't reset it !
         */
#if 0
        disable_dma( FLOPPY_DMA);
        fdc_reset();
#endif
        /* hack (?) to allow gracefull exit
         * just mask every signal that interferes
         */
        current->blocked |= (current->signal & ~current->blocked) ;
        signal_pending = 1;
      } else {
        if (signal_pending) {
          TRACE( 1, "fdc_interrupt_wait", "error: aborted on signal");
          current->timeout = 0;
          remove_wait_queue( &wait_intr, &wait);
          restore_flags( flags);
          return -EINTR;
        } else {
          break;
        }
      }
    }
    /* get here only on timeout or qic117 interrupt, TASK_RUNNING */
    if (interrupt_seen) {
      current->timeout = 0;     /* interrupt hasn't cleared this */
      result = 0;
    }
    remove_wait_queue( &wait_intr, &wait);
  } else {
    result = 0;
  }
  restore_flags( flags);
  interrupt_seen = 0;		/* should be somewhere else ! */
  return result;
}

/*      Start/stop drive motor. Enable DMA mode.
 */
void
fdc_motor( int motor)
{
  int data;

  ftape_motor = motor;
  data = (ftape_unit & 1) | FDC_RESET_NOT | FDC_DMA_MODE;
  if (ftape_motor) {
    data |= (ftape_unit & 1) ? FDC_MOTOR_1 : FDC_MOTOR_0;
    TRACE( 4, "fdc_motor", "turning motor on");
  } else {
    TRACE( 4, "fdc_motor", "turning motor off");
  }
  outb_p( data, FDC_CONTROL);
  ftape_sleep( 10 * MILLISECOND);
}

/*      Reset the floppy disk controller. Leave the FDTAPE unit selected.
 */
void
fdc_reset(void)
{
  int control_word;

  expect_stray_interrupt = 1;
  control_word = (ftape_unit & 1) | FDC_DMA_MODE;
  if (ftape_motor) {
    control_word |= (ftape_unit & 1) ? FDC_MOTOR_1 : FDC_MOTOR_0;
  }
  /* Assert the reset line.  Leave the proper unit selected. */
  outb_p( control_word, FDC_CONTROL);
  ftape_sleep( MILLISECOND);    /* > 14 fdc clocks */

  /* Now release the reset line. */
  control_word |= FDC_RESET_NOT;
  outb_p( control_word, FDC_CONTROL);
  ftape_sleep( 10 * MILLISECOND); /* ??? */

  /* Select clock for fdc.  May need to be changed to select data rate. */
  outb_p( 0, FDC_VFO);          /* set high density (500 Kbps) mode */

  expect_stray_interrupt = 0;
}

/* When we're done, put the fdc into reset mode so that the regular
   floppy disk driver will figure out that something is wrong and
   initialize the controller the way it wants. */
void
fdc_disable( void)
{
  outb_p( (ftape_unit & 1), FDC_CONTROL);
}

/*      Specify FDC seek-rate
 */
int
fdc_set_seek_rate( int seek_rate)
{
  unsigned char in[ 3];

  in[ 0] = FDC_SPECIFY;
  in[ 1] = ((16 - seek_rate) << 4) | 13 /* head_unload_time */;
  in[ 2] = (1 /* head_load_time */ << 1) | 0 /* non_dma */;

  return fdc_command( in, 3);
}

/*      Sense drive status: get unit's drive status (ST3)
 */
int
fdc_sense_drive_status( int *st3)
{
  int result;
  unsigned char out[ 2];
  unsigned char in[ 1];

  out[ 0] = FDC_SENSED;
  out[ 1] = FTAPE_UNIT;
  result = fdc_issue_command( out, 2, in, 1);
  if (result < 0) {
    TRACE( 1, "fdc_sense_drive_status", "issue_command failed");
    return result;
  } else {
    *st3 = in[ 0];
    return 0;
  }
}

/*      Sense Interrupt Status command:
 *      should be issued at the end of each seek.
 *      get ST0 and current cylinder.
 */
int
fdc_sense_interrupt_status( int *st0, int *current_cylinder)
{
  int result;
  unsigned char out[ 1];
  unsigned char in[ 2];

  out[ 0] = FDC_SENSEI;
  result = fdc_issue_command( out, 1, in, 2);
  if (result) {
    TRACE( 1, "fdc_sense_interrupt_status", "issue_command failed");
    return result;
  } else {
    *st0 = in[ 0];
    *current_cylinder = in[ 1];
    return 0;
    }
}

/*      step to track
 */
int
fdc_seek( int track)
{
  int result;
  unsigned char out[3];
  int st0, pcn;

  ftape_sleep( MILLISECOND);    /* NEEDED to prevent problems */

  out[ 0] = FDC_SEEK;
  out[ 1] = FTAPE_UNIT;
  out[ 2] = track;
  seek_completed = 0;
  result = fdc_command( out, 3);
#ifdef CHECK_TIMING
  do_gettimeofday( &t1);
  new_command = abs( track - current_cylinder);
#endif
  if (result != 0) {
    TRACEi( 1, "fdc_seek", "failed, status =", result);
    TRACEx1( 4, "fdc_seek", "destination was: %d, resetting FDC...", track);
    /*  We really need this command to work !
     */
    fdc_reset();
    fdc_set_seek_rate( 2);
    return result;
  }

  /*    Handle interrupts until seek_completed or timeout.
   */
  for (;;) {
    result = fdc_interrupt_wait( 2 * SECOND);
    if (result < 0) {
      TRACEi( 2, "fdc_seek", "fdc_interrupt_wait timeout, status =", result);
      return result;
    } else if (seek_completed) {
      result = fdc_sense_interrupt_status( &st0, &pcn);
      if (result != 0) {
        TRACEi( 1, "fdc_seek",
               "fdc_sense_interrupt_status failed, status =", result);
        return result;
      }
      if ((st0 & ST0_SEEK_END) == 0) {
        TRACE( 1, "fdc_seek", "no seek-end after seek completion !??");
        return -EIO;
      }
      break;
    }
  }
  /*    Verify whether we issued the right tape command.
   */
  /* Verify that we seek to the proper track. */
  if (pcn != track) {
    TRACE( 1, "fdc_seek", "bad seek..");
    return -EIO;
  }
  current_cylinder = pcn;
  return 0;
}

/*      Recalibrate and wait until home.
 */
int
fdc_recalibrate( void)
{
  int result;
  unsigned char out[ 2];
  int st0;
  int pcn;
  int retry;

  TRACE( 5, "fdc_recalibrate", "called");
  result = fdc_set_seek_rate( 6);
  if (result) {
    TRACEi( 1, "fdc_recalibrate",
           "fdc_set_seek_rate failed, status =", result);
    return result;
  }
  out[ 0] = FDC_RECAL;
  out[ 1] = FTAPE_UNIT;
  seek_completed = 0;
  result = fdc_command( out, 2);
  if (result) {
    TRACEi( 1, "fdc_recalibrate", "fdc_command failed, status =", result);
    return result;
  }
  /*    Handle interrupts until seek_completed or timeout.
   */
  for (retry = 0;; ++retry) {
    result = fdc_interrupt_wait( 2 * SECOND);
    if (result < 0) {
      TRACE( 1, "fdc_recalibrate", "fdc_interrupt_wait failed");
      return result;
    } else if (result == 0 && seek_completed) {
      result = fdc_sense_interrupt_status( &st0, &pcn);
      if (result != 0) {
        TRACEi( 1, "fdc_recalibrate",
               "fdc_sense_interrupt_status failed, status =", result);
        return result;
      }
      if ((st0 & ST0_SEEK_END) == 0) {
        if (retry < 1) {
          continue;             /* some drives/fdc's give an extra interrupt */
        } else {
          TRACE( 1, "fdc_recalibrate",
                "no seek-end after seek completion !??");
          return -EIO;
        }
      }
      break;
    }
  }
  current_cylinder = pcn;
  if (pcn != 0) {
    TRACEi( 1, "fdc_recalibrate", "failed: resulting track =", pcn);
  }
  result = fdc_set_seek_rate( 2);
  if (result != 0) {
    TRACEi( 1, "fdc_recalibrate",
           "fdc_set_seek_rate failed, status =", result);
    return result;
  }
  return 0;
}

