/*
 * statdir.c
 *
 * Copyright (C), 1994, Graeme W. Wilford. (Wilf.)
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * routines used in statting dirs 
 *
 * Mon May  2 17:36:33 BST 1994  Wilf. (G.Wilford@ee.surrey.ac.uk)
 */

#define MAN_MAIN        /* to not define config_file */
#define MANPATH_MAIN    /* to not define *std_sections[] */

#include <string.h>
#include <stdio.h>
#include <dirent.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/stat.h>

#include "config.h"
#include "mydbm.h"
#include "order.h"
#include "util.h"

extern char *prognam;
extern char *manp;
extern char *database;
extern int debug;
extern MYDBM_FILE dbf;
extern short quiet;
extern short dups;

int is_duplicate(char *oldpage, char *manpage)
{
	char *old, *man;

	old = strrchr(oldpage, '/');
	man = strrchr(manpage, '/');

	/* need to do some checks on their time stamps and keep the newest */

	if ( dups && strcmp(++old, ++man) == 0) {
		if (is_newer(oldpage, manpage) & 1) {
			if (!quiet)	
				printf("WARNING - duplicate: %s & (%s ignored)\n",
				  oldpage, manpage);
			return 1;
		} else {
			if (!quiet)
				printf("WARNING - duplicate: (%s ignored) & %s\n",
				  oldpage, manpage);
			return -1;
		}
	}
	
	if ( strncmp(old, man, (size_t) (strrchr(old, '.') - old + 2) ) == 0) 
		if (!quiet)
			printf("WARNING - possible duplicate:\n\t %s & (%s NOT ignored)\n",
			  oldpage, manpage);
	return 0;
}

int dup_test(char *oldpage, char *manpage)
{
	char *t1, *newlist;
	int ret;

	*(newlist = (char *) malloc (strlen(oldpage) + 2)) = '\0'; 
	
	while( (t1 = strrchr(oldpage, ':')) != NULL){
		*t1 = '\0';

		ret = is_duplicate(++t1, manpage);

		if (ret == 1) {				/* 1 */
			free(newlist);
			return 1;
		} else if (ret == 0) {			/* 0 */
			strcat(newlist, t1);
			strcat(newlist, ":");
		} 					/* -1 */
	}

	ret = is_duplicate(oldpage, manpage);

	if (ret == 1) {
		free(newlist);
		return 1;
	} else if (ret == 0) {
		strcat(newlist, oldpage);
		strcat(newlist, ":");
	}

	if ( (t1 = strrchr(newlist, ':')) != NULL)
		*t1 = '\0';

	strcpy(oldpage, newlist);
	free(newlist);
	return 0;
}

void add_dir_entrys(char *path, char *infile)
{
	char manpage[MAXPATHLEN];
	char keyc[80];
	char *oldpage, *t1, *t2;
	datum key, content;
	size_t len;
	struct dirent *newdir;
	DIR *dir;

	strcpy(manpage, path);
	strcat(manpage, "/");
	strcat(manpage, infile);
	strcat(manpage, "/");
	len = strlen(manpage);

	/*
	 * All filename entries in this dir should either be valid manpages
	 * or . files (such as current, parent dir).
	 */

	if ( (dir = opendir(infile)) == NULL){
                fprintf(stderr, "%s:add_dir_entrys: "
                  "could not open dir for reading: ", prognam);
                perror(manpage);
                exit(1);
        }
        
	while ( (newdir = readdir(dir)) != NULL){
		if ( *newdir->d_name == '.')
			continue;
		else {
			strcpy(manpage + len, newdir->d_name);
			content.dptr = manpage;
	
			t1 = strrchr(manpage, '/');
			key.dptr = strcpy(keyc, t1 + 1); /* !don't optimise! */
	
			*(t2 = strrchr(keyc, '.')) = '\0'; 

			/* 
			 * A bogus manpage is defined as having it's section
			 * number (alpha/num) != to the section it is
			 * stored under. (only the first char of the manpage
			 * section is compared with the last char of the 
			 * section it is under)
			 * ie. .../man5/shells.5.orig, .../man1/xast.man.
			 *
			 * .../manavs/intro.avs WOULD be wrong. - 
			 * better to store in a separate man tree.
			 * 
			 * things like .../man1/ld.so.1 are fine and MUST 
			 * be accepted.
			 */
	
			if (*(--t1) != *(++t2)){
				if (!quiet)
					printf("WARNING - bogus manpage:"
				  	  " %s ignored\n", manpage);
				continue;
			}

			content.dsize = strlen(content.dptr) +1;
			key.dsize = strlen(key.dptr) +1;

	
			if (MYDBM_INSERT(dbf, key, content) != 0){
			
				/*
				 * this is where it gets a little bit tricky
				 * we already have a manpage registered 
				 * for this key.
				 *
				 * we need to check for duplicates, and 
				 * also maintain
				 * the appropriate order set out in 
				 *   char *std_sections[].
				 */
	
				oldpage = MYDBM_FETCH(dbf, key).dptr;
	
				/*
				 * check to see if it's a duplicate man page.  
				 * it's more subtle than I thought, need 
				 * to compare 
				 * with all of the other key pages, not 
				 * just the last
				 *
				 * dup_test() returns NULL if a duplicate was 
				 * found and the newer file was already in the
				 * db, else the replacement string is returned.
				 *
				 * This could either be an empty string, or 
				 * one or more pages in a list.
				 */
	
				if (dup_test(oldpage, manpage)) {
					MYDBM_FREE(oldpage);
					continue;	/* skip this page */
				}

				if (*oldpage != '\0')
					content.dptr = orderthem(manpage, oldpage);
				else
					content.dptr = manpage;

				/* contents of oldpage are now dead */
				MYDBM_FREE(oldpage); 

				content.dsize = strlen(content.dptr) + 1; 
				
				if (debug) 
					fprintf(stderr, "REPLACE: %s -> %s\n",
					  key.dptr, content.dptr);
	
				MYDBM_REPLACE(dbf, key, content);
			}
			else {
				if (debug) 
					fprintf(stderr, "INSERT: %s -> %s\n",
					  key.dptr, content.dptr);
			}
		}
	}
}

short testmandirs(char *path, time_t last)
{
	DIR *dir;
	struct dirent *mandir;
	struct stat stbuf;
	short amount = 0;

	if (debug)
		fprintf(stderr, "Testing %s for new files\n", path);

	if ( (dir = opendir(path)) == NULL){
		fprintf(stderr, "%s: testmandirs: could not open dir for reading: ", prognam);
		perror(path);
		exit(1);
	}

	chdir(path);

	while( (mandir = readdir(dir)) != NULL){
		if (*mandir->d_name != 'm' 
		  && *mandir->d_name != 'M')
			continue;
			
		if ( stat(mandir->d_name, &stbuf) == 0 
		  && S_ISDIR(stbuf.st_mode) && stbuf.st_mtime > last){

		  	/* 
		  	 * file exists, is a DIR, and has been 
		  	 * 'modified' since last full db update
		  	 * - could be a new manpage for us.
		  	 */

			if (debug)
				fprintf(stderr,
				  "\tsubdirectory %s has been 'modified'\n",
				  mandir->d_name);
		  	add_dir_entrys(path, mandir->d_name);
		  	amount++;
		}
	}

	closedir(dir);
	return amount;
}

short check_mandirs(short globalman)
{
	datum key, content;
	char *c1;
	time_t now;
	short amount = 0;
	char *path;
	
	key.dptr = KEY;
	key.dsize = sizeof KEY;

	content = MYDBM_FETCH(dbf, key);

	if (content.dptr == NULL)
		now = 0;
	else
		now = atol(content.dptr);

	MYDBM_FREE(content.dptr);

	if (globalman){
		path = strdup(manp);
		while( (c1 = strrchr(path, ':')) != NULL){
			*c1 = '\0';
			if (strncmp(++c1, MAN_ROOT, sizeof MAN_ROOT -1) == 0)
				amount += testmandirs(c1, now);
		}
		if (strncmp(path, MAN_ROOT, sizeof MAN_ROOT -1) == 0)
			amount += testmandirs(path, now);

	} else {
		/* strip the actual database name */
		path = strdup(database);
		*(strrchr(path, '/')) = '\0';
		amount += testmandirs(path, now);
	}

	free(path);

	if (amount){
		now = time(NULL);
		content.dptr = (char *) malloc(11); /* max long with '\0' */
		sprintf(content.dptr, "%ld", now);
		content.dsize = strlen(content.dptr) + 1;
		MYDBM_REPLACE(dbf, key, content);
		free(content.dptr);
	}
	return amount;
}
