/* 
   Unix SMB/Netbios implementation.
   Version 1.7.
   Copyright (C) Karl Auer 1993,1994

   Re-working by Martin Kiff, 1994
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *  Parse printcap file.
 *
 *  This module does exactly one thing - it looks into the printcap file
 *  and tells callers if a specified string appears as a printer name.
 *
 *  The way this module looks at the printcap file is very simplistic.
 *  Only the local printcap file is inspected (no searching of NIS
 *  databases etc).
 *
 *  There are assumed to be one or more printer names per record, held
 *  as a set of sub-fields separated by vertical bar symbols ('|') in the
 *  first field of the record. The field separator is assumed to be a colon
 *  ':' and the record separator a newline.
 * 
 *  Lines ending with a backspace '\' are assumed to flag that the following
 *  line is a continuation line so that a set of lines can be read as one
 *  printcap entry.
 *
 *  A line stating with a hash '#' is assumed to be a comment and is ignored
 *  Comments are discarded before the record is strung together from the
 *  set of continuation lines.
 *
 *
 *  Method:
 *  1 The getc_pcap function reads bytes from a printcap format file, handling
 *    backslash flagged continuation lines and ignoring comment lines
 *    starting with a "#".
 *    It therefore returns one 'line' per entry, an EOL marking the end
 *    of the entry.  Returns an EOF at the end of file.
 *
 *  2 The collect_string function returns the next 'printer name' from the
 *    file by calling getc_pcap. This is either the first field in the
 *    record or, if a set of aliases have been set up separated by '|', then
 *    each alias is returned in turn.
 *    Any leading and trailing whitespace is trimming off the printer name.
 *
 *  3 Compare the resulting string to the target string. If it matches,
 *    stop searching and return True. Otherwise read the next string and
 *    test again.
 *
 *  4 Return False.
 *    
 *
 *  Opening a pipe for "lpc status" and reading that would probably 
 *  be pretty effective. Code to do this already exists in the freely
 *  distributable PCNFS server code.
 */

#include "includes.h"

#include "smb.h"
#include "loadparm.h"
#include "pcap.h"

extern int DEBUGLEVEL;

/* maximum string to parse from printcap */
#define MAX_COLLECT 256

/* these things terminate a field in printcap (for our purposes, anyway)*/
#define IS_TERM(c) ((c) == EOF || (c) == ':' || (c) == '|')

/* local prototypes */
static int getc_pcap(FILE *pfile);
static int getc_pcapx(FILE *pfile);
static char *collect_string(FILE *pfile);

/* Static state values for handling backslash prompted continuation lines
   in the printcap file. */

#define PCAP_EOF (-1)
#define PCAP_UNKNOWN (0)
#define PCAP_NEWLINE (1)
#define PCAP_READ (2)
#define PCAP_SKIP_TO_EOL (3)
#define PCAP_QUEUE (4)
#define PCAP_DEQUEUE (5)

static int get_state = PCAP_UNKNOWN;
static int final_state = PCAP_UNKNOWN;

#define PCAP_LOOKAHEAD (2)
static char pcap_buff[PCAP_LOOKAHEAD+1];
static char *readp;
static char *sendp;

/***************************************************************************
Read a character, if it is a backslash use this as a hint that a continuation
line may follow and see if we can skip the trailing whitespace and newline.
***************************************************************************/
static int getc_pcap(FILE *pfile)
{
   int c;

   while (1)
      switch (get_state)
         {
         case PCAP_NEWLINE:
            c = getc_pcapx (pfile);
            if (c == EOF)
               return (EOF);
            else if (c == '#')
               get_state = PCAP_SKIP_TO_EOL;
            else if (c == '\\')
               {
               final_state = PCAP_READ;
               get_state = PCAP_QUEUE;
               *(readp++) = c;
               }
            else if (!isspace(c))
               {
               get_state = PCAP_READ;
               return (c);
               }
            else
               return (c);
            break;

         case PCAP_READ:
            c = getc_pcapx (pfile);
            if (c == EOF)
               return (EOF);
            else if (c == '\\')
               {
               final_state = PCAP_READ;
               get_state = PCAP_QUEUE;
               *(readp++) = c;
               }
            else
               return (c);
            break;

         case PCAP_SKIP_TO_EOL:
            c = getc_pcapx (pfile);
            if (c == EOF)
               return (EOF);
            else if (c == '\n')
               get_state = PCAP_NEWLINE;
            break;

         case PCAP_QUEUE:
            c = getc_pcapx (pfile);
            if (c == EOF)
               {
               final_state = PCAP_EOF;
               get_state = PCAP_DEQUEUE;
               }
            else if (c == '\n')
               {
               readp = sendp = pcap_buff;
               get_state = final_state = PCAP_NEWLINE;
               }
            else if (isspace (c))
               {
               if (readp >= &pcap_buff[PCAP_LOOKAHEAD])
                  {
                  DEBUG(0,( "Lookahead for continuation line overflowed\n"));
                  get_state = PCAP_EOF;
                  return (EOF);
                  }
               *(readp++) = c;
               }
            else
               {
               *(readp++) = c;
               get_state = PCAP_DEQUEUE;
               }
            break;

         case PCAP_DEQUEUE:
            if (sendp < readp)
               return (*(sendp++));
            sendp = readp = pcap_buff;
            get_state = final_state;
            break;

         case PCAP_UNKNOWN:
            readp = sendp = pcap_buff; 
            get_state = final_state = PCAP_NEWLINE;
            break;
   
         case PCAP_EOF:
            return (EOF);
         }
   }

/***************************************************************************
Read a character, setting the get_state to PCAP_EOF on end-of-file or error.
***************************************************************************/
static int getc_pcapx(FILE *pfile)
{
   int c;

   c = getc (pfile);
   if (c == EOF)
      {
      if (ferror (pfile))
         DEBUG(0,( "Read error on printcap file\n"));
      get_state = PCAP_EOF;
      }
   return (c);
}


/***************************************************************************
Collect a string from the passed file.
There is an arbitrary limit on the length of a collected string. Collection
continues until EOF, a colon, a vertical bar or a non-continued newline is
encountered, whereupon the terminating character is discarded.

Return a pointer to the collected string. If this function returns NULL, 
then an error occurred and has been logged. If it returns an empty string,
then a normal EOF has been encountered. Empty fields are logged for 
information.

Warning:

The returned string is stored statically - each call to this function 
destroys the previously collected string.

***************************************************************************/
static char *collect_string(FILE *pfile)
{
   static char szBuf[MAX_COLLECT + 1];
   static char *szBufp, *szBufendp;

   int c;

   /* Skip leading whitespace. */

   szBufp = szBufendp = szBuf;
   do {
      c = getc_pcap(pfile);
      }
   while (!IS_TERM (c) && isspace (c));

   /* If there is something interesting, copy it to the collection buffer.
      Keep a pointer to the last copied non-space characetr as we aill be
      writing a null over it to trim any trailing spaces. */

   if (!isspace (c) && !IS_TERM (c))
      {
      *(szBufp++) = c;
      szBufendp = szBufp;
      do {
         c = getc_pcap(pfile);
         if (IS_TERM (c))
            break;
         if (szBufp >= &szBuf[MAX_COLLECT])
            {
            DEBUG(0,( "Printcap field overflowing working buffer\n"));
            return (NULL);
            }
         *(szBufp++) = c;
         if (!isspace (c))
            szBufendp = szBufp;
         }
      while (1);
      }
   *szBufendp = '\0';

   /* If we've read a colon, then we've finished the first field in the
      record and can skip everything to the next. */

   if (c == ':')
      get_state = PCAP_SKIP_TO_EOL;

   /* If no errors have occurred, return the string after terminating it. */
   /* Otherwise, return NULL. */

   if (c != EOF)
   {
      if (*szBuf == '\0')
         DEBUG(5,( "Empty printer name field in printcap (warning only).\n"));
#if 0
      else
         DEBUG(3,( "collect_string() returning: \"%s\"\n", szBuf));
#endif
      return (szBuf);
   }

   return (NULL);
}

/***************************************************************************
Scan printcap file pszPrintcapname for a printer called pszPrintername. 
Return True if found, else False. Returns False on error, too, after logging 
the error at level 0. For generality, the printcap name may be passed - if
passed as NULL, the configuration will be queried for the name.
***************************************************************************/
BOOL pcap_printername_ok(char *pszPrintername, char *pszPrintcapname)
{
   BOOL bRetval;
   FILE *pfile;
   char *psz;

   bRetval = False;
   if (pszPrintername == NULL || pszPrintername[0] == '\0')
   {
      DEBUG(0,( "Attempt to locate null printername! Internal error?\n"));
      return (bRetval);
   }

   /* only go looking if no printcap name supplied */
   if ((psz = pszPrintcapname) == NULL || psz[0] == '\0')
      if (((psz = lp_printcapname()) == NULL) || (psz[0] == '\0'))
      {
         DEBUG(0,( "No printcap file name configured!\n"));
         return (bRetval);
      }

   get_state = PCAP_UNKNOWN;

   if ((pfile = fopen(psz, "rt")) == NULL)
      DEBUG(0,( "Unable to open printcap file %s for read!\n", psz));
   else   
   {
      while (get_state != PCAP_EOF)
      {
         /* collect a string for comparison */
         char *psz_collect = collect_string(pfile);
         if (psz_collect == NULL)
	 {
            DEBUG(2,( "Error parsing printcap file %s!\n", psz));
            break;
	 }

         /* stop on first match */
         if (strcmp(pszPrintername, psz_collect) == 0)
         {
            bRetval = True;
            break;
	 }
      }  /* end while */

      fclose(pfile);
   }

   return (bRetval);
}



