/* kasitem.cpp
**
** Copyright (C) 2001-2004 Richard Moore <rich@kde.org>
** Contributor: Mosfet
**     All rights reserved.
**
** KasBar is dual-licensed: you can choose the GPL or the BSD license.
** Short forms of both licenses are included below.
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
** MA 02111-1307, USA.
*/

/*
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
*/

/*
** Bug reports and questions can be sent to kde-devel@kde.org
*/
#include <qcursor.h>
#include <qpainter.h>
#include <qdrawutil.h>
#include <qregexp.h>
#include <qtimer.h>

#include <kdebug.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>

#include "kasitem.h"

#include "kaspopup.h"
#include "kasitem.moc"

/* XPM */
static const char *tiny_arrow[]={
"5 9 2 1",
". c None",
"# c #ffffff",
"....#",
"...##",
"..###",
".####",
"#####",
".####",
"..###",
"...##",
"....#"};

KasItem::KasItem( KasBar *parent )
   : QObject( parent ),
     kas( parent ), pop( 0 ), popupTimer( 0 ), dragTimer( 0 ),
     title( i18n( "Kasbar" ) ),
     mouseOver( false ), activated( false ), customPopup( false ), groupItem( false ),
     prog( -1 )
{
    update();
}

KasItem::~KasItem()
{
    delete pop;
}

void KasItem::setActive( bool yesno )
{
    if ( activated == yesno )
	return;

    activated = yesno;
    update();
}

void KasItem::setText( const QString &text )
{
    if ( title == text )
	return;

    title = text;
    update();
}

void KasItem::setProgress( int percent )
{
    if ( percent == prog )
	return;

    prog = percent;
    update();
}

void KasItem::mouseEnter()
{
   static const int POPUP_DELAY = 300;

   if ( (!customPopup) && (popupTimer == 0) ) {
      popupTimer = new QTimer( this, "popupTimer" );
      connect( popupTimer, SIGNAL( timeout() ), SLOT( showPopup() ) );
      popupTimer->start( POPUP_DELAY, true );
   }

   mouseOver = true;
   update();
}

// Check periodically if the popup can be hidden (hack)
bool KasItem::checkPopup()
{
   QPoint cursorPos = QCursor::pos();

   if ( !QApplication::widgetAt( cursorPos ) ) {
     if ( popupTimer ) {
        delete popupTimer;
        popupTimer = 0;
     }
     if ( (!customPopup) && pop ) {
        hidePopup();
	return false;
     }
   }
   else {
      static const int CHECK_POPUP_DELAY = 200;
      QTimer::singleShot( CHECK_POPUP_DELAY, this, SLOT( checkPopup() ) );
   }

   return true;
}

void KasItem::dragEnter()
{
   static const int DRAG_SWITCH_DELAY = 1000;

   if ( dragTimer == 0 ) {
      dragTimer = new QTimer( this, "dragTimer" );
      connect( dragTimer, SIGNAL( timeout() ), SLOT( dragOverAction() ) );
      dragTimer->start( DRAG_SWITCH_DELAY, true );
   }

   mouseOver = true;
   update();
}

void KasItem::mouseLeave()
{
   if ( popupTimer ) {
      delete popupTimer;
      popupTimer = 0;
   }
   if ( (!customPopup) && pop )
      hidePopup();

   mouseOver = false;
   update();
}

void KasItem::dragLeave()
{
   if ( dragTimer ) {
      delete dragTimer;
      dragTimer = 0;
   }

   mouseOver = false;
   update();
}

void KasItem::showPopup()
{
   if ( !checkPopup() )
       return;
   if ( pop )
      return;

   activated = true;
   pop = createPopup();
   if ( pop )
      pop->show();
}

void KasItem::hidePopup()
{
    delete pop;
    pop = 0;
    activated = false;
}

void KasItem::togglePopup()
{
   if ( pop )
      hidePopup();
   else
      showPopup();
}

void KasItem::paintFrame( QPainter *p )
{
   qDrawShadePanel(p, 0, 0, extent(), extent(), colorGroup(), false, 2);

   if ( mouseOver )
      p->setPen(Qt::white);
   else
      p->setPen(Qt::black);

   p->drawRect(0, 0, extent(), extent());
}

void KasItem::paintLabel( QPainter *p )
{
    QString text = title;

    if ( !groupItem ) {
	p->fillRect( 2, 2, extent()-4, 13, QBrush( kasbar()->labelBgColor() ) );

	if ( isProgressItem() ) {
	    QRegExp reg( "(1?[0-9]?[0-9])%" );
	    if ( -1 != reg.search( text ) ) {
		prog = reg.cap(1).toInt();
		paintProgress( p, prog );
	    }
	    else {
		prog = 0;
	    }
	}

	p->setFont( KGlobalSettings::taskbarFont() );
	p->setPen( kasbar()->labelPenColor() );

	if ( fontMetrics().width( text ) > extent()-4 )
	    p->drawText( 2, 2, extent()-4, 12, AlignLeft | AlignVCenter, text );
	else
	    p->drawText( 2, 2, extent()-4, 12, AlignCenter, text );

	return;
    }
    else {
	QPixmap arrow( tiny_arrow );

	QPoint popupPos = KasPopup::calcPosition( this, 10, 10 );
	QPoint iPos = kasbar()->mapToGlobal( kasbar()->itemPos( this ) );
	QWMatrix turn;

	if ( popupPos.x() < iPos.x() ) {
	    paintArrowLabel( p, arrow.width(), true );
	    p->drawPixmap( 3, 4, arrow );
	}
	else if ( popupPos.x() == iPos.x() ) {
	    if ( popupPos.y() < iPos.y() ) {
		turn.rotate( 90.0 );
		arrow = arrow.xForm( turn );
		paintArrowLabel( p, arrow.width(), true );
		p->drawPixmap( 3, 6, arrow );
	    }
	    else {
		turn.rotate( 270.0 );
		arrow = arrow.xForm( turn );
		paintArrowLabel( p, arrow.width(), false );
		p->drawPixmap( extent()-12, 6, arrow );
	    }
	}
	else {
	    turn.rotate( 180.0 );
	    arrow = arrow.xForm( turn );
	    paintArrowLabel( p, arrow.width(), false );
	    p->drawPixmap( extent()-8, 4, arrow );
	}
    }
}

void KasItem::paintArrowLabel( QPainter *p, int arrowSize, bool arrowOnLeft )
{
    QString text = title;
    int lx = 2;
    int ly = 2;
    int w = extent()-4;
    int h = 13;
    arrowSize+=2; // Add a space

    p->fillRect( lx, ly, w, h, QBrush( kasbar()->labelBgColor() ) );

    // Adjust for arrow
    if ( arrowOnLeft ) {
	lx += arrowSize;
	w -= arrowSize;
    }
    else {
	w -= arrowSize;
    }

    p->setFont( KGlobalSettings::taskbarFont() );
    p->setPen( kasbar()->labelPenColor() );
    if ( fontMetrics().width( text ) > w )
	p->drawText( lx, ly, w, h-1, AlignLeft | AlignVCenter, text );
    else
	p->drawText( lx, ly, w, h-1, AlignCenter, text );
}

void KasItem::paintBackground( QPainter *p )
{
    if ( activated )
	p->drawPixmap( 2, 15, *(kas->activeBg()) );
    else if ( !kas->isTransparent() )
	p->drawPixmap( 2, 15, *(kas->inactiveBg()) );
}

void KasItem::paintProgress( QPainter *p, int percent )
{
#if 1
    double amt = (extent()-4) * (percent / 100.0L);
    p->fillRect( 2, 13, (int) amt, 2, QBrush( kasbar()->progressColor() ) );
#else
    double angle = 360*16;
    angle = angle * (percent/100.0L);

    QRect pos;
    pos.setRect( 6, 18, 30, 30 );

    p->setPen( Qt::black );
    p->setBrush( QBrush( Qt::black ) );
    p->drawEllipse( pos );

    p->setPen( Qt::gray );
    p->setBrush( QBrush( Qt::gray ) );
    p->drawPie( pos, 90*16, qRound(-angle) );

    p->setPen( Qt::white );
    QString text = i18n("%1%");
    p->drawText( pos, AlignCenter, text.arg(percent) );
#endif
}

void KasItem::paint( QPainter *p )
{
   paintFrame( p );
   paintLabel( p );
   paintBackground( p );

//    QPixmap pixmap = KGlobal::iconLoader()->loadIcon( "wizard",
// 						     KIcon::NoGroup,
// 						     KIcon::SizeMedium );

//    if ( !pixmap.isNull() )
//       p->drawPixmap( 4, 16, pixmap );
}

void KasItem::paint( QPainter *p, int x, int y )
{
    p->save();
    p->translate( x, y );
    paint( p );
    p->restore();
}

void KasItem::repaint()
{
   repaint( true );
}

void KasItem::repaint( bool erase )
{
   kas->repaintItem( this, erase );
}

void KasItem::update()
{
   kas->updateItem( this );
}

