/* 
 * Copyright (C) 2003-2005 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: script_engine.h,v 1.14 2005/01/03 22:06:04 dsalt Exp $
 *
 * gate to the spider monkey javascript interpreter
 *
 * provides functions to generate interpreter instances
 * customized for controlling the xine engine and the gxine
 * frontend. 
 *
 */

#ifndef HAVE_SCRIPT_ENGINE_H
#define HAVE_SCRIPT_ENGINE_H

#include <glib.h>

#define XP_UNIX
#ifdef HAVE_LIBSMJS
# ifdef LIBJS_IS_SMJS
#  include <js/jsapi.h>
# else
#  include <smjs/jsapi.h>
# endif
#else
# include "spidermonkey/jsapi.h"
#endif

typedef int (*se_prop_int_cb_t) (void *user_data, char *property, int *num);
typedef void (*se_print_cb_t) (void *user_data, char *str, ...);

typedef struct se_s se_t; /* se: _s_cript _e_ngine */
typedef struct se_o_s se_o_t; /* script engine object */

typedef enum {
  SE_GROUP_HIDDEN,
  SE_GROUP_ENGINE,
  SE_GROUP_DIALOGUE,
  SE_GROUP_PLAYLIST,
  SE_GROUP_FILE,
  SE_GROUP_STATUS,
  SE_GROUP_PROPERTIES,
  SE_GROUP_INPUT,
  SE_GROUP_EXTERNAL,
} se_group_t;

struct se_s {

  JSContext     *cx;
  JSObject      *global;
  se_o_t        *g;
  
  jsval          rval;
  JSString      *str;

  se_print_cb_t  print_cb;
  void          *print_cb_data;
};

typedef struct {
  char		*id;
  JSFunction	*fun;
  se_group_t	 group;
  const char	*arghelp;
  const char	*funchelp;
} se_f_t;

#define SE_TYPE_STRING 0
#define SE_TYPE_BOOL   1
#define SE_TYPE_INT    2
 
typedef struct {
  char              *id;
  char              *value;
  GList             *listeners;
  int                se_type;
} se_prop_t;
 
typedef int (*se_prop_cb_t) (void *user_data, se_prop_t *prop);
 
typedef struct {
 
  se_prop_cb_t   cb;
  void          *user_data;
 
} se_prop_listener_t;
 
struct se_o_s {

  JSObject          *obj;

  se_o_t            *parent;
  char              *id;

  GList             *children;

  GList             *functions; /* methods */
 
  GList             *properties;

  void              *user_data;
};

/*
 * create a javascript engine with predefined xine functions 
 * and objects
 */
se_t *se_new (void);

int se_eval (se_t *se, const gchar *script, se_o_t *obj,
	     se_print_cb_t print_cb,
	     void *print_cb_data);

gchar *se_result_str (se_t *se) ;

int se_result_num (se_t *se, JSInt32 *num);

/*
 * create a script engine object (in javascript name space)
 */

se_o_t *se_create_object (se_t *se, se_o_t *parent /* may be NULL */,
                          const char *name, void *user_data);
 
/*
 * create functions (methods)
 */

typedef struct {
  const char *name;
  JSNative func;
  uintN nargs, attrs;
  se_group_t group;
  const char *arghelp, *funchelp;
} se_f_def_t;

se_f_t *se_defun (se_t *se, se_o_t *o /* may be NULL */,
                  const char *name, JSNative fun, uintN nargs, uintN attrs,
                  se_group_t, const char *arg_help, const char *func_help);
void se_defuns (se_t *se, se_o_t *o /* may be NULL */, const se_f_def_t defs[]);

/*
 * get/set/observe properties (attributes)
 */

/* does nothing if property already exists */
void   se_prop_create (se_t *se, se_o_t *o,
                       const char *id, const char *value,
                       int se_type);
void   se_prop_create_int (se_t *se, se_o_t *o,
                           const char *id, int value);
void   se_prop_create_bool (se_t *se, se_o_t *o,
                            const char *id, int value);

/* creates property if it doesn't exist */
void   se_prop_set_int (se_t *se, se_o_t *o,
                        const char *id, int value);
void   se_prop_set_bool (se_t *se, se_o_t *o,
                         const char *id, int value);
void   se_prop_set (se_t *se, se_o_t *o,
                    const char *id, const char *value,
                    int se_type);

char *se_prop_get (se_t *se, se_o_t *o, const char *id);
int   se_prop_get_int (se_t *se, se_o_t *o, const char *id);

/* add a property observer (callback will be called whenever the
 * properties value changes
 */
void   se_prop_add_listener (se_t *se, se_o_t *o,
                             const char *id,
                             se_prop_cb_t prop_cb,
                             void *user_data);

#ifdef LOG
#define se_log_fncall(func) printf ("script_engine: %s() called\n", (func))
#define se_log(FMT, ...) printf ("script_engine: " FMT, ## __VA_ARGS__)
#else
#define se_log_fncall(func)
#define se_log(FMT, ...)
#endif

/* Validation - assumes se_t *se, uintN argc, jsval *argv */

#define se_argc_check(value, FUNC) \
  if (argc != (value)) { \
    se->print_cb (se->print_cb_data, \
		  _("error: %s() needs %d parameters\n"), (FUNC), (value)); \
    return JS_TRUE; \
  }

#define se_argc_check_range(min, max, FUNC) \
  if (argc < (uintN)(min) || argc > (uintN)(max)) { \
    se->print_cb (se->print_cb_data, \
	   _("error: %s() needs %d...%d parameters\n"), (FUNC), (min), (max)); \
    return JS_TRUE; \
  }

#define se_argc_check_max(max, FUNC) \
  if (argc > (uintN)(max)) { \
    se->print_cb (se->print_cb_data, \
		  _("error: %s() needs 0...%d parameters\n"), (FUNC), (max)); \
    return JS_TRUE; \
  }

#define se_arg_is_int(n, FUNC) \
  if (!JSVAL_IS_INT (argv[(n)])) { \
    se->print_cb (se->print_cb_data, \
		 _("error: %s() argument %d is not an int\n"), (FUNC), (n)+1); \
    return JS_TRUE; \
  }

#define se_arg_is_string(n, FUNC) \
  if (!JSVAL_IS_STRING (argv[(n)])) { \
    se->print_cb (se->print_cb_data, \
	       _("error: %s() argument %d is not a string\n"), (FUNC), (n)+1); \
    return JS_TRUE; \
  }

#endif
