/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif
#include <errno.h>
#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_REGEX_H
#include <regex.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <dbh.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <libxfce4util/util.h>

#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "add_node_contents.h"
#include "add_file.h"
#include "bookmarks.h"
#include "callbacks.h"
#include "dummies.h"
#include "entry.h"
#include "filter.h"
#include "gdir.h"
#include "icons.h"
#include "ls.h"
#include "misc.h"
#include "monitor.h"
#include "refresh.h"
#include "widgets.h"


extern int stop;

enum
{
    XFCE_TRASH,
    KDE_TRASH,
    GNOME_TRASH,
    HOW_MANY_OTHER_TRASH
};

static gboolean did_erase;
static GtkTreeIter *target_iter;
static GtkTreeView *target_treeview;
static unsigned int trashcount;
static unsigned int smallcount, countbyte;
static gdir_t trash_gdir;
static int target_type;
static regex_t *target_preg;
static gboolean just_count;

static DBHashTable *trashbin = NULL;
static DBHashTable *newtrashbin = NULL;

static void delete_trash(DBHashTable * dbh)
{
    struct stat st;
    char *fullpath = (char *)DBH_DATA(dbh);
    if(lstat(fullpath, &st) < 0)
	return;
    if(S_ISDIR(st.st_mode))
    {
	if(rmdir(fullpath) == 0)
	    did_erase = TRUE;
	/*printf("rmdir %s\n",fullpath); */
    }
    else
    {
	/*printf("unlink %s\n",fullpath); */
	if(unlink(fullpath) == 0)
	    did_erase = TRUE;
    }

    /* now try to zap wastebasket */
    if(strchr(fullpath, '/'))
    {
	*(strrchr(fullpath, '/')) = 0;
	/*printf("rmdir %s\n",fullpath); */
	if(rmdir(fullpath) == 0)
	    did_erase = TRUE;
    }
    return;
}

int delete_all_trash(GtkTreeView *treeview)
{
    char trash_path[_POSIX_PATH_MAX];
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTreeIter iter;
    tree_entry_t *en;

    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);

    if((trashbin = DBH_open(trash_path)) == NULL)
	return -1;
    if(!set_load_wait(&tree_details))
	return -1;

    cursor_wait(treeview);

    do
    {
	did_erase = FALSE;
	DBH_foreach_sweep(trashbin, delete_trash);
    }
    while(did_erase);
    DBH_close(trashbin);
    unlink(trash_path);

    if(!get_trash_root(treeview, &iter, &en))
	return 1;

    if(IS_LOADED(en->type))
    {
	GtkTreePath *treepath;
	remove_folder(treeview, &iter);
	if(en->tag){
	    g_free(en->tag);
	    en->tag=NULL;
	}
	treepath = gtk_tree_model_get_path(treemodel, &iter);
	gtk_tree_view_collapse_row(treeview, treepath);
	gtk_tree_path_free(treepath);
	set_icon(treeview, &iter);
    }
    cursor_reset(treeview);
    print_status(treeview, "xf_INFO_ICON", _("Trash has been deleted"), NULL);
    unset_load_wait(&tree_details);
    local_monitor(treeview,TRUE);
    return 0;

}

static int check_dir(char *path)
{
    struct stat st;
    if(stat(path, &st) < 0)
    {
	if(mkdir(path, 0770) < 0)
	    return FALSE;
	return TRUE;
    }
    if(S_ISDIR(st.st_mode))
    {
	if(access(path, W_OK) < 0)
	    return FALSE;
	return TRUE;
    }
    return FALSE;
}


static void count_check(DBHashTable * dbh)
{
    char *p;
    struct stat st;
    char *fullpath = (char *)DBH_DATA(dbh);
    p = strrchr(fullpath, '/');

    /*printf("DBG:%s -> %s\n",fullpath,p); */
    if(!p || strlen(p) <= 1)
	return;
    p++;
    if(lstat(fullpath, &st) < 0)
	return;
    trashcount++;
    return;
}

int count_trash(void)
{
    char trash_path[_POSIX_PATH_MAX];

    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm");

    if(!check_dir(trash_path))
	return -1;

    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);

    trashcount = 0;		/* for count step */
    if((trashbin = DBH_open(trash_path)) == NULL)
	return -1;

    DBH_foreach_sweep(trashbin, count_check);
    DBH_close(trashbin);

    return (trashcount);
}



static void add_bin(DBHashTable * dbh)
{
    char *p;
    struct stat st;
    char *fullpath = (char *)DBH_DATA(dbh);
    p = strrchr(fullpath, '/');

    /*printf("DBG:%s -> %s\n",fullpath,p); */
    if(!p || strlen(p) <= 1)
	return;
    p++;
    if(regexec(target_preg, (const char *)p, 0, NULL, 0))
	return;
    if(lstat(fullpath, &st) < 0)
	return;
    if(!just_count)
    {
	trash_gdir.gl[trash_gdir.pathc].en = stat_entry(fullpath, target_type);
	if(!trash_gdir.gl[trash_gdir.pathc].en)
	{
	    printf("xffm: aarrgg! could not stat %s!!\n", fullpath);
	    return;
	}
	trash_gdir.gl[trash_gdir.pathc].pathv = g_strdup(p);
	trash_gdir.pathc++;
	/*printf("DBG:added %s\n",p); */
    }
    else
	trashcount++;
    return;
}

int open_trash(GtkTreeView * treeview, GtkTreeIter * iter, GtkTreePath * treepath, gpointer user_data)
{
    char trash_path[_POSIX_PATH_MAX];
    tree_entry_t *en = get_entry(treeview, iter);
    tree_details_t *tree_details = get_tree_details(treeview);

    target_iter = iter;
    target_treeview = treeview;
    target_type = en->type;
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm");
    if(!check_dir(trash_path))
	return -1;
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);
    smallcount = 0, countbyte = 0x10;	/* for dummy tag */
    trashcount = 0;		/* for count step */
    trash_gdir.pathc = 0;	/* for read step */
    if((trashbin = DBH_open(trash_path)) == NULL)
	return -1;

    cursor_wait(treeview);

/* returns compiled regex and sets filter in tree_entry */
    target_preg = get_regex_filter(treeview, en);


    just_count = TRUE;
    DBH_foreach_sweep(trashbin, add_bin);
    /*printf("DBG:pathc=%d\n",trash_gdir.pathc); */

    if(trashcount)
    {
	trash_gdir.gl = (dir_t *) malloc(trashcount * sizeof(dir_t));
	if(!trash_gdir.gl)
	{
	    DBH_close(trashbin);
	    return -1;
	}
	just_count = FALSE;
	DBH_foreach_sweep(trashbin, add_bin);
	/* printf("DBG:count=%d\n",trashcount); */
	if(trashcount != trash_gdir.pathc)
	{
	    SET_ERASED_SPACE(en->type);
	    /*printf ("DBG:listed=%d, counted=%d\n", trash_gdir.pathc,
	       trashcount); */
	}


	add_node_contents(treeview, iter, &trash_gdir);
	gdirfree(&trash_gdir);
    }
    else
    {
	SET_LOADED(en->type);
	reset_dummy(treeview, iter, 1);
    }
    if(DBH_ERASED_SPACE(trashbin))
    {
	/*printf("DBG: erased space=%d\n",DBH_ERASED_SPACE(trashbin)); */
	SET_ERASED_SPACE(en->type);
    }
    else
	UNSET_ERASED_SPACE(en->type);
    DBH_close(trashbin);
    if(!en->tag)
	en->tag = (char *)malloc(_POSIX_PATH_MAX);
    if(stop)
    {
	stop = FALSE;
	sprintf(en->tag, "%s : %s", FILENAME(en), _("load aborted."));
	cursor_reset(treeview);
	regfree(target_preg);
	return -2;

    }
    else
    {
	hide_stop(tree_details->window);
	sprintf(en->tag, "%s : %d %s", _("Trash"), trash_gdir.pathc, _("items"));
    }

    /*gtk_widget_thaw_child_notify  ((GtkWidget *)treeview);*/
    cursor_reset(treeview);
    regfree(target_preg);

    return 0;
}

static int read_trash(GtkTreeView * treeview, char *path)
{
    DIR *directory;
    int count = 0;
    struct dirent *d;
    struct stat st;
    char newpath[_POSIX_PATH_MAX];
    GString *gs;

    if(stop)
	return 0;
    set_progress(treeview, -1, 0);
    directory = opendir(path);
    if(!directory)
	return 0;
    while((d = readdir(directory)) != NULL)
    {
	if(stop)
	{
	    closedir(directory);
	    return 0;
	}
	if(strcmp(d->d_name, ".") == 0 || strcmp(d->d_name, "..") == 0)
	    continue;
	sprintf(newpath, "%s/%s", path, d->d_name);
	if(lstat(newpath, &st) < 0)
	{
	    continue;
	}
	/* if (d->d_type == DT_DIR) HAVE_DIRENT_D_TYPE */
	if(S_ISDIR(st.st_mode))
	{
	    print_status(treeview, "xf_TRASH_CLOSED_ICON", _("collecting trash from"), " ", abreviate(newpath), "...", NULL);
	    count += read_trash(treeview, newpath);
	}
	gs = g_string_new(newpath);
	sprintf((char *)DBH_KEY(trashbin), "%10u", g_string_hash(gs));
	if(!DBH_load(trashbin))
	{
	    memcpy(DBH_DATA(trashbin), (void *)newpath, strlen(newpath) + 1);
	    DBH_set_recordsize(trashbin, strlen(newpath) + 1);
	    DBH_update(trashbin);
	}
	g_string_free(gs, TRUE);
	count++;
    }
    closedir(directory);
    return (count);
}


static int read_trash_locations(GtkTreeView * treeview, char *path)
{
    DIR *directory;
    int count = 0;
    struct dirent *d;
    regex_t shortpreg;
    regex_t fullpreg;
    char newpath[_POSIX_PATH_MAX];
    char ot[1024];
    struct stat st;

    if(stop)
	return 0;
    set_progress(treeview, -1, 0);

    directory = opendir(path);
    if(!directory)
	return 0;

    /* XFCE trash: */
    sprintf(ot, "^%s/\\.xfce/trash$", xfce_get_homedir());
    /* XFCE4 trash: */
    strcat(ot, "|");
    sprintf(newpath, "^%s/trash$", xfce_get_userdir());
    strcat(ot, newpath);
    /* KDE trash: */
    strcat(ot, "|");
    sprintf(newpath, "^%s/Desktop/Trash$", xfce_get_homedir());
    strcat(ot, newpath);
    /* gnome trash: */
    strcat(ot, "|");
    sprintf(newpath, "^%s/\\.Trash$", xfce_get_homedir());
    strcat(ot, newpath);

    regcomp(&fullpreg, ot, REG_EXTENDED | REG_ICASE | REG_NOSUB);
    regcomp(&shortpreg, "^\\.\\.Wastebasket$", REG_EXTENDED | REG_ICASE | REG_NOSUB);
    while((d = readdir(directory)) != NULL)
    {
	if(stop)
	{
	    closedir(directory);
	    return 0;
	}
	if(strcmp(d->d_name, ".") == 0 || strcmp(d->d_name, "..") == 0)
	    continue;
	set_progress(treeview, -1, 0);

	sprintf(newpath, "%s/%s", path, d->d_name);
	if(!regexec(&fullpreg, newpath, 0, NULL, 0))
	{
	    count += read_trash(treeview, newpath);
	    continue;
	}
	if(lstat(newpath, &st) < 0)
	{
	    continue;
	}
	if(S_ISDIR(st.st_mode))
	{
	    if(!regexec(&shortpreg, d->d_name, 0, NULL, 0))
	    {
		/*printf("DBG:read trash from %s\n",newpath); */
		count += read_trash(treeview, newpath);
	    }
	    else
	    {
		/*printf("DBG:recursing into %s\n",newpath); */
		print_status(treeview, "xf_TRASH_CLOSED_ICON", _("collecting trash from"), " ", abreviate(newpath), "...", NULL);
		count += read_trash_locations(treeview, newpath);
	    }
	}
    }
    closedir(directory);
    regfree(&shortpreg);
    regfree(&fullpreg);
    return (count);
}

void purge_trash(DBHashTable * dbh)
{
    char *p;
    char *fullpath = (char *)DBH_DATA(dbh);
    struct stat st;

    if(!newtrashbin)
	g_assert_not_reached();
    p = strrchr(fullpath, '/');
    if(p)
    {
	p++;
	if(lstat(fullpath, &st) < 0)
	{
	    return;
	}
    }
    /* copy the key from the old record to the new record */
    memcpy(DBH_KEY(newtrashbin), DBH_KEY(trashbin), DBH_KEYLENGTH(trashbin));
    /* copy the data from the old record to the new record */
    memcpy(newtrashbin->data, trashbin->data, DBH_RECORD_SIZE(trashbin));
    /* I always forget this instruction, and it is the most important
     * (since DBH records have a variable size): */
    DBH_set_recordsize(newtrashbin, DBH_RECORD_SIZE(trashbin));
    /* write the record to the new file; */
    if(!DBH_update(newtrashbin))
	g_assert_not_reached();
    /*printf("DBG:clean record: %s\n",fullpath); */
    return;
}

/* returns -1 on failure, otherwise count of elements gathered */
int collect_trash(GtkTreeView * treeview, char *path)
{
    int count=0;
    char trash_path[_POSIX_PATH_MAX];
    tree_details_t *tree_details = get_tree_details(treeview);

    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm");
    if(!check_dir(trash_path))
	return -1;
    chdir(trash_path);		/* so the DBH_regen tmp file is generated here */
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);

    if((trashbin = DBH_open(trash_path)) == NULL)
    {
	trashbin = DBH_create(trash_path, 11);
	if(!trashbin)
	    return -1;
    }
    print_status(treeview, "xf_TRASH_CLOSED_ICON", _("collecting trash from"), " ", abreviate(path), "...", NULL);
    /*printf("collecting trash from %s\n",path); */


    cursor_wait(treeview);
    /* step one: add collected entries: */
    show_stop(tree_details->window);
    if (strrchr(path,'/')&&strcmp(strrchr(path,'/')+1,"..Wastebasket")==0)
	    count = read_trash(treeview, path);
    count += read_trash_locations(treeview, path);
    if(stop)
    {
	stop = FALSE;
	print_status(treeview, "xf_INFO_ICON",
		strerror(ETIMEDOUT),
		NULL);
	DBH_close(trashbin);
    }
    else
    {
	/* step two:   make dbh logical and physical structures the same 
	 *             eliminating stale entries while you're at it.*/
	gchar *s;
	char fname[_POSIX_PATH_MAX];
	hide_stop(tree_details->window);
	s = g_strdup_printf("%d", count);
	print_status(treeview, "xf_TRASH_CLOSED_ICON", s, " ", _("trash items collected."), NULL);
	g_free(s);s=NULL;
	strcpy(fname, "trashbin.XXXXXX");
	close(mkstemp(fname));
	newtrashbin = DBH_create(fname, DBH_KEYLENGTH(trashbin));
	DBH_foreach_sweep(trashbin, purge_trash);
	DBH_close(trashbin);
	DBH_close(newtrashbin);
	rename(fname, trash_path);

    }

    cursor_reset(treeview);
    return count;
}

/* returns -1 on failure, otherwise count of elements gathered */
int add2trash(GtkTreeView * treeview, char *path)
{
    GString *gs;
    char trash_path[_POSIX_PATH_MAX];

    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm");
    if(!check_dir(trash_path))
	return -1;
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);

    if((trashbin = DBH_open(trash_path)) == NULL)
    {
	trashbin = DBH_create(trash_path, 11);
	if(!trashbin)
	    return -1;
    }
    print_status(treeview, "xf_TRASH_CLOSED_ICON", _("adding to trash:"), " ", abreviate(path), "...", NULL);

    gs = g_string_new(path);
    sprintf((char *)DBH_KEY(trashbin), "%10u", g_string_hash(gs));
    if(!DBH_load(trashbin))
    {
	memcpy(DBH_DATA(trashbin), (void *)path, strlen(path) + 1);
	DBH_set_recordsize(trashbin, strlen(path) + 1);
	DBH_update(trashbin);
    }
    g_string_free(gs, TRUE);

    DBH_close(trashbin);
    return 1;
}

int bookmarks_collect_trash(GtkTreeView * treeview)
{
    GList *tmp, *list;
    int r, t = 0;
    char s[64];
    list = get_bookmark_dirlist();
    for(tmp = list; tmp; tmp = tmp->next)
    {
	r = collect_trash(treeview, (char *)tmp->data);
	g_free((char *)tmp->data);
	tmp->data=NULL;
	if(r > 0)
	    t += r;
    }
    g_list_free(list);
    sprintf(s, "%d", t);
    print_status(treeview, "xf_TRASH_CLOSED_ICON", s, " ", _("trash items collected."), NULL);
    return t;
}

/****************** callbacks *****************************/

void on_collect_trash1_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    GtkTreeView *treeview = get_selected_treeview((GtkWidget *) menuitem);
    /*GtkTreeModel *treemodel=gtk_tree_view_get_model(treeview); */
    /*tree_details_t *tree_details = get_tree_details(treeview);*/
    GtkTreeIter iter;
    tree_entry_t *en;
    int caso = 0;

    if(!get_selectpath_iter(treeview, &iter, &en))
	return;
    if(IS_DIR(en->type))
	caso = 1;
    else if(IS_BOOKMARK_TYPE(en->type) && IS_ROOT_TYPE(en->type))
	caso = 2;
    else
    {
	print_status(treeview, "xf_ERROR_ICON", strerror(EINVAL), NULL);
	return;
    }

    if(caso == 1)
	collect_trash(treeview, en->path);
    else if(caso == 2)
	bookmarks_collect_trash(treeview);

}

/* all this big function is just to erase the dbh file: */
void on_uncollect_trash(GtkMenuItem * menuitem, gpointer user_data)
{

    GtkTreeView *treeview = get_treeview((GtkWidget *) menuitem);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    tree_entry_t *en;
    GtkTreeIter iter;
    GtkTreePath *treepath;
    char trash_path[_POSIX_PATH_MAX];
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);
    if(unlink(trash_path))
    {
	print_diagnostics(treeview, "xf_ERROR_ICON", strerror(errno), "\n", NULL);
	print_status(treeview, "xf_ERROR_ICON", strerror(EIO), NULL);
	return;
    }
    else
    {
	print_status(treeview, "xf_INFO_ICON", _("Trash has been uncollected"), NULL);
    }


    if(!gtk_tree_model_get_iter_first(treemodel, &iter))
	g_assert_not_reached();
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
    while(!IS_TRASH_TYPE(en->type))
    {
	if(!gtk_tree_model_iter_next(treemodel, &iter))
	    g_assert_not_reached();
	gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
    }
    if(IS_LOADED(en->type))
    {
	remove_folder(treeview, &iter);
	if(en->tag){
	    g_free(en->tag);
	    en->tag=NULL;
	}
	treepath = gtk_tree_model_get_path(treemodel, &iter);
	gtk_tree_view_collapse_row(treeview, treepath);
	gtk_tree_path_free(treepath);
	set_icon(treeview, &iter);
    }
    print_status(treeview, "xf_INFO_ICON", _("Trash has been uncollected"), NULL);

}


void on_delete_all_trash(GtkMenuItem * menuitem, gpointer user_data)
{
    GtkTreeView *treeview = get_treeview((GtkWidget *) menuitem);
    delete_all_trash(treeview);
}


void on_schred_all_trash(GtkMenuItem * menuitem, gpointer user_data)
{

}


void on_uncollect_from_trash_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    char bm[_POSIX_PATH_MAX];
    GtkTreeIter iter, parent;
    tree_entry_t *en;
    GtkTreeView *treeview = get_selected_treeview((GtkWidget *) menuitem);
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);

    /*get selection */
    en = get_selected_entry(treeview, &iter);
    if(!en)
	g_assert_not_reached();
    xfce_get_userfile_r(bm, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);

    if((trashbin = DBH_open(bm)) != NULL)
    {
	GString *gs;
	gs = g_string_new(en->path);
	sprintf((char *)DBH_KEY(trashbin), "%10u", g_string_hash(gs));
	g_string_free(gs, TRUE);
	DBH_erase(trashbin);
	DBH_close(trashbin);
    }
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    if(gtk_tree_selection_get_selected(selection, &treemodel, &iter))
    {

	gtk_tree_store_remove((GtkTreeStore *) treemodel, &iter);
    }
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);
    get_trash_root(treeview, &parent, &en);
    if(!gtk_tree_model_iter_children(treemodel, &iter, &parent))
    {
	add_dummy(treeview, &parent);
	reset_dummy(treeview, &parent, 1);
    }
    SET_ERASED_SPACE(en->type);

}


void on_purge_trash_activate(GtkMenuItem * menuitem, gpointer user_data)
{
    GtkTreeView *treeview = get_treeview((GtkWidget *) menuitem);
    tree_details_t *tree_details = get_tree_details(treeview);
    GtkTreeIter iter;
    tree_entry_t *en;
    char fname[_POSIX_PATH_MAX];
    char trash_path[_POSIX_PATH_MAX];
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm");
    chdir(fname);
    strcpy(fname, "trashbin.XXXXXX");
    close(mkstemp(fname));
    xfce_get_userfile_r(trash_path, _POSIX_PATH_MAX-1, "xffm%ctrashbin.dbh",
		    G_DIR_SEPARATOR);
    trashbin = DBH_open(trash_path);
    if(!trashbin)
	return;
    newtrashbin = DBH_create(fname, DBH_KEYLENGTH(trashbin));
    if(!newtrashbin)
	g_assert_not_reached();
    DBH_foreach_sweep(trashbin, purge_trash);
    DBH_close(trashbin);
    DBH_close(newtrashbin);
    rename(fname, trash_path);
    get_trash_root(treeview, &iter, &en);
    UNSET_ERASED_SPACE(en->type);
    on_refresh((GtkButton *) tree_details->window, NULL);
}
