/**
 * Copyright (C) 2000-2002 the KGhostView authors. See file AUTHORS.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


// Add header files alphabetically

#include <qlayout.h>

#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kprocess.h>
#include <kstandarddirs.h>

#include "kgv_view.h"
#include "kgvfactory.h"

#include "generalsettingswidget.h"
#include "gssettingswidget.h"

#include "kgvconfigdialog.h"

KGVConfigDialog::KGVConfigDialog( QWidget *parent, const char *name,
				  bool modal )
  : KDialogBase( IconList, i18n( "Configure" ), Ok|Cancel|Default,
		 Ok, parent, name, modal, true )
{
    /*== General ==========================================================*/
    QFrame* generalPage = addPage( i18n( "General" ),
				    i18n( "General Settings" ),
				    SmallIcon( "kghostview", KIcon::SizeMedium ) );

    mGeneralWidget = new GeneralSettingsWidget( generalPage );

    QVBoxLayout* generalLayout = new QVBoxLayout( generalPage, 0,
				    spacingHint() );
    generalLayout->addWidget( mGeneralWidget );
    generalLayout->addStretch();

    /*== Ghostscript ======================================================*/
    QFrame* ghostscriptPage = addPage( i18n( "Ghostscript" ),
				    i18n( "Ghostscript Configuration" ),
				    SmallIcon( "pdf", KIcon::SizeMedium  ) );

    mGSWidget = new GSSettingsWidget( ghostscriptPage );

    QVBoxLayout* gsLayout = new QVBoxLayout( ghostscriptPage, 0,
				    spacingHint() );
    gsLayout->addWidget( mGSWidget );

    connect( mGSWidget, SIGNAL( configClicked() ),
	     SLOT( slotConfigureGhostscript() ) );

	setMinimumSize( sizeHint() );

    readSettings();
}

KGVConfigDialog::~KGVConfigDialog()
{
    writeSettings();
}


namespace {
    QString getGSVersion( QString fullPathToExec )
    {
	QString res;
	QString chkVersion = KProcess::quote(fullPathToExec) + " --version";
	FILE* p = popen( QFile::encodeName(chkVersion), "r" );
	if( p ) {
	    // FIXME: a badly configured interpreter can hang us
	    QFile qp;
	    qp.open( IO_ReadOnly, p );
	    qp.readLine( res, 80 );
	    qp.close();
	    pclose( p );
	    res = res.stripWhiteSpace();
	}
	kdDebug(4500) << "kgvconfigdialog.cpp::{unamed}::getGSVersion() returning " << res <<  endl;
	return res;
    }


    /* A mechanism for triggering redetection of gs versions:
     *
     * The idea is that whenever we find we need to workaround a certain version of gs,
     * we cannot rely that there will solelly be a new kghostviewrc, but that users will
     * upgrade kghostview. Therefore, whenever we want to trigger redection on a new version,
     * we increment the counter below. It should have the old value from the previous version
     * of kghostview and this function will get called.
     * 
     */

    /* On a related note:
     * We don't detect upgrades (or downgrades, for that matter) of gs.
     * I am seeing if I can get the version out of gs as a side effect to displaying a file.
     * This way, using kconfig:/Ghostscript/GS Version we will see whether the version has changed
     * and trigger a redetection without the trouble of running "gs --version" on each launch.
     *
     * LPC (9 April 2003)
     */
    /* currentRedetection value log:
     * 
     * 1- remove -dMaxBitmap for gs 6.5x
     * 2- see if it supports .setsafe ( see bug:57291 ).
     */
    const int currentRedetection = 2;


    /* Here are the issues we found so far in version of gs:
     *
     * - gs before 6.50 uses device x11alpha instead of x11
     * - gs 6.5x does not work well with -dMaxBitmap=...
     * - gs 6.52 and earlier as well as 7.0x for x < 4 don't support the .setsafe operator
     *
     */

    QString recommendSetSafe( QString version )
    {
	if ( version < QString::number( 6.53 ) ) return QString::number( 6.53 );
	if ( version[ 0 ] == '7' && version < QString::number( 7.04 ) ) return QString::number( 7.05 );
	return QString::null;
    }
    // This function should contain all the gs version specific workarounds.
    void redoGSDetection( KConfig* config ) 
    {
	kdDebug() << "kgvconfigdialog.cpp::{unnamed}::redoGSDetection()" << endl;
	QString version = getGSVersion( config->readPathEntry( "Interpreter", "gs" ) );
	QString recommended = recommendSetSafe( version );
	if ( !recommended.isNull() ) {
	    KMessageBox::sorry( 0,
		    i18n( "Your version of gs (version %1) is too old, since it has security issues "
			"which are impossible to resolve. Please upgrade to a newer version.\n"
			"KGhostView will try to work with it, but it may not display any files at all.\n"
			"Version %2 seems to be appropriate on your system, although newer versions will work as well." )
		    .arg( version )
		    .arg( recommended ) );
	}
	if ( version < QString::number( 7.00 ) ) 
	{
	    QStringList arguments = QStringList::split( ' ', config->readEntry( "Antialiasing arguments", "" ) );
	    arguments.remove( QString::fromLatin1( "-dMaxBitmap=10000000" ) );
	    QString antiAliasArgs = arguments.join( " " );

	    config->writeEntry( "Antialiasing arguments", antiAliasArgs );
	}
	config->writeEntry( "GS Version", version );
	config->writeEntry( "Redetection Counter", currentRedetection );
	config->sync();
    }
} // namespace

bool KGVConfigDialog::slotConfigureGhostscript()
{
    kdDebug(4500) << "KGVConfigDialog::slotConfigureGhostscript" << endl;

    QString exe = "gs";
    if( !KStandardDirs::findExe( exe ) ) {
	return false;
    }
    QString versionString = getGSVersion( exe );
    if ( versionString.isNull() ) return false;

    mInterpreterPath = exe;
    mNonAntiAliasArgs = "-sDEVICE=x11";
    mAntiAliasArgs = "-sDEVICE=x11 -dTextAlphaBits=4 -dGraphicsAlphaBits=2 -dMaxBitmap=10000000";
    setup();

    return true;
}

void KGVConfigDialog::setup()
{
    mGeneralWidget->setUseAntialias( mAntialias );
    mGeneralWidget->setUsePlatformFonts( mPlatformFonts );
    mGeneralWidget->setShowMessages( mShowMessages );
    mGeneralWidget->setPaletteType( mPaletteType );

    mGSWidget->setInterpreterURL( mInterpreterPath );
    mGSWidget->setNonAntialiasArgs( mNonAntiAliasArgs );
    mGSWidget->setAntialiasArgs( mAntiAliasArgs );
    mGSWidget->setDetectedVersion( mDetectedVersion );
}

void KGVConfigDialog::slotOk()
{
    mAntialias = mGeneralWidget->useAntialias();
    mPlatformFonts = mGeneralWidget->usePlatformFonts();
    mShowMessages = mGeneralWidget->showMessages();
	mPaletteType = mGeneralWidget->paletteType();

    mInterpreterPath = mGSWidget->interpreterURL();
    mNonAntiAliasArgs = mGSWidget->nonAntialiasArgs();
    mAntiAliasArgs = mGSWidget->antialiasArgs();

    writeSettings();
    accept();
}


void KGVConfigDialog::slotDefault()
{
    mInterpreterPath	= "gs";
    mAntialias		= true;
    mShowMessages	= true;
    mPlatformFonts	= false;
    mPaletteType	= COLOR_PALETTE;
    setup();
}

void KGVConfigDialog::readSettings()
{
    kdDebug(4500) << "KGVConfigDialog::readSettings" << endl;

    KConfig* config = KGVFactory::instance()->config();
    QString oldGroup = config->group();

    config->setGroup( "General" );

    mAntialias = config->readBoolEntry( "Antialiasing", true );
    mShowMessages = config->readBoolEntry( "Messages", false );
    mPlatformFonts = config->readBoolEntry( "Platform fonts", false );

    QString text = config->readEntry( "Palette" );
    if( text.find( "mono" ) == 0 )
	mPaletteType = MONO_PALETTE;
    else if( text.find( "gray" ) == 0 )
	mPaletteType = GRAY_PALETTE;
    else
	mPaletteType = COLOR_PALETTE;

   if ( !config->hasGroup( "Ghostscript" ) && slotConfigureGhostscript() ) {
       // We first do it cleanly so that we can immediatelly afterwards trigger the redetection code.
	config->setGroup( "Ghostscript" );
	config->writeEntry( "Interpreter", mInterpreterPath );
	config->writeEntry( "Non-antialiasing arguments", mNonAntiAliasArgs );
	config->writeEntry( "Antialiasing arguments", mAntiAliasArgs );
   }
   if( config->hasGroup( "Ghostscript" ) ) {
	config->setGroup( "Ghostscript" );
	if ( config->readNumEntry( "Redetection Counter", 0 ) < currentRedetection ) {
	    redoGSDetection( config );
	}
	mInterpreterPath = config->readPathEntry( "Interpreter" );
	mNonAntiAliasArgs = config->readEntry( "Non-antialiasing arguments" );
	mAntiAliasArgs = config->readEntry( "Antialiasing arguments" );
	mDetectedVersion = config->readEntry( "GS Version" );
	config->sync();
    }

    config->setGroup( oldGroup );

    setup();
}

void KGVConfigDialog::writeSettings()
{
    kdDebug(4500) << "KGVConfigDialog::writeSettings" << endl;

    KConfig* config = KGVFactory::instance()->config();
    QString oldGroup = config->group();

    config->setGroup( "General" );

    config->writePathEntry( "Interpreter", mInterpreterPath );
    config->writeEntry( "Antialiasing", mAntialias );
    config->writeEntry( "Platform fonts", mPlatformFonts );
    config->writeEntry( "Messages", mShowMessages );

    QString text;
    if( mPaletteType == COLOR_PALETTE )
	text = "color";
    else if( mPaletteType == GRAY_PALETTE )
	text = "grayscale";
    else if ( mPaletteType == MONO_PALETTE )
	text = "monochrome";
    else {
	kdWarning( 4500 ) << "KGVConfigDialog::writeSettings(): unkown palette type [ " << mPaletteType << "] " << endl;
	text = "color";
    }
    config->writeEntry( "Palette", text );

    config->setGroup( "Ghostscript" );
    config->writePathEntry( "Interpreter", mInterpreterPath );
    config->writeEntry( "Non-antialiasing arguments", mNonAntiAliasArgs );
    config->writeEntry( "Antialiasing arguments", mAntiAliasArgs );

    config->setGroup( oldGroup );
    config->sync();
}

#include "kgvconfigdialog.moc"

// vim:sw=4:sts=4:ts=8:noet
